package name.matthewgreet.strutscommons.view;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;




/**
 * Abstract Template that aids creation of OPTION tags in JSPs for multi selection SELECT elements by formatting model 
 * for display.  See {@link SelectBoxItemDisplay2}.  The model is expected to be collection of Data Transfer Objects 
 * whose type is defined by the implementing subclass.
 * 
 * <P>Example JSP code:<BR/>
 * <CODE>&lt;SELECT SIZE="6" NAME="serviceIds" MULTIPLE&gt;<BR>
 * &nbsp;&nbsp;&lt;s:iterator value="serviceMultiSelectBoxDisplay.list" var="serviceDisplay"&gt;<BR>
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;OPTION VALUE="&lt;s:property value="#serviceDisplay.value"/&gt;" &lt;s:property value="#serviceDisplay.selectedAttribute"/&gt;&gt;&lt;s:property value="#serviceDisplay.text"/&gt;&lt;/OPTION&gt;<BR>
 * &nbsp;&nbsp;&lt;/s:iterator&gt;<BR>
 * &lt;/SELECT&gt;</CODE></P>
 */
public abstract class MultiSelectBoxDisplay<K,T> {
    public class TextComparator implements Comparator<SelectBoxItemDisplay2<K,T>> {
        public int compare(SelectBoxItemDisplay2<K,T> o1, SelectBoxItemDisplay2<K,T> o2) {
            return o1.getText().compareToIgnoreCase(o2.getText());
        }
    }
    

    private List<SelectBoxItemDisplay2<K,T>> formattedModel;
    private Map<K,SelectBoxItemDisplay2<K,T>> formattedModelTableByKey;
    private Map<String,SelectBoxItemDisplay2<K,T>> formattedModelTableByValue;
    
    public MultiSelectBoxDisplay() {
        formattedModel = new ArrayList<SelectBoxItemDisplay2<K,T>>();
        formattedModelTableByKey = new HashMap<K,SelectBoxItemDisplay2<K,T>>();
        formattedModelTableByValue = new HashMap<String,SelectBoxItemDisplay2<K,T>>();
    }

    /**
     * May be overridden by subclasses to add additional, formatted items to the display list.  This is typically used 
     * to add a blank item to represent no value or text that demands a real value is selected.  This is called after 
     * sorting.
     * 
     * @param formattedModel List of {@link SelectBoxItemDisplay}
     */
    protected void addItems(List<SelectBoxItemDisplay2<K,T>> formattedModel) {
        // Empty
    }
    
    /**
     * Overridden by subclasses to return unformatted identifier of item record.  This is the value used to match the 
     * parameters of {@link #setSelectedValues}.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected abstract K getKey(T item);
    
    /**
     * May be overridden by subclasses to return a comparator for defining display order of formatted items.  Comparator
     * compares instances of {@link SelectBoxItemDisplay2} that are allowed by subclass.  Defaults to comparator that 
     * sorts by ascending display text. 
     */
    protected Comparator<SelectBoxItemDisplay2<K,T>> getSortComparator() {
        return new TextComparator();
    }
    
    /**
     * Overridden by subclasses to return formatted text to be displayed to user as part of OPTION element.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected abstract String getText(T item);
    
    /**
     * Overridden by subclasses to return string to be used in VALUE attribute of OPTION element.  This is the formatted 
     * version of the item record identifier and is the value used to match the parameters of 
     * {@link #setSelectedFormattedValues #setSelectedFormattedValues}.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected abstract String getValue(T item);
    
    /**
     * May be overridden by subclasses to filter items in lookup list from display.  Defaults to allow all items.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected boolean isAllowed(T item) {
        return true;
    }
    
    /**
     * Returns formatted version of list for human display.
     */
    public List<SelectBoxItemDisplay2<K,T>> getList() {
        return formattedModel;
    }
    
    /**
     * Returns selected, formatted item or null if none selected. 
     */
    public Collection<SelectBoxItemDisplay2<K,T>> getSelectedItems() {
    	Collection<SelectBoxItemDisplay2<K,T>> result;
    	
    	result = new ArrayList<>();
    	for (SelectBoxItemDisplay2<K,T> item: formattedModel) {
    		if (item.getSelected()) {
    			result.add(item);
    		}
    	}
    	return result;
    }
    
    /**
     * Directly sets formatted list.  The value must already be sorted.
     */
    public void setList(List<SelectBoxItemDisplay2<K,T>> value) {
        formattedModel = value;
    }
    
    /**
     * Sets unformatted values and creates formatted, sorted list from it.
     */
    public void setModel(Collection<T> model) {
        SelectBoxItemDisplay2<K,T> formattedItem;
        K key;
        String value, text;
        
        formattedModel.clear();
        if (model != null) {
            for (T item: model) {
                if (isAllowed(item)) {
                	key = getKey(item);
                    value = getValue(item);
                    text = getText(item);
                    formattedItem = new SelectBoxItemDisplay2<K,T>(value, text, key, item);
                    formattedModel.add(formattedItem);
                }
            }
        }
        
        Collections.sort(formattedModel, getSortComparator());
        
        addItems(formattedModel);
        
        formattedModelTableByKey.clear();
        formattedModelTableByValue.clear();
        for (SelectBoxItemDisplay2<K,T> formattedItem2: formattedModel) {
            formattedModelTableByKey.put(formattedItem2.getKey(), formattedItem2);
            formattedModelTableByValue.put(formattedItem2.getValue(), formattedItem2);
        }
    }

    /**
     * Selects items matching formatted identifiers.  These match the OPTION tag VALUE attributes.
     * 
     * @param values Formatted identifiers of selected items or null or empty collection to select none.
     */
    public void setSelectedFormattedValues(Collection<String> values) {
        SelectBoxItemDisplay2<K,T> item;
        
        // Reset all items.
        for (SelectBoxItemDisplay2<K,T> item2: formattedModel) {
            item2.setSelected(false);
        }
        
        if (values != null) {
            for (String value: values) {
                item = formattedModelTableByValue.get(value);
                if (item != null) {
                    item.setSelected(true);
                }
            }
        }
    }
    
    /**
     * Selects items matching unformatted identifiers.  Formatted version of key matches the OPTION tag VALUE attribute.
     * 
     * @param keys ids of selected items or null or empty collection to select none.
     */
    public void setSelectedValues(Collection<K> keys) {
        SelectBoxItemDisplay2<K,T> item;
        
        // Reset all items.
        for (SelectBoxItemDisplay2<K,T> item2: formattedModel) {
            item2.setSelected(false);
        }
        
        if (keys != null) {
            for (K key: keys) {
                item = formattedModelTableByKey.get(key);
                if (item != null) {
                    item.setSelected(true);
                }
            }
        }
    }
    
}
