package name.matthewgreet.strutscommons.view;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.conversion.impl.ConversionData;



/**
 * Template that aids creation of OPTION and OPTGROUP tags in JSPs for single selection SELECT elements by formatting 
 * list for display.  Concrete subclasses override {@link #getValue} and {@link #getText} to extract item values and 
 * display text from a raw list.  See also {@link SelectBoxGroupDisplay} and {@link SelectBoxItemDisplay2}.
 * 
 * <P>Example JSP code:<BR/>
 * <CODE>
 * &lt;SELECT SIZE="1" NAME="category" &gt;<BR/>
 * &nbsp;&nbsp;&lt;s:iterator value="categoryDisplay.list" var="categoryGroupDisplay"&gt;<BR/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;OPTGROUP LABEL="&lt;s:property value="#categoryGroupDisplay.text"/&gt;"&gt;<BR/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;s:iterator value="#categoryGroupDisplay.children" var="categoryItemDisplay"&gt;<BR/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&lt;OPTION VALUE="&lt;s:property value="#categoryItemDisplay.value"/&gt;" &lt;s:property value="#categoryItemDisplay.selectedAttribute"/&gt;&gt;&lt;s:property value="#categoryItemDisplay.text"/&gt;&lt;/OPTION&gt;<BR/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/s:iterator&gt;<BR/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/OPTGROUP&gt;<BR/>
 * &nbsp;&nbsp;&lt;/s:iterator&gt;<BR/>
 * &lt;/SELECT&gt;<BR/>
 * </CODE></P>
 * 
 * @param <K> Type of unique identifier of T.
 * @param <T> Record type of list used to generate dropdown.
 * @param <G> Type of describing group of T, which can be String.
 */
public abstract class GroupedSingleSelectBoxDisplay<K,T,G> {
    
    private List<SelectBoxGroupDisplay<K,T,G>> formattedModel;
    private Map<K, SelectBoxItemDisplay2<K,T>> formattedModelTableByKey;
    private Map<String, SelectBoxItemDisplay2<K,T>> formattedModelTableByValue;
    private K selectedKey;
    private String selectedValue;
    
    public GroupedSingleSelectBoxDisplay() {
        formattedModel = getInitialList();
        if (formattedModel == null) {
            formattedModel = new ArrayList<>();
        }
        selectedValue = null;
    }

    /**
     * May be overridden by subclasses to add additional, formatted items to the display list.  This is typically used 
     * to add a blank item to represent no value or text that demands a real value is selected.  This is called after 
     * sorting.
     * 
     * @param formattedModel List of {@link SelectBoxGroupDisplay} containing child {@link SelectBoxItemDisplay}.
     */
    protected void addItems(List<SelectBoxGroupDisplay<K,T,G>> formattedModel) {
        // Empty
    }
    
    /**
     * If {@link #hasBlankValue} returns true or {@link #addBlankValue} is called, returns text to display for blank 
     * value.  Defaults to blank.
     */
    protected String getBlankText() {
        return "";
    }
    
    /**
     * Overridden by subclasses to return item's group. 
     */
    protected abstract G getGroup(T item);
    
    /**
     * Overridden by subclass to return label for a group, which becomes the LABEL attribute for OPTGROUP tags.
     */
    protected abstract String getGroupLabel(G group);

    /**
     * May be overridden by subclasses to return a comparator that defines display order of groups.  Defaults to 
     * comparator that sorts by ascending display text.
     */
    protected Comparator<SelectBoxGroupDisplay<K,T,G>> getGroupSortComparator() {
        return (o1,o2) -> o1.getText().compareToIgnoreCase(o2.getText());
    }
    
    /**
     * May be overridden by subclasses to return initial selection, especially if the list is hardcoded and setModel 
     * will not be used.  Can return null for no initial list.
     */
    protected List<SelectBoxGroupDisplay<K,T,G>> getInitialList() {
        return null;
    }
    
    /**
     * Overridden by subclasses to return unformatted identifier of item record.  This is the value used to match the 
     * parameter of {@link #setSelectedValue}.
     *  
     * @param item Member of raw data list to be displayed.
     */
    public abstract K getKey(T item);
    
    /**
     * May be overridden by subclasses to return a comparator for defining display order of formatted items.  Comparator
     * compares instances of {@link SelectBoxItemDisplay} that are allowed by subclass.  Defaults to comparator that 
     * sorts by ascending display text. 
     */
    protected Comparator<SelectBoxItemDisplay2<K,T>> getSortComparator() {
        return (o1,o2) -> o1.getText().compareToIgnoreCase(o2.getText());
    }
    
    /**
     * Overridden by subclasses to return formatted text to be displayed to user as part of OPTION element.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected abstract String getText(T item);
    
    /**
     * Overridden by subclasses to return string to be used in VALUE attribute of OPTION element.  This is the value 
     * return in form data. 
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected abstract String getValue(T item);
    
    /**
     * May be overridden by subclasses to return true, inserting a blank value at the beginning of the SELECT list.
     * Defaults to false. 
     */
    protected boolean hasBlankValue() {
        return false;
    }
    
    /**
     * May be overridden by subclasses to filter items in lookup list from display.  Defaults to allow all items.
     *  
     * @param item Member of raw data list to be displayed.
     */
    protected boolean isAllowed(T item) {
        return true;
    }
    
    /**
     * May be overridden by subclasses to return formatted text of an enumerated value for display in a select box.  
     * data will be null for a blank entry.
     */
    protected SelectBoxItemDisplay2<K,T> makeSelectBoxItemDisplay(String value, String text, K key, T data) {
        return new SelectBoxItemDisplay2<K,T>(value, text, key, data);
    }
    
    /**
     * If called after {@link #setModel}, inserts blank value to beginning of list.  This is useful for search fields 
     * where a blank value means no search.
     */
    public void addBlankValue() {
        SelectBoxGroupDisplay<K,T,G> groupDisplay;
        
        groupDisplay = new SelectBoxGroupDisplay<K,T,G>("", null);
        groupDisplay.getChildren().add(makeSelectBoxItemDisplay("", getBlankText(), null, null));
        formattedModel.add(0, groupDisplay);
    }
    
    /**
     * Returns formatted version of account list for human display.
     */
    public List<SelectBoxGroupDisplay<K,T,G>> getList() {
        return formattedModel;
    }
    
    /**
     * Returns selected, formatted item or null if none selected. 
     */
    public SelectBoxItemDisplay2<K,T> getSelectedItem() {
        if (selectedKey != null) {
            return formattedModelTableByKey.get(selectedKey);
        } else {
            return null;
        }
    }
    
    /**
     * Directly sets formatted version of account list for human display.  The value must already be sorted.
     */
    public void setList(List<SelectBoxGroupDisplay<K,T,G>> value) {
        formattedModel = value;
    }
    
    /**
     * Formats lookup list for use in single selection HTML SELECT element.  This replaces any initial list.
     */
    public void setModel(Collection<T> model) {
        SelectBoxItemDisplay2<K,T> formattedItem;
        SelectBoxGroupDisplay<K,T,G> groupDisplay;
        Comparator<SelectBoxItemDisplay2<K,T>> childComparator;
        Comparator<SelectBoxGroupDisplay<K,T,G>> groupComparator;
        K key;
        G group;
        Map<String,SelectBoxGroupDisplay<K,T,G>> workingGroups;
        String value, text, groupLabel;
        
        workingGroups = new HashMap<>();
        formattedModelTableByKey = new HashMap<>();
        formattedModelTableByValue = new HashMap<>();
        selectedValue = null;
        if (model != null) {
            for (T item: model) {
                if (isAllowed(item)) {
                	key = getKey(item);
                    value = getValue(item);
                    text = getText(item);
                    group = getGroup(item);
                    groupLabel = getGroupLabel(group);
                    groupDisplay = workingGroups.get(groupLabel);
                    if (groupDisplay == null) {
                        groupDisplay = new SelectBoxGroupDisplay<K,T,G>(groupLabel, group);
                        workingGroups.put(groupLabel, groupDisplay);
                    }
                    formattedItem = makeSelectBoxItemDisplay(value, text, key, item);
                    groupDisplay.getChildren().add(formattedItem);
                    formattedModelTableByKey.put(key, formattedItem);
                    formattedModelTableByValue.put(value, formattedItem);
                }
            }
        }
        
        formattedModel = new ArrayList<>();
        childComparator = getSortComparator();
        groupComparator = getGroupSortComparator();
        for (Entry<String, SelectBoxGroupDisplay<K,T,G>> groupEntry: workingGroups.entrySet()) {
            Collections.sort(groupEntry.getValue().getChildren(), childComparator);
            formattedModel.add(groupEntry.getValue());
        }
        Collections.sort(formattedModel, groupComparator);
        
        addItems(formattedModel);
        if (hasBlankValue()) {
            addBlankValue();
        }
    }

    /**
     * Selects item matching named Struts 2 conversion error (rejected form field value), or formatted identifier if 
     * conversion error not found.  Only appropriate in a Struts 2 context.
     * 
     * @param value Formatted identifier of item..
     * @param conversionErrorName Name of Struts 2 conversion error, which will have the "&lt;form name&gt;.&lt;field name&gt;" 
     *                            format.
     */
    public void setSelectedFormattedValueWithConversionError(String value, String conversionErrorName) {
    	ActionContext actionContext;
    	ConversionData conversionData;
    	Map<String, ConversionData> conversionErrors;
    	String rejectedValue;
    	
    	rejectedValue = null;
		actionContext = ActionContext.getContext();
		if (actionContext != null) {
			conversionErrors = actionContext.getConversionErrors();
			conversionData = conversionErrors.get(conversionErrorName);
			if (conversionData != null) {
				rejectedValue = (String)conversionData.getValue();
				setSelectedFormattedValue(rejectedValue);
			}
		}
		if (rejectedValue == null) {
			setSelectedFormattedValue(value);
		}
    }

    /**
     * Selects item matching named Struts 2 conversion error (rejected form field value), or unformatted identifier if 
     * conversion error not found.  Only appropriate in a Struts 2 context.
     * 
     * @param key Unformatted identifier of item..
     * @param conversionErrorName Name of Struts 2 conversion error, which will have the "&lt;form name&gt;.&lt;field name&gt;" 
     *                            format.
     */
    public void setSelectedValueWithConversionError(K key, String conversionErrorName) {
    	ActionContext actionContext;
    	ConversionData conversionData;
    	Map<String, ConversionData> conversionErrors;
    	String rejectedValue;
    	
    	rejectedValue = null;
		actionContext = ActionContext.getContext();
		if (actionContext != null) {
			conversionErrors = actionContext.getConversionErrors();
			conversionData = conversionErrors.get(conversionErrorName);
			if (conversionData != null) {
				rejectedValue = (String)conversionData.getValue();
				setSelectedFormattedValue(rejectedValue);
			}
		}
		if (rejectedValue == null) {
			setSelectedValue(key);
		}
    }

    /**
     * Sets selected item of list identified by its value.
     * 
     * @param value id of selected item or null or empty string to select none.
     */
    public void setSelectedFormattedValue(String value) {
        SelectBoxItemDisplay2<K,T> item;
        
        // Deselect existing, selected item.
        if (selectedValue != null) {
            item = formattedModelTableByValue.get(selectedValue);
            item.setSelected(false);
        }
        
        selectedKey = null;
        selectedValue = null;
        if (value != null && value.length() > 0) {
            item = formattedModelTableByValue.get(value);
            if (item != null) {
                item.setSelected(true);
                selectedKey = item.getKey();
                selectedValue = item.getValue();
            }
        }
    }

    /**
     * Sets selected item of list identified by its value.
     * @param key unformatted id of selected item or null to select none.
     */
    public void setSelectedValue(K key) {
        SelectBoxItemDisplay2<K,T> item;
        
        // Deselect existing, selected item.
        if (selectedKey != null) {
            item = formattedModelTableByKey.get(selectedKey);
            item.setSelected(false);
        }
        
        selectedKey = null;
        selectedValue = null;
        if (key != null) {
            item = formattedModelTableByKey.get(key);
            if (item != null) {
                item.setSelected(true);
                selectedKey = item.getKey();
                selectedValue = item.getValue();
            }
        }
    }
}
