package name.matthewgreet.strutscommons.view;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.conversion.impl.ConversionData;


/**
 * <P>Template that aids creation of OPTION and OPTGROUP tags in JSPs for single selection SELECT elements by formatting 
 * an enumerated type for display and selecting from a current value.  See also {@link SelectBoxGroupDisplay} and 
 * {@link SelectBoxItemDisplay}.</P>
 * 
 * <P>Example JSP code:<BR/>
 * <CODE>
 * &lt;SELECT SIZE="1" NAME="category" &gt;<BR/>
 * &nbsp;&nbsp;&lt;s:iterator value="categoryDisplay.list" var="categoryGroupDisplay"&gt;<BR/>
 * &nbsp;&nbsp;&lt;OPTGROUP LABEL="&lt;s:property value="#categoryGroupDisplay.text"/&gt;"&gt;<BR/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;s:iterator value="#categoryGroupDisplay.children" var="categoryItemDisplay"&gt;<BR/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&lt;OPTION VALUE="&lt;s:property value="#categoryItemDisplay.value"/&gt;" &lt;s:property value="#categoryItemDisplay.selectedAttribute"/&gt;&gt;&lt;s:property value="#categoryItemDisplay.text"/&gt;&lt;/OPTION&gt;<BR/>
 * &nbsp;&nbsp;&nbsp;&nbsp;&lt;/s:iterator&gt;<BR/>
 * &nbsp;&nbsp;&lt;/OPTGROUP&gt;<BR/>
 * &nbsp;&nbsp;&lt;/s:iterator&gt;<BR/>
 * &lt;/SELECT&gt;<BR/>
 * </CODE></P>
 */
public abstract class GroupedEnumSingleSelectBoxDisplay<E extends Enum<E>> {
    private List<SelectBoxGroupDisplay<E,E,Integer>> formattedModel;
    private Map<E, SelectBoxItemDisplay2<E,E>> formattedModelTableByKey;
    private Map<String, SelectBoxItemDisplay2<E,E>> formattedModelTableByValue;
    private E selectedKey;
    private String selectedValue;
    
    public GroupedEnumSingleSelectBoxDisplay() {
        SelectBoxItemDisplay2<E,E> formattedItem;
        SelectBoxGroupDisplay<E,E,Integer> group;
        Comparator<SelectBoxItemDisplay2<E,E>> sortComparator;
        Map<Integer,SelectBoxGroupDisplay<E,E,Integer>> workingGroups;
        String value, text, groupLabel;
        int groupOrder;
        
        workingGroups = new HashMap<>();
        formattedModelTableByKey = new HashMap<>();
        formattedModelTableByValue = new HashMap<>();
        selectedKey = null;
        selectedValue = null;
        for (E item: getEnumValues()) {
            if (isAllowed(item)) {
                value = getValue(item);
                text = getText(item);
                groupOrder = getGroupOrder(item);
                group = workingGroups.get(groupOrder);
                if (group == null) {
                    groupLabel = getGroupLabel(groupOrder);
                    group = new SelectBoxGroupDisplay<E,E,Integer>(groupLabel, groupOrder);
                    workingGroups.put(groupOrder, group);
                }
                formattedItem = makeSelectBoxItemDisplay(value, text, item);
                group.getChildren().add(formattedItem);
                formattedModelTableByKey.put(item, formattedItem);
                formattedModelTableByValue.put(value, formattedItem);
            }
        }
        
        formattedModel = new ArrayList<>();
        sortComparator = getSortComparator();
        for (Entry<Integer, SelectBoxGroupDisplay<E,E,Integer>> groupEntry: workingGroups.entrySet()) {
            Collections.sort(groupEntry.getValue().getChildren(), sortComparator);
            formattedModel.add(groupEntry.getValue());
        }
        Collections.sort(formattedModel, (o1,o2) -> o1.getData().intValue() - o2.getData().intValue());
        
        addItems(formattedModel);
        if (hasBlankValue()) {
            group = new SelectBoxGroupDisplay<E,E,Integer>("", -1);
            group.getChildren().add(makeSelectBoxItemDisplay("", getBlankText(), null));
            formattedModel.add(0, group);
        }
        
    }

    /**
     * May be overridden by subclasses to add additional, formatted items to the
     * display list.  This is typically used to add a blank item to represent no
     * value or text that demands a real value is selected.
     * @param formattedModel List of {@link SelectBoxItemDisplay}
     */
    protected void addItems(List<SelectBoxGroupDisplay<E,E,Integer>> formattedModel) {
        // Empty
    }
    
    /**
     * If {@link #hasBlankValue} returns true is called, returns text to display for blank 
     * value.  Defaults to blank.
     */
    protected String getBlankText() {
        return "";
    }
    
    /**
     * Overridden by subclass to return the values of the enumerated type.  This
     * function exists because the values cannot be extracted from a generic,
     * enumerated type;  
     */
    protected abstract E[] getEnumValues();
    
    /**
     * Overridden by subclass to return label for a group, which becomes the LABEL attribute for OPTGROUP tags.
     */
    protected abstract String getGroupLabel(int groupOrder);
    
    /**
     * Overridden by subclass to return the display order for the item's group.  The list is sorted by group order, then 
     * according to sort comparator.
     */
    protected abstract int getGroupOrder(E item);
    
    /**
     * May be overridden by subclasses to return a comparator for defining 
     * display order of formatted items.  Comparator compares instances of 
     * {@link SelectBoxItemDisplay} that are allowed by subclass.  Defaults to 
     * comparator that sorts by ascending enumeration order. 
     */
    protected Comparator<SelectBoxItemDisplay2<E,E>> getSortComparator() {
        return (o1,o2) -> o1.getData().compareTo(o2.getData());
    }
    
    /**
     * May be overridden by subclasses to return text to be displayed to user as
     * part of OPTION element.  This is typically overridden as enumeration
     * names aren't friendly for user display. 
     */
    protected String getText(E item) {
        return item.toString();
    }
    
    /**
     * May be overridden by subclasses to return string to be used in VALUE
     * attribute of OPTION element.  It is rarely useful to override this. 
     */
    protected String getValue(E item) {
        return item.toString();
    }
    
    /**
     * May be overridden by subclasses to return true, inserting a blank value
     * at the beginning of the SELECT list.  Defaults to false. 
     */
    protected boolean hasBlankValue() {
        return false;
    }
    
    /**
     * May be overridden by subclasses to filter items in lookup list from
     * display.  Defaults to allow all items. 
     */
    protected boolean isAllowed(E item) {
        return true;
    }
    
    /**
     * May be overridden by subclasses to return formatted text of an enumerated value for display in a select box.  
     * data will be null for a blank entry.
     */
    protected SelectBoxItemDisplay2<E,E> makeSelectBoxItemDisplay(String value, String text, E data) {
        return new SelectBoxItemDisplay2<E,E>(value, text, data, data);
    }
    
    /**
     * Returns formatted version of account list for human display.
     */
    public List<SelectBoxGroupDisplay<E,E,Integer>> getList() {
        return formattedModel;
    }
    
    /**
     * Directly sets formatted version of account list for human display.  The
     * value must already be sorted.
     */
    public void setList(List<SelectBoxGroupDisplay<E,E,Integer>> value) {
        formattedModel = value;
    }
    
    /**
     * Selects item matching named Struts 2 conversion error (rejected form field value), or formatted identifier if 
     * conversion error not found.  Only appropriate in a Struts 2 context.
     * 
     * @param value Formatted identifier of item..
     * @param conversionErrorName Name of Struts 2 conversion error, which will have the "&lt;form name&gt;.&lt;field name&gt;" 
     *                            format.
     */
    public void setSelectedFormattedValueWithConversionError(String value, String conversionErrorName) {
    	ActionContext actionContext;
    	ConversionData conversionData;
    	Map<String, ConversionData> conversionErrors;
    	String rejectedValue;
    	
    	rejectedValue = null;
		actionContext = ActionContext.getContext();
		if (actionContext != null) {
			conversionErrors = actionContext.getConversionErrors();
			conversionData = conversionErrors.get(conversionErrorName);
			if (conversionData != null) {
				rejectedValue = (String)conversionData.getValue();
				setSelectedFormattedValue(rejectedValue);
			}
		}
		if (rejectedValue == null) {
			setSelectedFormattedValue(value);
		}
    }

    /**
     * Selects item matching named Struts 2 conversion error (rejected form field value), or unformatted identifier if 
     * conversion error not found.  Only appropriate in a Struts 2 context.
     * 
     * @param key Unformatted identifier of item..
     * @param conversionErrorName Name of Struts 2 conversion error, which will have the "&lt;form name&gt;.&lt;field name&gt;" 
     *                            format.
     */
    public void setSelectedValueWithConversionError(E key, String conversionErrorName) {
    	ActionContext actionContext;
    	ConversionData conversionData;
    	Map<String, ConversionData> conversionErrors;
    	String rejectedValue;
    	
    	rejectedValue = null;
		actionContext = ActionContext.getContext();
		if (actionContext != null) {
			conversionErrors = actionContext.getConversionErrors();
			conversionData = conversionErrors.get(conversionErrorName);
			if (conversionData != null) {
				rejectedValue = (String)conversionData.getValue();
				setSelectedFormattedValue(rejectedValue);
			}
		}
		if (rejectedValue == null) {
			setSelectedValue(key);
		}
    }

    /**
     * Sets selected item of list identified by its value.
     * @param value formatted value of selected item or null or empty string to select none.
     */
    public void setSelectedFormattedValue(String value) {
        SelectBoxItemDisplay2<E,E> item;
        
        // Deselect existing, selected item.
        if (selectedValue != null) {
            item = formattedModelTableByValue.get(selectedValue);
            item.setSelected(false);
        }
        
        selectedKey = null;
        selectedValue = null;
        if (value != null && value.length() > 0) {
            item = formattedModelTableByValue.get(value);
            if (item != null) {
                item.setSelected(true);
                selectedKey = item.getKey();
                selectedValue = item.getValue();
            }
        }
    }

    /**
     * Sets selected item of list identified by its value.
     * @param key unformatted id of selected item or null to select none.
     */
    public void setSelectedValue(E key) {
        SelectBoxItemDisplay2<E,E> item;
        
        // Deselect existing, selected item.
        if (selectedKey != null) {
            item = formattedModelTableByKey.get(selectedKey);
            item.setSelected(false);
        }
        
        selectedKey = null;
        selectedValue = null;
        if (key != null) {
            item = formattedModelTableByKey.get(key);
            if (item != null) {
                item.setSelected(true);
                selectedKey = item.getKey();
                selectedValue = item.getValue();
            }
        }
    }
}
