package name.matthewgreet.strutscommons.util;

import java.lang.annotation.Annotation;
import java.util.Collection;

import name.matthewgreet.strutscommons.exception.CustomPolicyNotApplicableException;
import name.matthewgreet.strutscommons.exception.PolicyLookupRejectionException;
import name.matthewgreet.strutscommons.policy.CollectionConverter;
import name.matthewgreet.strutscommons.policy.Converter;
import name.matthewgreet.strutscommons.policy.Policy;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.DefaultCollectionConverterEntry;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.DefaultConverterEntry;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.PolicyEntry;

/**
 * Defines a helper that returns the policy implementation (adjuster, converter or validator) from a form field 
 * policy annotation (but not custom policies as these already refer to their implementing policy), returns the default 
 * converter for a form field type, and adds/updates such lookup lists. 
 */
public interface PolicyLookup {
	/**
	 * Returns all default collection converters in use and related metadata. 
	 */
	public Collection<DefaultCollectionConverterEntry<?,?>> getDefaultCollectionConverterEntries();
	
	/**
	 * Returns default collection converter and related metadata for an item class of a collection form field, or null 
	 * if not found. 
	 */
	public <C extends CollectionConverter<?,T>,T> DefaultCollectionConverterEntry<T,C> getDefaultCollectionConverterEntry(Class<? extends T> itemClass);
	
	/**
	 * Returns all default converters in use and related metadata. 
	 */
	public Collection<DefaultConverterEntry<?,?>> getDefaultConverterEntries();
	
	/**
	 * Returns default converter and related metadata for a recipient type form field, or null if not found. 
	 */
	public <C extends Converter<?,T>,T> DefaultConverterEntry<T,C> getDefaultConverterEntry(Class<? extends T> fieldClass);
	
	/**
	 * Returns all policy classes and related metadata. 
	 */
	public Collection<PolicyEntry<?,?,?>> getPolicyEntries();

	/**
	 * Returns policy class and related metadata to be used for a form field annotation, or null if not found.  Returns 
	 * null for custom annotations as they already refer to their policy class.
	 */
	public <A extends Annotation,P extends Policy<A>,T> PolicyEntry<A,P,T> getPolicyEntry(Class<? extends A> annotationClass);

	/**
	 * Adds default collection converter so it can be found from its recipient type, returning description of any 
	 * existing collection converter that was replaced.  Throws {@link CustomPolicyNotApplicableException} if custom 
	 * policy added.  Implementations may throw similar exceptions to enforce their own rules.
	 */
	public <C extends CollectionConverter<?,T>,T> DefaultCollectionConverterEntry<T,?> putDefaultCollectionConverter(Class<C> collectionConverterClass) throws PolicyLookupRejectionException;
	
	/**
	 * Adds default converter so it can be found from its recipient type, returning description of any existing 
	 * converter that was replaced.  Throws {@link CustomPolicyNotApplicableException} if custom policy added.  
	 * Implementations may throw similar exceptions to enforce their own rules.
	 */
	public <C extends Converter<?,T>,T> DefaultConverterEntry<T,?> putDefaultConverter(Class<C> converterClass) throws PolicyLookupRejectionException;
		
	/**
	 * Adds policy class so it can be found from its annotation class, returning description of any existing policy that 
	 * was replaced.  Throws {@link CustomPolicyNotApplicableException} if custom policy added.  Implementations may 
	 * throw similar exceptions to enforce their own rules.    
	 */
	public <A extends Annotation,P extends Policy<A>,T> PolicyEntry<A,?,?> putPolicy(Class<P> policyClass) throws PolicyLookupRejectionException;
	
	/**
	 * Removes default collection converter class from lookup, returning description of any existing converter that was 
	 * removed.  Implementations may throw {@link PolicyLookupRejectionException} or subclasses to enforce their own 
	 * rules.    
	 */
	public <C extends CollectionConverter<?,T>,T> DefaultCollectionConverterEntry<T,C> removeDefaultCollectionConverter(Class<? extends T> itemClass) throws PolicyLookupRejectionException;

	/**
	 * Removes default converter class from lookup, returning description of any existing converter that was removed.  
	 * Implementations may throw {@link PolicyLookupRejectionException} or subclasses to enforce their own rules.    
	 */
	public <C extends Converter<?,T>,T> DefaultConverterEntry<T,C> removeDefaultConverter(Class<? extends T> fieldClass) throws PolicyLookupRejectionException;
	
	/**
	 * Removes policy class from lookup, returning description of any existing policy that was removed.  Implementations 
	 * may throw {@link PolicyLookupRejectionException} or subclasses to enforce their own rules.    
	 */
	public <A extends Annotation,P extends Policy<A>,T> PolicyEntry<A,P,T> removePolicy(Class<? extends A> annotationClass) throws PolicyLookupRejectionException;
	
}
