package name.matthewgreet.strutscommons.util;

import java.io.Serializable;
import java.util.List;


/**
 * <P>Strategy for lazy loading a page of records to display from an index range (0 based), typically with list size 
 * found by {@link ListSizeFinder}.  This is particularly useful for reading MySQL tables traversing an index using 
 * OFFSET and LIMIT keywords.</P>
 * 
 * @param <M> Record type of master list or NA if finding for the master list.
 * @param <T> Record type returned by this finder.
 */
@FunctionalInterface
public interface PageByIndexRangeFinder<M extends Serializable,T extends Serializable> extends Serializable {
    
    /**
     * Returns items within a range of indices, where indices start at 0.  If the range covers indices that don't exist, 
     * any records that are in range should be returned.  No item should ever be null.  For convenience of MySQL 
     * adapters, OFFSET is startIndex and LIMIT is endIndex - startIndex + 1. 
     * 
     * @param selectedMaster Selected record in master list or null for loading master list.
     * @param startIndex Start of index range, where indices start at zero.
     * @param endIndex End of index range (inclusive), where indices start at zero.
     */
    public List<T> getItems(M selectedMaster, int startIndex, int endIndex) throws Exception;
    
    /**
     * Returns true if finder eager loads any optional extensions of each item.  This is usually false as loading
     * extensions results in poor performance or there are no optional extensions.  
     */
    default public boolean getLoadsDetails() {
        return false;
    }
}
