package name.matthewgreet.strutscommons.util;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.UnaryOperator;


/**
 * <P>Template, record list framework that uses finder Commands to lazy load lists, pages of a list, and details of
 *    the currently selected record as needed, and typically combined with others in a framework like 
 *    {@link AbstractCompositeCache}.  Such Commands are typically set by a Struts action using form based search 
 *    criteria, or by a framework to retrieve child lists of a selected master record.  Struts actions can examine the 
 *    list, usually the currently selected record, select a new record as the current selection, update items, or 
 *    invalidate items or the entire list, so changed items or lists are refound and displayed.</P>
 * 
 * <P>Record data is partitioned into list data and, optionally, detail data.  List data is displayed in list-based 
 *    pages, such as search result pages, and list and page finders know how to retrieve them.  List data is typically 
 *    just the base record from a single table to avoid performance problems with table joins.  Detail 
 *    data, if any, is displayed in pages displaying the currently selected record in detail, as well as list data, 
 *    often for editing, and single item finders know how to retrieve it (as well as list data).   List and page finders 
 *    need not retrieve any detail data, and usually don't for performance reasons.  However, if detail data applies and 
 *    no single item finder exists, list and page finders must retrieve details.</P>
 *    
 * <P>The record list itself may not be fully loaded and may lazy load pages from placeholder data as requested, 
 *    depending on pagination mode.  The pagination mode is set by setting the placeholder and page finders needed for 
 *    the mode.  Currently, four pagination modes are supported.</P>
 * 
 * <TABLE CLASS="main">
 *   <TR>
 *     <TH>Type</TH>
 *     <TH>Method</TH>
 *     <TH>Usage</TH>
 *     <TH>Requires</TH>
 *     <TH>Notes</TH>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Base record list</TD>
 *     <TD>Loads all base records from single table, loads additional records of requested page</TD>
 *     <TD>Large result list with additional list data from related tables</TD>
 *     <TD>baseRecordlistFinder, pageExtensionAssembler</TD>
 *     <TD>pageExtensionChecker can also be set to filter record that don't need page extensions</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Full list</TD>
 *     <TD>Loads entire list at once</TD>
 *     <TD>Small result list and no back-end pagination support</TD>
 *     <TD>listFinder</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>Page by ids</TD>
 *     <TD>Loads primary keys of entire list, loads records from primary keys of requested page</TD>
 *     <TD>Large result list where ids are quicker to read and order doesn't matter</TD>
 *     <TD>idListFinder, pageByIdsFinder</TD>
 *     <TD></TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>Page by index range</TD>
 *     <TD>Loads record count of entire list, loads records from index range of requested page</TD>
 *     <TD>Large result list where an index allows order by without reading entire list first</TD>
 *     <TD>listSizeFinder, pageByIndexRangeFinder</TD>
 *     <TD>listSizeFinder and pageByIndexRangeFinder must work in tandem, use {@link #setListSizeFinder setListSizeFinder} or 
 *         {@link #setListSizeAndFinder setListSizeAndFinder} to set both</TD>
 *   </TR>
 *   <CAPTION>List Finder Types</CAPTION>
 * </TABLE>
 * 
 * <P>Some caches can be slave lists, meaning their contents depends on a master record given to it by another combining 
 *    framework. Such frameworks will invalidate slave lists if another master record is selected or it's reloaded.</P>
 * 
 * @param <M> Record type of master list or Object if this is the master list.
 * @param <K> Record primary key or id type.
 * @param <T> Record type.
 */
public class ListCache<M extends Serializable,K extends Serializable,T extends Serializable> implements Serializable {
	private static final long serialVersionUID = -8481658486616110929L;

    private static class ItemDataComparator<K,T> implements Comparator<ItemData<K,T>> {
        private Comparator<T> itemComparator;
        
        public ItemDataComparator(Comparator<T> itemComparator) {
            super();
            this.itemComparator = itemComparator;
        }
        
        @Override
        public int compare(ItemData<K, T> o1, ItemData<K, T> o2) {
            return itemComparator.compare(o1.getItem(), o2.getItem());
        }
    }
    
	private static class ItemData<K,T> implements Serializable {
        private static final long serialVersionUID = 2092750648185163932L;
        
        private boolean detailsReload;			// Whether further data must be added to base record to display in a detail page
        private K id;							// Placeholder id or primary of item
        private T item;							// Record, which is null if not yet loaded
        private boolean pageExtensionReload;	// Whether further data must be added to base record to display in a search results page
        private boolean reload;                 // Whether item must be loaded or reloaded. 
        
        public ItemData() {
            detailsReload = true;
            id = null;
            item = null;
            pageExtensionReload = false;
            reload = false;
        }
        
        public ItemData(ItemData<K,T> other, UnaryOperator<T> itemDeepCopier) {
            detailsReload = other.detailsReload;
            id = other.id;
            if (itemDeepCopier != null) {
            	item = itemDeepCopier.apply(other.item);
            } else {
            	item = other.item;
            }
            pageExtensionReload = other.pageExtensionReload;
            reload = other.reload;
        }
        
        /**
         * Whether details, if needed, must be reloaded. 
         */
        public boolean getDetailsReload() {
            return detailsReload;
        }
        public void setDetailsReload(boolean value) {
            detailsReload = value;
        }
        
        /**
         * Returns id or primary key of list item. 
         */
        public K getId() {
            return id;
        }
        public void setId(K value) {
            id = value;
        }
        
        public T getItem() {
            return item;
        }

        public void setItem(T value) {
            item = value;
        }

        /**
         * Returns whether additional record data not loaded by base record list finder need to be loaded to display a 
         * page.  This only applies in base record list pagination mode. 
         */
        public boolean getPageExtensionReload() {
            return pageExtensionReload;
        }
        public void setPageExtensionReload(boolean pageExtensionReload) {
            this.pageExtensionReload = pageExtensionReload;
        }
        
        /**
         * Whether item must be reloaded. 
         */
        public boolean getReload() {
            return reload;
        }
        public void setReload(boolean value) {
            reload = value;
        }
	}

    /**
     * Configuration set by concrete class to control template behaviour for lists.
     *  
	 * @param <M> Record type of master list or Object if this is the master list.
	 * @param <K> Record primary key or id type.
	 * @param <T> Record type.
     */
    public static class ListCacheConfig<M extends Serializable,K extends Serializable,T extends Serializable> {
        private ListFinder<M,T> baseRecordListFinder;
    	private boolean hasDetails = false;
        private IdListFinder<M,K> idListFinder = null; 
        private UnaryOperator<T> itemDeepCopier = null;
        private SingleItemFinder<K,T> itemFinder = null;
        private ListCacheRecordComparator<T> itemSorter = null;
        private KeyExtractor<K,T> keyExtractor = null;
        private ListFinder<M,T> listFinder = null;
        private ListSizeFinder<M> listSizeFinder = null;
        private String name;
        private PageByIdsFinder<K,T> pageByIdsFinder = null;
        private PageByIndexRangeFinder<M,T> pageByIndexRangeFinder = null;
        private PageExtensionAssembler<T> pageExtensionAssembler = null;
        private int pageSize = 10;
        private PaginationMode paginationMode = PaginationMode.FULL_LIST;
        
        public ListCacheConfig() {
            // Empty
        }

        /**
         * Returns base record list finder Strategy for loading base record list, if in base record list mode.
         */
        public ListFinder<M,T> getBaseRecordListFinder() {
            return baseRecordListFinder;
        }
        public void setBaseRecordListFinder(ListFinder<M,T> value) {
        	baseRecordListFinder = value;
        }

        /**
         * Returns whether items have data displayed in detail pages that aren't needed for list pages. 
         */
		public boolean getHasDetails() {
			return hasDetails;
		}
		public void setHasDetails(boolean hasDetails) {
			this.hasDetails = hasDetails;
		}

        /**
         * Returns finder Strategy for loading id list, if in page-by-id mode.
         */
        public IdListFinder<M,K> getIdListFinder() {
            return idListFinder;
        }
        public void setIdListFinder(IdListFinder<M,K> value) {
            idListFinder = value;
        }

        /**
         * Returns Strategy for creating a deep copy of an item, or null if none set. 
         */
        public UnaryOperator<T> getItemDeepCopier() {
			return itemDeepCopier;
		}
		public void setItemDeepCopier(UnaryOperator<T> itemDeepCopier) {
			this.itemDeepCopier = itemDeepCopier;
		}

        /**
         * Returns single item finder Strategy for full details of a single item.
         */
		public SingleItemFinder<K,T> getItemFinder() {
			return itemFinder;
		}
		public void setItemFinder(SingleItemFinder<K,T> itemFinder) {
			this.itemFinder = itemFinder;
		}

		/**
         * Returns item sorter Strategy, if any. 
         */
        public ListCacheRecordComparator<T> getItemSorter() {
            return itemSorter;
        }
        public void setItemSorter(ListCacheRecordComparator<T> value) {
            itemSorter = value;
        }
        
        /**
         * Returns adapter for obtaining unique key, usually the primary key, from records.
         */
		public KeyExtractor<K, T> getKeyExtractor() {
			return keyExtractor;
		}
		public void setKeyExtractor(KeyExtractor<K, T> keyExtractor) {
			this.keyExtractor = keyExtractor;
		}

        /**
         * Returns list finder Strategy for loading full list, if in full list mode.
         */
        public ListFinder<M,T> getListFinder() {
            return listFinder;
        }
        public void setListFinder(ListFinder<M,T> value) {
            listFinder = value;
        }

        /**
         * Returns finder Command for getting list size, if in page-by-index-range mode. 
         */
        public ListSizeFinder<M> getListSizeFinder() {
            return listSizeFinder;
        }
        public void setListSizeFinder(ListSizeFinder<M> listSizeFinder) {
            this.listSizeFinder = listSizeFinder;
        }
        
        /**
         * Returns name of this cache, used in logs.
         */
        public String getName() {
            return name;
        }
        public void setName(String value) {
            name = value;
        }
        
        /**
         * Returns finder Strategy for loading multiple items by id, if in page-by-id mode.
         */
        public PageByIdsFinder<K,T> getPageByIdsFinder() {
            return pageByIdsFinder;
        }
        public void setPageByIdsFinder(PageByIdsFinder<K,T> value) {
            pageByIdsFinder = value;
        }
        
        /**
         * Returns finder Strategy for loading multiple items by index range, if in page-by-index-range mode. 
         */
        public PageByIndexRangeFinder<M,T> getPageByIndexRangeFinder() {
            return pageByIndexRangeFinder;
        }
        public void setPageByIndexRangeFinder(PageByIndexRangeFinder<M,T> pageByIndexRangeFinder) {
            this.pageByIndexRangeFinder = pageByIndexRangeFinder;
        }
        
        /**
         * Returns Transfer Object Assembler for adding record extensions needed for requested page that is not loaded 
         * by a base record list finder.  Only applies for base record list pagination mode.
         */
        public PageExtensionAssembler<T> getPageExtensionAssembler() {
            return pageExtensionAssembler;
        }
        public void setPageExtensionAssembler(PageExtensionAssembler<T> pageExtensionAssembler) {
            this.pageExtensionAssembler = pageExtensionAssembler;
        }
        
		public int getPageSize() {
			return pageSize;
		}
		public void setPageSize(int pageSize) {
			this.pageSize = pageSize;
		}
		
        /**
         * Returns pagination mode that will be used. 
         */
        public PaginationMode getPaginationMode() {
            return paginationMode;
        }
        public void setPaginationMode(PaginationMode paginationMode) {
            this.paginationMode = paginationMode;
        }

    }
    
    private class SlaveListListener implements ListCacheListener<M> {
        private static final long serialVersionUID = -1080730371566520579L;

        public SlaveListListener() {
            // Nothing
        }
        
        @Override
        public void notifyChanged(ListCache<?,?,M> listCache) {
            ListCache.this.markReloadInternal();
        }
    }

    public enum NotifyMode {NONE, NOTIFY, NOTIFYING}
    public enum PaginationMode {BASE_RECORD_LIST, FULL_LIST, PAGE_BY_IDS, PAGE_BY_INDEX_RANGE}
    
    private ListFinder<M,T> baseRecordListFinder;   // Strategy to load base record list in base record list pagination mode
    private boolean hasDetails;                     // Whether records have additional data for a detail page.
    private IdListFinder<M,K> idListFinder;         // Strategy to load ids of list for page by ids pagination mode
    private UnaryOperator<T> itemDeepCopier;		// If not null, creates a deep copy of an item
    private SingleItemFinder<K,T> itemFinder;      // Reloads single item with details
    private ListCacheRecordComparator<T> itemSorter; // Re-sorts entire list
    private KeyExtractor<K,T> keyExtractor;
    private List<ItemData<K,T>> list;               // List of records or placeholders
    private Collection<ListCacheListener<T>> listeners;
    private ListFinder<M,T> listFinder;             // Strategy to load entire list in full list pagination mode
    private ListSizeFinder<M> listSizeFinder;       // Strategy to get list size for page by index pagination mode
    private ListCache<?,?,M> masterList;           // Master list if this cache is a slave list.
    private String name;                            // Name of list used in logging or errors 
    private NotifyMode notifyMode;
    private PageByIdsFinder<K,T> pageByIdsFinder;	// Strategy to retrieve page in page by ids pagination mode
    private PageByIndexRangeFinder<M,T> pageByIndexRangeFinder; // Strategy to retrieve page by indices
    private PageExtensionAssembler<T> pageExtensionAssembler; // Adds additional data for current page for base record list pagination mode
    private int pageSize;
    private PaginationMode paginationMode;
    private boolean reload;                         // Whether list must be reloaded
    private boolean reSort;                         // Whether list must be re-sorted.
    private int selectedIndex;                      // Index of currently selected record, 0 based, or -1 for no selected record

    

    /**
     * Copy constructor.
     */
    public ListCache(ListCache<M,K,T> other, ListCache<?,?,M> newMaster) throws Exception {
    	baseRecordListFinder = other.getBaseRecordListFinder();
    	hasDetails = other.getHasDetails();
        idListFinder = other.getIdListFinder();
        itemDeepCopier = other.getItemDeepCopier();
        itemFinder = other.getItemFinder();
        itemSorter = other.getItemSorter();
        keyExtractor = other.getKeyExtractor();
        list = new ArrayList<ItemData<K,T>>();
        for (ItemData<K,T> otherItem: other.list) {
        	list.add(new ItemData<K,T>(otherItem, itemDeepCopier));
        }
        listeners = new ArrayList<ListCacheListener<T>>();
        listFinder = other.getListFinder();
        listSizeFinder = other.getListSizeFinder();
       	masterList = newMaster;
        name = other.getName();
        notifyMode = NotifyMode.NONE;
        pageByIdsFinder = other.getPageByIdsFinder();
        pageByIndexRangeFinder = other.getPageByIndexRangeFinder();
        pageExtensionAssembler = other.getPageExtensionAssembler();;
        pageSize = other.getPageSize();
        paginationMode = other.getPaginationMode();
        reload = other.getReload();
        reSort = other.getReSort();
        selectedIndex = other.getSelectedIndex();
    }
    
    /**
     * @param listCacheConfig Configuration supplied by template functions.
     */
    public ListCache(ListCacheConfig<M,K,T> listCacheConfig, ListCache<?,?,M> masterList) {
    	baseRecordListFinder = listCacheConfig.getBaseRecordListFinder();
    	hasDetails = listCacheConfig.getHasDetails();
        idListFinder = listCacheConfig.getIdListFinder();
        itemDeepCopier = listCacheConfig.getItemDeepCopier();
        itemFinder = listCacheConfig.getItemFinder();
        itemSorter = listCacheConfig.getItemSorter();
        keyExtractor = listCacheConfig.getKeyExtractor();
        list = new ArrayList<ItemData<K,T>>();
        listeners = new ArrayList<ListCacheListener<T>>();
        listFinder = listCacheConfig.getListFinder();
        listSizeFinder = listCacheConfig.getListSizeFinder();
        this.masterList = masterList;
        name = listCacheConfig.getName();
        notifyMode = NotifyMode.NONE;
        pageByIdsFinder = listCacheConfig.getPageByIdsFinder();
        pageByIndexRangeFinder = listCacheConfig.getPageByIndexRangeFinder();
        pageExtensionAssembler = listCacheConfig.getPageExtensionAssembler();;
        pageSize = listCacheConfig.getPageSize();
        paginationMode = listCacheConfig.getPaginationMode();
        reload = true;
        reSort = false;
        selectedIndex = -1;
        
        // Receive notifications from master list, if this is a slave list.
        if (masterList != null) {
            masterList.addListCacheListener(new SlaveListListener());
        }
    }
    
    private synchronized void addItemInternal(T item) throws IllegalStateException, Exception {
        ItemData<K,T> itemData;
        
        checkReloadAccess();
        // Add new item to list and add item control flags about it.
        itemData = new ItemData<K, T>();
        itemData.setId(keyExtractor.getKey(item));
        itemData.setItem(item);
        list.add(itemData);
        setSelectedIndexInternal(list.size() - 1);
        markChanged();
    }
    /**
     * Checks all list data has been loaded, not just placeholders such as base record or id.  Records can be 
     * deleted if the page-by-id finder can't find it.   
     */
    private void checkEntireListData() throws Exception {
        switch (paginationMode) {
        case BASE_RECORD_LIST:
            checkEntireListDataByBaseRecords();
            break;
        case FULL_LIST:
            // Entire list already loaded
            break;
        case PAGE_BY_IDS:
            checkEntireListDataByIds();
            break;
        case PAGE_BY_INDEX_RANGE:
            checkEntireListDataByIndexRange();
            break;
        }
    }

    /**
     * Assembles any missing data for all items not loaded by the base record list finder.   
     */
    private void checkEntireListDataByBaseRecords() throws Exception {
        ItemData<K,T> itemData;
        T item;
        int size, i;
        Collection<T> loadingItems;
        
        size = list.size();
        loadingItems = new ArrayList<>();
        for (i = 0; i < size; i++) {
            itemData = list.get(i);
            item = itemData.getItem();
            loadingItems.add(item);
        }
        pageExtensionAssembler.assemblePageExtensions(loadingItems);
        for (i = 0; i < size; i++) {
            itemData = list.get(i);
            itemData.setPageExtensionReload(false);
        }
    }

    /**
     * Loads any missing records from their ids.   
     */
    private void checkEntireListDataByIds() throws Exception {
        ItemData<K,T> itemData;
        K id;
        T item;
        Collection<K> ids;
        Collection<T> loadingItems;
        List<Integer> removalIndices;
        Map<K,Integer> loadingIds;
        int i, index, size;
        boolean detailLoaded;
        
        size = list.size();
        // Find items that need to be loaded
        ids = new ArrayList<K>();
        loadingIds = new HashMap<K,Integer>();
        for (i = 0; i < size; i++) {
            itemData = list.get(i);
            if (itemData.getReload()) {
                ids.add(itemData.getId());
                loadingIds.put(itemData.getId(), i);
            }
        }
        // Load items from page-by-id finder
        loadingItems = pageByIdsFinder.getItems(ids);
        detailLoaded = pageByIdsFinder.getLoadsDetails();
        for (T loadingItem: loadingItems) {
            id = keyExtractor.getKey(loadingItem);
            index = loadingIds.get(id);
            itemData = list.get(index);
            itemData.setDetailsReload(!detailLoaded);
            itemData.setItem(loadingItem);
            itemData.setPageExtensionReload(false);
            itemData.setReload(false);
        }
        // Delete items that page-by-id finder did not load as they no longer exist
        removalIndices = new ArrayList<Integer>();
        for (K loadingId: loadingIds.keySet()) {
            index = loadingIds.get(loadingId);
            item = list.get(index).getItem();
            if (item == null) {
                removalIndices.add(index);
            }
        }
        Collections.sort(removalIndices, Collections.reverseOrder());
        for (int removeIndex: removalIndices) {
            list.remove(removeIndex);
        }
    }
    
    /**
     * Loads any missing records from their indices.   
     */
    private void checkEntireListDataByIndexRange() throws Exception {
        ItemData<K,T> itemData;
        M selectedMaster;
        T item;
        List<T> loadingItemList;
        int i, size, rangeSize, lowestNotLoaded, highestNotLoaded;
        boolean detailLoaded;
        
        size = list.size();
        // Find what needs to be loaded.  If that's multiple ranges, one big load is done anyway as it's probably faster
        // than multiple loads
        lowestNotLoaded = -1;
        highestNotLoaded = -1;
        for (i = 0; i < size; i++) {
            if (list.get(i).getReload()) {
                if (lowestNotLoaded < 0) {
                    lowestNotLoaded = i;
                }
                highestNotLoaded = i;
            }
        }
        
        if (lowestNotLoaded > -1) {
            rangeSize = highestNotLoaded - lowestNotLoaded + 1;
            if (masterList == null) {
                selectedMaster = null;
            } else {
                selectedMaster = masterList.getSelectedInternal();
            }
            loadingItemList = pageByIndexRangeFinder.getItems(selectedMaster, lowestNotLoaded, highestNotLoaded);
            detailLoaded = pageByIndexRangeFinder.getLoadsDetails();
            // If result size less than requested range, end of source list unexpectedly reached, so shrink our list 
            if (rangeSize > loadingItemList.size()) { 
                list.subList(lowestNotLoaded + loadingItemList.size(), size).clear();
            }
            // If result size greater than requested range, WTF?, ignore the excess
            // Load items from page-by-index-range finder
            for (i = 0; i < loadingItemList.size() && i < rangeSize; i++) {
                item = loadingItemList.get(i);
                itemData = list.get(i + lowestNotLoaded);
                itemData.setDetailsReload(!detailLoaded);
                itemData.setId(keyExtractor.getKey(item));
                itemData.setItem(item);
                itemData.setPageExtensionReload(false);
                itemData.setReload(false);
            }
        }
    }
    
    /**
     * Loads any missing data for items in a range of indices, usually the current page.  Records can be deleted if in 
     * page-by-id mode and the page-by-id page finder can't find the required records.   
     */
    private void checkPageListData(int startIndex, int endIndex) throws Exception {
        switch (paginationMode) {
        case BASE_RECORD_LIST:
            checkPageListDataByBaseRecords(startIndex, endIndex);
            break;
        case FULL_LIST:
            // Entire list already loaded
            break;
        case PAGE_BY_IDS:
            checkPageListDataByIds(startIndex, endIndex);
            break;
        case PAGE_BY_INDEX_RANGE:
            checkPageListDataByIndexRange(startIndex, endIndex);
            break;
        }
    }

    /**
     * Assembles any missing data for items in a range of indices, usually the current page, not loaded by the base 
     * record list finder.   
     */
    private void checkPageListDataByBaseRecords(int startIndex, int endIndex) throws Exception {
        ItemData<K,T> itemData;
        T item;
        int size, i, workingStartIndex, workingEndIndex;
        Collection<T> loadingItems;
        
        size = list.size();
        workingStartIndex = (startIndex >= 0 && startIndex < size)?startIndex:0;
        workingEndIndex = (endIndex >= 0 && endIndex < size)?endIndex:(size-1);
        loadingItems = new ArrayList<>();
        for (i = workingStartIndex; i <= workingEndIndex; i++) {
            itemData = list.get(i);
            item = itemData.getItem();
            if (itemData.getPageExtensionReload()) {
                loadingItems.add(item);
            }
        }
        if (loadingItems.size() > 0) {
            pageExtensionAssembler.assemblePageExtensions(loadingItems);
            for (i = workingStartIndex; i <= workingEndIndex; i++) {
                itemData = list.get(i);
                itemData.setPageExtensionReload(false);
            }
        }
    }
    
    /**
     * Loads any missing records for items in a range of indices, usually the current page, from their ids.
     */
    private void checkPageListDataByIds(int startIndex, int endIndex) throws Exception {
        ItemData<K,T> itemData;
        Collection<K> ids;
        Collection<T> loadingItems;
        List<Integer> removalIndices;
        Map<K,Integer> loadingIds;
        K id;
        T item;
        int i, index, listSize;
        boolean loadFailed;
        
        do {
            loadFailed = false;
            listSize = list.size();
            // Find items that need to be loaded
            ids = new ArrayList<K>();
            loadingIds = new HashMap<K,Integer>();
            for (i = startIndex; i <= endIndex; i++) {
                if (i >= 0 && i <= listSize - 1) {
                    itemData = list.get(i);
                    if (itemData.getReload()) {
                        ids.add(itemData.getId());
                        loadingIds.put(itemData.getId(), i);
                    }
                }
            }
            if (ids.size() > 0) {
                // Load items from multi item finder
                loadingItems = pageByIdsFinder.getItems(ids);
                for (T loadingItem: loadingItems) {
                    id = keyExtractor.getKey(loadingItem);
                    index = loadingIds.get(id);
                    itemData = list.get(index); 
                    itemData.setDetailsReload(!pageByIdsFinder.getLoadsDetails());
                    itemData.setItem(loadingItem);
                    itemData.setReload(false);
                    itemData.setPageExtensionReload(false);
                }
                // Delete items that multi item finder did not load as they no longer exist
                removalIndices = new ArrayList<Integer>();
                for (K loadingId: loadingIds.keySet()) {
                    index = loadingIds.get(loadingId);
                    item = list.get(index).getItem();
                    if (item == null) {
                        removalIndices.add(index);
                        loadFailed = true;
                    }
                }
                Collections.sort(removalIndices, Collections.reverseOrder());
                for (int removalIndex: removalIndices) {
                    list.remove(removalIndex);
                }
            }
        }
        // If there any failed items, which are removed, higher entries may 
        // shift to take their place in the requested page and some may be unloaded.
        while (loadFailed);
    }
    
    /**
     * Loads any missing records for items in a range of indices, usually the current page, from the indices.
     */
    private void checkPageListDataByIndexRange(int startIndex, int endIndex) throws Exception {
        ItemData<K,T> itemData;
        M selectedMaster;
        T item;
        List<T> loadingItemList;
        int i, listSize, rangeSize, lowestNotLoaded, highestNotLoaded;
        boolean detailLoaded;
        
        listSize = list.size();
        // Find what needs to be loaded.  If that's multiple ranges, one big load is done anyway as it's probably faster
        // multiple loads
        lowestNotLoaded = -1;
        highestNotLoaded = -1;
        for (i = startIndex; i <= endIndex; i++) {
            if (list.get(i).getReload()) {
                if (lowestNotLoaded < 0) {
                    lowestNotLoaded = i;
                }
                highestNotLoaded = i;
            }
        }
        
        if (lowestNotLoaded > -1) {
            rangeSize = highestNotLoaded - lowestNotLoaded + 1;
            if (masterList == null) {
                selectedMaster = null;
            } else {
                selectedMaster = masterList.getSelectedInternal();
            }
            loadingItemList = pageByIndexRangeFinder.getItems(selectedMaster, lowestNotLoaded, highestNotLoaded);
            detailLoaded = pageByIndexRangeFinder.getLoadsDetails();
            // If result size less than requested range, end of source list unexpectedly reached, so shrink our list 
            if (rangeSize > loadingItemList.size()) {
                list.subList(startIndex + loadingItemList.size(), listSize).clear();
            }
            // If result size greater than requested range, WTF?, ignore the excess
            // Load items from page-by-index-range finder
            for (i = 0; i < loadingItemList.size() && i < rangeSize; i++) {
                item = loadingItemList.get(i);
                itemData = list.get(i + lowestNotLoaded);
                itemData.setDetailsReload(!detailLoaded);
                itemData.setId(keyExtractor.getKey(item));
                itemData.setItem(item);
                itemData.setPageExtensionReload(false);
                itemData.setReload(false);
            }
        }
    }
    
    /**
     * Checks list has been loaded and accessed.  This is used by functions that change records as changing a record 
     * without checking it even exists is an error.
     */
    private void checkReloadAccess() throws IllegalStateException {
        if (reload) {
            throw new IllegalStateException("Attempt to blindly set selected record for " + 
                    name + " list without accessing list first.");
        }
    }
    
    /**
     * Checks if details need to be loaded for records, loading them if not.
     */
    private void checkReloadItemDetail(int startIndex, int endIndex) throws Exception {
        Set<Integer> indicesToLoad;
        boolean reloadNeeded, reloadList;
        
        indicesToLoad = new HashSet<>();
        for (int index = startIndex; index <= endIndex; index++) {
            reloadNeeded = getItemReloadNeeded(index);
            if (reloadNeeded) {
                indicesToLoad.add(index);
            }
        }
        if (indicesToLoad.size() > 0) {
            if (hasDetails && itemFinder == null) {
                throw new IllegalStateException("Single item finder for " + name + 
                    " list is not set but details need to be reloaded.");
            } else if (itemFinder == null)  {
                reloadList = true;
            } else {
                reloadList = !loadItemDetails(indicesToLoad);
            }
            if (reloadList) {
                loadList();
            }
        }
    }
    
    /**
     * Checks if items were marked for reload, loading if so. 
     */
    private void checkReloadItems(int startIndex, int endIndex) throws Exception {
        ItemData<K,T> itemData;
        Set<Integer> indicesToLoad;
        boolean reloadList;
        
        indicesToLoad = new HashSet<>();
        for (int index = startIndex; index <= endIndex; index++) {
            itemData = list.get(index);
            if (itemData.getReload()) {
                indicesToLoad.add(index);
            }
        }
        if (indicesToLoad.size() > 0) {
            if (itemFinder == null)  {
                reloadList = true;
            } else {
                reloadList = !loadItemDetails(indicesToLoad);
            }
            if (reloadList) {
                loadList();
                checkSelectedIndex();
            }
        }
    }
    
    /**
     * Loads list if not yet loaded or reload requested.
     */
    private synchronized void checkReloadState() throws Exception {
        if (reload) {
            loadList();
        } else {
        	checkSort();
        }
    }
    
    /**
     * Ensures selected item index is within list boundary or -1 for empty list.
     */
    private void checkSelectedIndex() {
        int size;
        
        size = list.size();
        if (selectedIndex >= size) {
            selectedIndex = size - 1;
            markChanged();
        } else if (selectedIndex < 0 && size > 0) {
            selectedIndex = 0;
            markChanged();
        }
    }
    
    /**
     * Sorts entire list if requested and an item sorter is set.
     */
    private void checkSort() throws Exception {
    	if (reSort && itemSorter != null) {
	        switch (paginationMode) {
			case BASE_RECORD_LIST:
            	if (itemSorter.usesPageExtensionData()) {
            		checkEntireListDataByBaseRecords();
            	}
				break;
			case FULL_LIST:
				// Full list already loaded
				break;
			case PAGE_BY_IDS:
				checkEntireListDataByIds();
				break;
			case PAGE_BY_INDEX_RANGE:
				checkEntireListDataByIndexRange();
				break;
	        }
        	Collections.sort(list, new ItemDataComparator<K,T>(itemSorter));
            markChanged();
    	}
        reSort = false;
    }
    
    private synchronized void forceReloadInternal() throws Exception {
        list.clear();
        reload = true;
        loadList();
    }
    /**
     * Determine if selected item or its extensions needs to be reloaded.
     */
    private boolean getItemReloadNeeded(int index) {
        ItemData<K,T> itemData;
        boolean reloadNeeded;
        
        if (index >= 0) {
            itemData = list.get(index);
            if (itemData.getItem() == null || itemData.getReload()) {
                reloadNeeded = true;
            } else if (hasDetails && itemData.getDetailsReload()) {
                reloadNeeded = true;
            } else {
                reloadNeeded = false;
            }
        } else {
            reloadNeeded = false;
        }
        
        return reloadNeeded;
    }
    
    private synchronized List<T> getListInternal() throws Exception {
        checkReloadState();
        checkEntireListData();
        checkReloadItems(0, list.size()-1);
        return getListUnchecked();
    }
    
    /**
     * Returns cached, record list from session, if it is present.  
     */
    private List<T> getListUnchecked() {
        List<T> result;
        
        result = new ArrayList<T>(list.size());
        for (ItemData<K,T> itemData: list) {
            result.add(itemData.getItem());
        }
        return result;
    }

    private synchronized List<T> getPageInternal() throws Exception {
        int pageNo, startIndex, endIndex;
        
        checkReloadState();
        if (list.size() > 0) {
            pageNo = getPageNoInternal();
            pageNo = (pageNo < 1)?1:pageNo;
            startIndex =  (pageNo - 1) * pageSize;
            endIndex = startIndex + pageSize - 1;
            endIndex = (endIndex >= list.size())?list.size() - 1:endIndex;
            checkPageListData(startIndex, endIndex);
            // checkPageListData can shrink the list as it discovers items no longer exist
            startIndex = (startIndex >= list.size())?list.size() - 1:startIndex;
            startIndex = (startIndex < 0)?0:startIndex;
            endIndex = (endIndex >= list.size())?list.size() - 1:endIndex;
            checkReloadItems(startIndex, endIndex);
            return getListUnchecked().subList(startIndex, endIndex + 1);
        } else {
            return new ArrayList<T>();
        }
    }
    
    private int getPageNoInternal() throws Exception {
    	if (selectedIndex > -1) {
    		return (selectedIndex / pageSize) + 1;
    	} else {
    		return 0;
    	}
    }
    
    private synchronized List<T> getRangeInternal(int startIndex, int endIndex) throws Exception {
        int workingStart, workingEnd;
        
        checkReloadState();
        if (list.size() > 0) {
            workingStart = startIndex;
            workingStart = (workingStart < 0)?0:workingStart;
            workingStart = (workingStart >= list.size())?list.size() - 1:workingStart;
            workingStart = (workingStart < 0)?0:workingStart;
            workingEnd = endIndex;
            workingEnd = (workingEnd < 0)?0:workingEnd;
            workingEnd = (workingEnd >= list.size())?list.size() - 1:workingEnd;
            workingEnd = (workingEnd < 0)?0:workingEnd;
            
            checkPageListData(workingStart, workingEnd);
            // checkPageListData can shrink the list as it discovers items no longer exist
            workingStart = (workingStart >= list.size())?list.size() - 1:workingStart;
            workingStart = (workingStart < 0)?0:workingStart;
            workingEnd = (workingEnd >= list.size())?list.size() - 1:workingEnd;
            return getListUnchecked().subList(workingStart, workingEnd + 1);
        } else {
            return new ArrayList<T>();
        }
    }
    
    private synchronized T getSelectedInternal() throws Exception {
        checkReloadState();
        if (selectedIndex >= 0) {
        	checkPageListData(selectedIndex,selectedIndex);
            checkReloadItemDetail(selectedIndex,selectedIndex);
            return list.get(selectedIndex).getItem();
        } else {
            return null;
        }
    }
    /**
     * Loads individual items using the item finder and returns false if no record was found for any key. 
     */
    private boolean loadItemDetails(Collection<Integer> indices) throws Exception {
        ItemData<K,T> itemData;
        T item;
        boolean failed;
        
        failed = false;
        for (int index: indices) {
            itemData = list.get(index);
            item = itemFinder.getItem(itemData.getId());
            if (item != null) {
                itemData.setItem(item);
                itemData.setReload(false);
                itemData.setDetailsReload(false);
                markChanged();
            } else {
                failed = true;
            }
        }
        
        return !failed;
    }
    
    /**
     * Loads records or placeholders, according to pagination mode.
     */
    private void loadList() throws Exception {
        List<K> idList;
        List<T> itemList;
        M selectedMaster;
        int listSize;
        boolean pageExtensionsLoaded;
        
        if (masterList == null) {
            selectedMaster = null;
        } else {
            selectedMaster = masterList.getSelectedInternal();
        }
        switch (paginationMode) {
        case BASE_RECORD_LIST:
            itemList = baseRecordListFinder.getList(selectedMaster);
            pageExtensionsLoaded = baseRecordListFinder.getLoadsDetails();
            if (itemList == null) {
                itemList = new ArrayList<T>();
            } else if (itemSorter != null) {
            	if (!pageExtensionsLoaded && itemSorter.usesPageExtensionData()) {
                    pageExtensionAssembler.assemblePageExtensions(itemList);
            	}
                Collections.sort(itemList, itemSorter);
            }
            setList(itemList, pageExtensionsLoaded, baseRecordListFinder.getLoadsDetails());
            break;
        case FULL_LIST:
            itemList = listFinder.getList(selectedMaster);
            if (itemList == null) {
                itemList = new ArrayList<T>();
            } else if (itemSorter != null) {
                Collections.sort(itemList, itemSorter);
            }
            setList(itemList, true, listFinder.getLoadsDetails());
            break;
        case PAGE_BY_IDS:
            idList = idListFinder.getIds(selectedMaster);
            if (idList == null) {
                idList = new ArrayList<K>();
            }
            setIdList(idList);
	        if (itemSorter != null) {
	        	checkEntireListDataByIds();
	            Collections.sort(list, new ItemDataComparator<K,T>(itemSorter));
	        }
            break;
        case PAGE_BY_INDEX_RANGE:
            listSize = this.listSizeFinder.getSize(selectedMaster);
            setListSize(listSize);
	        if (itemSorter != null) {
	        	checkEntireListDataByIndexRange();
	            Collections.sort(list, new ItemDataComparator<K,T>(itemSorter));
	        }
            break;
        }
        markChanged();
    }

    private synchronized void markReloadInternal() {
        list.clear();
        reload = true;
        markChanged();
    }
    
    private synchronized void markReSortInternal() {
        reSort = true;
    }
    
    private synchronized T removeInternal(int index) throws IllegalStateException, Exception {
    	T result;
        ItemData<K, T> itemData;
        
        result = null;
        checkReloadAccess();
        itemData = list.get(index);
        if (itemData != null) {
            list.remove(index);
            checkSelectedIndex();
            result = itemData.getItem();
            markChanged();
        }
        return result;
    }

    private synchronized T removeSelectedInternal() throws IllegalStateException, Exception {
        T result;
        
        checkReloadAccess();
        if (selectedIndex >= 0) {
            result = list.get(selectedIndex).getItem();
            list.remove(selectedIndex);
            checkSelectedIndex();
            markChanged();
        } else {
            result = null;
        }
        return result;
    }

    private synchronized void setBaseRecordListAndFinderInternal(List<T> list, ListFinder<M,T> baseRecordListFinder) {
        if (baseRecordListFinder == null) {
            throw new IllegalStateException("List finder for " + name + " list must be set with its list.");
        }
        setList(list, baseRecordListFinder.getLoadsDetails(), baseRecordListFinder.getLoadsDetails());
        this.baseRecordListFinder = baseRecordListFinder;
        this.idListFinder = null;
        this.listFinder = null;
        this.listSizeFinder = null;
        this.pageByIndexRangeFinder = null;
        this.paginationMode = PaginationMode.BASE_RECORD_LIST;
        markReSort();
        setSelectedIndexInternal(0);
        markChanged();
    }
    
    private void setBaseRecordListFinderInternal(ListFinder<M,T> baseRecordListFinder) {
        if (baseRecordListFinder == null) {
            throw new IllegalStateException("Base record list finder for " + name + " list must be set.");
        }
        setList(new ArrayList<>(), baseRecordListFinder.getLoadsDetails(), baseRecordListFinder.getLoadsDetails());
        this.baseRecordListFinder = baseRecordListFinder;
        this.idListFinder = null;
        this.listFinder = null;
        this.listSizeFinder = null;
        this.pageByIndexRangeFinder = null;
        this.paginationMode = PaginationMode.BASE_RECORD_LIST;
        markReloadInternal();
    }
    
    /**
     * Caches page-by-id record markers, which are usually the record primary keys.
     */
    private synchronized void setIdList(List<K> ids) {
        ItemData<K,T> itemData;

        // Save list data and set control flags 
        reload = false;
        // Set and save item control flags
        list = new ArrayList<ItemData<K,T>>(ids.size());
        for (K id: ids) {
            itemData = new ItemData<K,T>();
            itemData.setId(id);
            itemData.setItem(null);
            itemData.setPageExtensionReload(false);
            itemData.setReload(true);
            list.add(itemData);
        }
        checkSelectedIndex();
        markChanged();
    }
    
    private synchronized void setIdListAndFinderInternal(List<K> idList, IdListFinder<M,K> idListFinder) {
        if (idListFinder == null) {
            throw new IllegalStateException("Id list finder for " + 
                    name + " list must be set with its id list.");
        }
        setIdList(idList);
        this.baseRecordListFinder = null;
        this.idListFinder = idListFinder;
        this.listFinder = null;
        this.listSizeFinder = null;
        this.pageByIndexRangeFinder = null;
        this.paginationMode = PaginationMode.PAGE_BY_IDS;
        markReSortInternal();
        setSelectedIndexInternal(0);
        markChanged();
    }
    
    private synchronized void setIdListFinderInternal(IdListFinder<M,K> idListFinder) {
        if (idListFinder == null) {
            throw new IllegalStateException("Id list finder for " + 
                    name + " list must be set with its id list.");
        }
        setIdList(new ArrayList<>());
        this.baseRecordListFinder = null;
        this.idListFinder = idListFinder;
        this.listFinder = null;
        this.listSizeFinder = null;
        this.pageByIndexRangeFinder = null;
        this.paginationMode = PaginationMode.PAGE_BY_IDS;
        markReloadInternal();
    }
    
    /**
     * Caches new, record list.
     */
    private synchronized void setList(List<T> value, boolean pageExtensionsLoaded, boolean detailLoaded) {
        ItemData<K,T> itemData;

        // Save list data and set control flags 
        reload = false;
        // Set and save item control flags
        list = new ArrayList<ItemData<K,T>>(value.size());
        for (T item: value) {
            itemData = new ItemData<K,T>();
            itemData.setDetailsReload(!detailLoaded);
            itemData.setId(keyExtractor.getKey(item));
            itemData.setItem(item);
            itemData.setPageExtensionReload(!pageExtensionsLoaded);
            itemData.setReload(false);
            list.add(itemData);
        }
        checkSelectedIndex();
        markChanged();
    }

    private synchronized void setListAndFinderInternal(List<T> list, ListFinder<M,T> listFinder) {
        if (listFinder == null) {
            throw new IllegalStateException("List finder for " + 
                    name + " list must be set with its list.");
        }
        setList(list, true, listFinder.getLoadsDetails());
        this.baseRecordListFinder = null;
        this.idListFinder = null;
        this.listFinder = listFinder;
        this.listSizeFinder = null;
        this.pageByIndexRangeFinder = null;
        this.paginationMode = PaginationMode.FULL_LIST;
        markReSort();
        setSelectedIndexInternal(0);
        markChanged();
    }
    
    private synchronized void setListFinderInternal(ListFinder<M,T> listFinder) {
        if (listFinder == null) {
            throw new IllegalStateException("List finder for " + 
                    name + " list must be set with its list.");
        }
        setList(new ArrayList<>(), true, listFinder.getLoadsDetails());
        this.baseRecordListFinder = null;
        this.idListFinder = null;
        this.listFinder = listFinder;
        this.listSizeFinder = null;
        this.pageByIndexRangeFinder = null;
        this.paginationMode = PaginationMode.FULL_LIST;
        markReloadInternal();
    }
    
    /**
     * Caches page-by-index-range record markers. 
     */
    private synchronized void setListSize(int listSize) {
        ItemData<K,T> itemData;
        int i;

        // Save list data and set control flags 
        reload = false;
        // Set and save item control flags
        list = new ArrayList<ItemData<K,T>>(listSize);
        for (i = 0; i < listSize; i++) {
            itemData = new ItemData<K,T>();
            itemData.setDetailsReload(true);
            itemData.setId(null);
            itemData.setItem(null);
            itemData.setPageExtensionReload(false);
            itemData.setReload(true);
            list.add(itemData);
        }
        checkSelectedIndex();
        markChanged();
    }
    
    private synchronized void setListSizeAndFinderInternal(int listSize, ListSizeFinder<M> listSizeFinder, 
            PageByIndexRangeFinder<M,T> pageByIndexRangeFinder) {
        if (listSizeFinder == null || pageByIndexRangeFinder == null) {
            throw new IllegalStateException("List size finder pr page by index tange finder for " + 
                    name + " list must be set with its list size.");
        }
        setListSize(listSize);
        this.baseRecordListFinder = null;
        this.idListFinder = null;
        this.listFinder = null;
        this.listSizeFinder = listSizeFinder;
        this.pageByIndexRangeFinder = pageByIndexRangeFinder;
        this.paginationMode = PaginationMode.PAGE_BY_INDEX_RANGE;
        markReSort();
        setSelectedIndexInternal(0);
        markChanged();
    }
    
    private synchronized void setListSizeFinderInternal(ListSizeFinder<M> listSizeFinder, PageByIndexRangeFinder<M,T> pageByIndexRangeFinder) {
        if (listSizeFinder == null || pageByIndexRangeFinder == null) {
            throw new IllegalStateException("List size finder pr page by index tange finder for " + 
                    name + " list must be set with its list size.");
        }
        setListSize(0);
        this.baseRecordListFinder = null;
        this.idListFinder = null;
        this.listFinder = null;
        this.listSizeFinder = listSizeFinder;
        this.pageByIndexRangeFinder = pageByIndexRangeFinder;
        this.paginationMode = PaginationMode.PAGE_BY_INDEX_RANGE;
        markReloadInternal();
    }

    private synchronized void setSelectedInternal(T value) {
        checkReloadAccess();
        if (selectedIndex != -1) {
            setSelectedItemUnchecked(value);
        }
        markChanged();
    }

    /**
     * Replaces selected record of list.  Current, selected item index cannot be -1.
     */
    private void setSelectedItemUnchecked(T item) {
        list.get(selectedIndex).setItem(item);
        markChanged();
    }
    
    private synchronized void setSelectedIndexInternal(int index) {
        int oldIndex;
        int size;
        
        checkReloadAccess();
        oldIndex = selectedIndex;
        size = list.size();
        if (index >= size) {
            selectedIndex = size - 1;
        } else if (index < 0) {
            selectedIndex = -1;
        } else {
            selectedIndex = index;
        }
        if (index != oldIndex) {
            markChanged();
        }
    }
    
    protected synchronized void markChanged() {
    	if (notifyMode == NotifyMode.NONE) {
        	notifyMode = NotifyMode.NOTIFY;
    	}
    }
    
    /**
     * Notification of change.  This typically notifies the parent container.  
     */
    protected void notifyChanged() {
        if (notifyMode == NotifyMode.NOTIFY) {
            try {
            	notifyMode = NotifyMode.NOTIFYING;
                for (ListCacheListener<T> listener: listeners) {
                    listener.notifyChanged(this);
                }
            }
            finally {
            	notifyMode = NotifyMode.NONE;
            }
        }
    }
    
    public void addListCacheListener(ListCacheListener<T> listener) {
        listeners.add(listener);
    }
    
    /**
     * Adds new item to end of list and sets it as selected item.
     * @param item New item to add to end of list.
     * 
     * @throws IllegalStateException List must be reloaded but list finder 
     *                               is missing.
     * @throws Exception Failed to retrieve extensions.
     */
    public void addItem(T item) throws IllegalStateException, Exception {
        addItemInternal(item);
        notifyChanged();
    }
    
    /**
     * Reloads list using currently set list finder, id list finder, or list size finder.
     */
    public void forceReload() throws Exception {
    	forceReloadInternal();
        notifyChanged();
    }
    
    /**
     * Returns Strategy object used to retrieve list.
     */
    public ListFinder<M,T> getBaseRecordListFinder() {
        return baseRecordListFinder;
    }

    /**
     * Returns whether records must lazy load additional data for detail pages. 
     */
	public boolean getHasDetails() {
		return hasDetails;
	}

    /**
     * Returns Strategy used to retrieve list.
     */
    public IdListFinder<M,K> getIdListFinder() {
        return idListFinder;
    }

    /**
     * Returns Strategy for creating a deep copy of an item, or null if none set. 
     */
	public UnaryOperator<T> getItemDeepCopier() {
		return itemDeepCopier;
	}

    /**
     * Returns Strategy used to retrieve single item of list, including any extensions, or null if not set. 
     */
    public SingleItemFinder<K,T> getItemFinder() {
        return itemFinder;
    }

    /**
     * Returns Strategy used to sort list, or null for no sorting. 
     */
    public ListCacheRecordComparator<T> getItemSorter() {
        return itemSorter;
    }

    /**
     * Returns Strategy that extracts the primary key from a record.
     */
	public KeyExtractor<K,T> getKeyExtractor() {
		return keyExtractor;
	}

    /**
     * Returns entire record list, loading it if needed.
     */
    public List<T> getList() throws Exception {
    	List<T> result;
    	
        result = getListInternal();
        notifyChanged();
        return result;
    }

    /**
     * Returns all listeners of any change to this. 
     */
	public Collection<ListCacheListener<T>> getListeners() {
		return listeners;
	}

    /**
     * Returns Strategy object used to retrieve list.
     */
    public ListFinder<M,T> getListFinder() {
        return listFinder;
    }

    /**
     * Returns list size. 
     */
    public int getListSize() throws Exception {
        checkReloadState();
	    notifyChanged();
        return list.size();
    }

    /**
     * Returns Strategy object used to retrieve list size.
     */
    public ListSizeFinder<M> getListSizeFinder() {
        return listSizeFinder;
    }

    /**
     * Returns master list cache, where records of this slave list cache are related to selected record of master list 
     * cache, or null if this is not a slave list.     
     */
	public ListCache<?,?,M> getMasterList() {
		return masterList;
	}

    public String getName() {
        return name;
    }
    
    /**
     * Returns current page containing selected record, loading records if needed. 
     */
    public List<T> getPage() throws Exception {
    	List<T> result;
    	
        result = getPageInternal();
        notifyChanged();
        return result;
    }
    
    /**
     * Returns Strategy object used to retrieve page by their record primary keys. 
     */
    public PageByIdsFinder<K,T> getPageByIdsFinder() {
        return pageByIdsFinder;
    }
    
    /**
     * Returns Strategy object used to retrieve page by their indices. 
     */
    public PageByIndexRangeFinder<M,T> getPageByIndexRangeFinder() {
        return pageByIndexRangeFinder;
    }
    
    /**
     * Returns Transfer Object Assembler for adding record extensions needed for requested page that is not loaded 
     * by a base record list finder.  Only applies for base record list pagination mode.
     */
    public PageExtensionAssembler<T> getPageExtensionAssembler() {
        return pageExtensionAssembler;
    }

    /**
     * Returns number of current page, starting at 1, containing selected record. 
     */
    public int getPageNo() throws Exception {
    	return getPageNoInternal();
    }
    
    public int getPageSize() throws Exception {
         return pageSize;
    }
    
    /**
     * Returns how pages are retrieved. 
     */
    public PaginationMode getPaginationMode() {
        return paginationMode;
    }
    
    /**
     * Returns range of records from cached list, loading them if needed.  If the start or end are out of range, they 
     * are treated as the index of the nearest item.  Thus, at least one item is always returned unless no items exist 
     * at all.  
     */
    public List<T> getRange(int startIndex, int endIndex) throws Exception {
    	List<T> result;
    	
    	result = getRangeInternal(startIndex, endIndex);
        notifyChanged();
        return result;
    }
    
    /**
     * Indicates if record list must be reloaded.
     */
    public boolean getReload() {
        return reload;
    }

    /**
     * Indicates if record list must be sorted.  
     */
	public boolean getReSort() {
		return reSort;
	}

    /**
     * Returns selected item from list.
     */
    public T getSelected() throws Exception {
    	T result;
    	
    	result = getSelectedInternal();
        notifyChanged();
        return result;
    }
    
    /**
     * Returns selected item from list, which won't automatically load details.
     * @throws IllegalStateException List finder not set and reload required.
     */
    public synchronized T getSelectedNoDetail() throws Exception {
        checkReloadState();
        if (selectedIndex >= 0) {
            checkPageListData(selectedIndex,selectedIndex);
            return list.get(selectedIndex).getItem();
        } else {
            return null;
        }
    }
    
    /**
     * Returns index of currently selected item, starting at 0, in list.
     */
    public int getSelectedIndex() throws Exception {
        checkReloadState();
        notifyChanged();
        return selectedIndex;
    }
    
    public int getTotalPages() {
		return list.size()/pageSize + (list.size()%pageSize == 0?0:1);
	}
    
    /**
     * Marks record list as requiring a reload.
     */
    public void markReload() {
    	markReloadInternal();
        notifyChanged();
    }
    
    /**
     * Marks record list as requiring a re-sort.
     */
    public void markReSort() {
    	markReSortInternal();
        notifyChanged();
    }

    /**
     * Marks single item in list as requiring a reload.
     */
    public synchronized void markSelectedReload() {
        ItemData<K,T> itemData;
        
        if (itemFinder != null) {
            itemData = list.get(selectedIndex);
            itemData.setReload(true);
            notifyChanged();
        } else {
            // No item finder means entire list must be reloaded
            markReload();
        }
    }

    /**
     * Removes item from list at index position and returns it. Returns null if no item exists at index.
     * 
     * @throws IllegalStateException List must be reloaded but list finder is missing.
     * @throws Exception Failed to retrieve extensions.
     */
    public T remove(int index) throws IllegalStateException, Exception {
    	T result;
    	
    	result = removeInternal(index);
        notifyChanged();
        return result;
    }

    public void removeListCacheListener(ListCacheListener<T> listener) {
        listeners.remove(listener);
    }

    /**
     * Removes currently, selected item from list and returns it.
     * <P>Returns null if the list is empty and there is no selected item.</P>
     * 
     * @throws IllegalStateException List must be reloaded but list finder 
     *                               is missing.
     * @throws Exception Failed to retrieve extensions.
     */
    public T removeSelected() throws IllegalStateException, Exception {
        T result;
        
        result = removeSelectedInternal();
        if (result != null) {
            notifyChanged();
        }
        return result;
    }

    /**
     * Caches list already loaded, sets base record list finder Strategy used to load it, and sets base record list pagination mode.
     */
    public void setBaseRecordListAndFinder(List<T> list, ListFinder<M,T> baseRecordListFinder) {
    	setBaseRecordListAndFinderInternal(list, baseRecordListFinder);
        notifyChanged();
    }
    
    /**
     * Sets base record list finder, marks list for reload, and sets base record list pagination mode..
     */
    public void setBaseRecordListFinder(ListFinder<M,T> baseRecordListFinder) {
        setBaseRecordListFinderInternal(baseRecordListFinder);
        notifyChanged();
    }
    
    /**
     * Caches id list already loaded, sets id list finder Strategy used to load it, and sets page by ids pagination mode.
     */
    public void setIdListAndFinder(List<K> idList, IdListFinder<M,K> idListFinder) {
    	setIdListAndFinderInternal(idList, idListFinder);
        notifyChanged();
    }
    
    /**
     * Sets Strategy used to retrieve ids, sets page by ids pagination mode and marks list for reload.
     */
    public void setIdListFinder(IdListFinder<M,K> idListFinder) {
    	setIdListFinderInternal(idListFinder);
        notifyChanged();
    }
    
	public void setItemDeepCopier(UnaryOperator<T> itemDeepCopier) {
		this.itemDeepCopier = itemDeepCopier;
	}

    /**
     * Sets Strategy for retrieving single item of list, including details.
     */
    public void setItemFinder(SingleItemFinder<K,T> value) {
        itemFinder = value;
    }

    /**
     * Sets Strategy for sorting list data, overriding the list order of a list finder, or null to preserve list
     * order.  
     */
    public void setItemSorter(ListCacheRecordComparator<T> itemSorter) {
        this.itemSorter = itemSorter;
    }
    
    /**
     * Sets Strategy that extracts the primary key from a record.
     */
	public void setKeyExtractor(KeyExtractor<K,T> keyExtractor) {
		this.keyExtractor = keyExtractor;
	}

    /**
     * Caches list already loaded, sets full list Strategy used to load it, and sets full list pagination mode.
     */
    public void setListAndFinder(List<T> list, ListFinder<M,T> listFinder) {
    	setListAndFinderInternal(list, listFinder);
        notifyChanged();
    }
    
    /**
     * Sets full list Strategy to retrieve list when needed and sets full list pagination mode.
     */
    public void setListFinder(ListFinder<M,T> listFinder) {
    	setListFinderInternal(listFinder);
        notifyChanged();
    }
    
    /**
     * Sets list size, the list size finder used to find that, page by index range finder that will be used to find 
     * pages from list, and sets page by index pagination mode.
     * 
     * @param listSize Size of list to cache.
     * @param listSizeFinder Command that found list size.
     * @param pageByIndexRangeFinder Command that will load pages of list.
     */
    public void setListSizeAndFinder(int listSize, ListSizeFinder<M> listSizeFinder, 
            PageByIndexRangeFinder<M,T> pageByIndexRangeFinder) {
    	setListSizeAndFinderInternal(listSize, listSizeFinder, pageByIndexRangeFinder);
        notifyChanged();
    }
    
    /**
     * Sets the list size finder, sets accompanying page by index range finder that will be used to find pages from list,  
     * sets page by index pagination mode, and marks list for reload.
     * 
     * @param listSizeFinder Command that found list size.
     * @param pageByIndexRangeFinder Command that will load pages of list..
     */
    public void setListSizeFinder(ListSizeFinder<M> listSizeFinder, PageByIndexRangeFinder<M,T> pageByIndexRangeFinder) {
    	setListSizeFinderInternal(listSizeFinder, pageByIndexRangeFinder);
        notifyChanged();
    }

    /**
     * Sets Strategy used to retrieve page by their record ids. 
     */
    public void setPageByIdsFinder(PageByIdsFinder<K,T> value) {
        pageByIdsFinder = value;
    }
    
    /**
     * Sets Transfer Object Assembler for adding record extensions needed for requested page that is not loaded 
     * by a base record list finder.  Only applies for base record list pagination mode.
     */
    public void setPageExtensionAssembler(PageExtensionAssembler<T> pageExtensionAssembler) {
        this.pageExtensionAssembler = pageExtensionAssembler;
    }

    /**
     * Sets index of currently selected item to be first item of page where pageNo starts at one.
     */
    public void setPageNo(int pageNo) {
        setSelectedIndexInternal((pageNo - 1) * pageSize);
    	notifyChanged();
    }

    public void setPageSize(int pageSize) {
        this.pageSize = pageSize;
    }

    /**
     * Replaces selected item in list. 
     */
    public void setSelected(T value) {
    	setSelectedInternal(value);
        notifyChanged();
    }

    /**
     * Sets currently selected item in list by index.  Sets to last item if index exceeds last item.  Sets no currently 
     * selected item if index is -1.
     */
    public void setSelectedIndex(int index) {
        setSelectedIndexInternal(index);
        notifyChanged();
    }


}
