package name.matthewgreet.strutscommons.util;

import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import io.github.classgraph.ClassGraph;
import io.github.classgraph.ClassInfo;
import io.github.classgraph.ClassInfoList;
import io.github.classgraph.ClassRefTypeSignature;
import io.github.classgraph.ClassTypeSignature;
import io.github.classgraph.ScanResult;
import io.github.classgraph.TypeArgument;
import io.github.classgraph.TypeParameter;
import name.matthewgreet.strutscommons.annotation.CustomAdjuster;
import name.matthewgreet.strutscommons.annotation.CustomConversion;
import name.matthewgreet.strutscommons.annotation.CustomValidation;
import name.matthewgreet.strutscommons.exception.ConverterNotDefaultException;
import name.matthewgreet.strutscommons.exception.CustomPolicyNotApplicableException;
import name.matthewgreet.strutscommons.exception.NotPolicyImplementationException;
import name.matthewgreet.strutscommons.exception.PolicyLookupRejectionException;
import name.matthewgreet.strutscommons.policy.AbstractAdjusterSupport;
import name.matthewgreet.strutscommons.policy.AbstractCollectionConverterSupport;
import name.matthewgreet.strutscommons.policy.AbstractCollectionPostConversionAdjusterSupport;
import name.matthewgreet.strutscommons.policy.AbstractCollectionPostConversionValidatorSupport;
import name.matthewgreet.strutscommons.policy.AbstractConverterSupport;
import name.matthewgreet.strutscommons.policy.AbstractNonConversionValidatorSupport;
import name.matthewgreet.strutscommons.policy.AbstractPostConversionAdjusterSupport;
import name.matthewgreet.strutscommons.policy.AbstractPostConversionValidatorSupport;
import name.matthewgreet.strutscommons.policy.Adjuster;
import name.matthewgreet.strutscommons.policy.CollectionConverter;
import name.matthewgreet.strutscommons.policy.CollectionPostConversionAdjuster;
import name.matthewgreet.strutscommons.policy.CollectionPostConversionValidator;
import name.matthewgreet.strutscommons.policy.Converter;
import name.matthewgreet.strutscommons.policy.DefaultPolicy;
import name.matthewgreet.strutscommons.policy.NonConversionValidator;
import name.matthewgreet.strutscommons.policy.Policy;
import name.matthewgreet.strutscommons.policy.PostConversionAdjuster;
import name.matthewgreet.strutscommons.policy.PostConversionValidator;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.AnnotationUsage;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.DefaultCollectionConverterEntry;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.DefaultConverterEntry;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.PolicyEntry;

/**
 * <P>Standard implementation of {@link PolicyLookup} that creates a Singleton instance from an initial configuration  
 *    that returns the built-in form field policies and, if classpath scanning is enabled, automatically finds and adds 
 *    bespoke ones.  Once created, policies can be added and removed but scanning cannot be re-run.  Uses 
 *    <A HREF="https://github.com/classgraph/classgraph">ClassGraph</A> for classpath scanning.  If scanning 
 *    enabled, scanning the entire classpath can impose a noticeable, three second delay, so it's recommended to set the 
 *    known policy packages in configuration.</P>   
 */
public class DefaultPolicyLookup implements PolicyLookup {
	public static class Configuration implements Serializable {
		private static final long serialVersionUID = 7574407573480814754L;
		
		private String acceptClasses = "";
		private String acceptPackages = "";
		private boolean classpathScanningReplaceBuiltIn = false;
		private boolean enableClasspathScanning = false;
		private String rejectClasses = "";
		private String rejectPackages = "";

		
		/**
		 * Returns comma separated class names to search in classpath scanning, in addition to packages to scan, or 
		 * empty string for none.  Ignored if enableClasspathScanning not set.  Defaults to empty string.
		 */
		public String getAcceptClasses() {
			return acceptClasses;
		}
		public void setAcceptClasses(String acceptClasses) {
			this.acceptClasses = acceptClasses;
		}
		
		/**
		 * Returns comma separated package names to search (including subpackages), in classpath scanning, in addition 
		 * to classes to scan, or empty string for all packages.  Ignored if enableClasspathScanning not set.  Defaults 
		 * to empty string.
		 */
		public String getAcceptPackages() {
			return acceptPackages;
		}
		public void setAcceptPackages(String acceptPackages) {
			this.acceptPackages = acceptPackages;
		}
		
		/**
		 * Returns whether bespoke default converters and collection converters can replace built-in ones when found 
		 * with classpath scanning.  Ignored if enableClasspathScanning not set.  
		 */
		public boolean getClasspathScanningReplaceBuiltIn() {
			return classpathScanningReplaceBuiltIn;
		}
		public void setClasspathScanningReplaceBuiltIn(boolean classpathScanningReplaceBuiltIn) {
			this.classpathScanningReplaceBuiltIn = classpathScanningReplaceBuiltIn;
		}
		
		/**
	     * Returns whether to enable classpath scanning for bespoke policies on creation.  Defaults to false.   
		 */
		public boolean getEnableClasspathScanning() {
			return enableClasspathScanning;
		}
		public void setEnableClasspathScanning(boolean enableClasspathScanning) {
			this.enableClasspathScanning = enableClasspathScanning;
		}
		
		/**
		 * Returns comma separated class names to exclude from classpath scanning, or empty string for none.  Defaults 
		 * to empty string.  Ignored if enableClasspathScanning not set.
		 */
		public String getRejectClasses() {
			return rejectClasses;
		}
		public void setRejectClasses(String rejectClasses) {
			this.rejectClasses = rejectClasses;
		}
		
		/**
		 * Returns comma separated package names to exclude from classpath scanning, or empty string for none.  Defaults 
		 * to empty string.  Ignored if enableClasspathScanning not set.
		 */
		public String getRejectPackages() {
			return rejectPackages;
		}
		public void setRejectPackages(String rejectPackages) {
			this.rejectPackages = rejectPackages;
		}
		
	}
	
	public static class GenericTypeDeclarationClassGraph {
		private List<TypeParameter> genericsDeclaration;
		private List<TypeArgument> superclassGenericsDeclaration;
		
		
		public List<TypeParameter> getGenericsDeclaration() {
			return genericsDeclaration;
		}
		public void setGenericsDeclaration(List<TypeParameter> genericsDeclaration) {
			this.genericsDeclaration = genericsDeclaration;
		}
		
		public List<TypeArgument> getSuperclassGenericsDeclaration() {
			return superclassGenericsDeclaration;
		}
		public void setSuperclassGenericsDeclaration(List<TypeArgument> superclassGenericsDeclaration) {
			this.superclassGenericsDeclaration = superclassGenericsDeclaration;
		}
		
		
		@Override
		public String toString() {
			return "GenericTypeDeclarationClassGraph [genericsDeclaration=" + genericsDeclaration
					+ ", superclassGenericsDeclaration=" + superclassGenericsDeclaration + "]";
		}
	}
	
	public static class GenericTypeDeclarationReflection {
		private Type[] genericsDeclaration;
		private Type[] superclassGenericsDeclaration;
		
		
		public Type[] getGenericsDeclaration() {
			return genericsDeclaration;
		}
		public void setGenericsDeclaration(Type[] types) {
			this.genericsDeclaration = types;
		}

		public Type[] getSuperclassGenericsDeclaration() {
			return superclassGenericsDeclaration;
		}
		public void setSuperclassGenericsDeclaration(Type[] types) {
			this.superclassGenericsDeclaration = types;
		}
		
		
		@Override
		public String toString() {
			return "GenericTypeDeclarationReflection [genericsDeclaration=" + genericsDeclaration
					+ ", superclassGenericsDeclaration=" + superclassGenericsDeclaration + "]";
		}
	}
	
    private static DefaultPolicyLookup instance;
    
    public static final String BASE_POLICY_PACKAGES = "name.matthewgreet.strutscommons.policy";
    
    /**
     * Removes Singleton instance.  This only exists for testing.
     */
	public synchronized static void deleteInstance() {
		instance = null;
	}
	
	/**
	 * Returns Singleton instance with default initialisation options, or existing one if already initialised. 
	 */
	public synchronized static DefaultPolicyLookup getInstance() {
		if (instance == null) {
			instance = new DefaultPolicyLookup(new Configuration());
		}
		return instance;
	}
	
	/**
	 * Returns Singleton instance with initialisation options.  Initialisation options have no effect if the instance is
	 * already initialised.
	 */
	public synchronized static DefaultPolicyLookup getInstance(Configuration configuration) {
		if (instance == null) {
			instance = new DefaultPolicyLookup(configuration);
		}
		return instance;
	}
	
	@SuppressWarnings("unchecked")
	public static <A extends Annotation,P extends Policy<A>,T> PolicyEntry<?,?,?> makePolicyEntryFromClassNames(String annotationClassName,
			String policyClassName, String recipientTypeName, AnnotationUsage annotationUsage, boolean builtIn, boolean defaultPolicy) throws ClassNotFoundException {
		Class<A> annotationClass;
		Class<P> policyClass;
		Class<T> recipientTypeClass;
		
		annotationClass = (Class<A>)Class.forName(annotationClassName);
		policyClass = (Class<P>)Class.forName(policyClassName);
		if (recipientTypeName != null) {
			recipientTypeClass = (Class<T>)Class.forName(recipientTypeName);
		} else {
			recipientTypeClass = null;
		}
		return new PolicyEntry<>(annotationClass, policyClass, recipientTypeClass, annotationUsage, builtIn, defaultPolicy);
	}
	
	private Logger LOG = LogManager.getLogger(DefaultPolicyLookup.class);

	private Collection<String> acceptClasses;
	private Collection<String> acceptPackages;
	private boolean classpathScanningReplaceBuiltIn;
	private boolean enableClasspathScanning;
	private Collection<String> rejectClasses;
	private Collection<String> rejectPackages;

    protected Map<Class<? extends Annotation>,PolicyEntry<?,?,?>> builtInPolicyMap, dynamicPolicyMap;
    protected Map<Class<?>,DefaultConverterEntry<?,?>> builtInDefaultConverterMap, dynamicDefaultConverterMap;
    protected Map<Class<?>,DefaultCollectionConverterEntry<?,?>> builtInDefaultCollectionConverterMap, dynamicDefaultCollectionConverterMap;
    
    protected Set<Class<?>> builtInPolicyClasses, builtInDefaultConverterClasses, builtInDefaultCollectionConverterClasses;
    
	protected DefaultPolicyLookup(Configuration configuration) {
		super();
		if (configuration.getAcceptClasses() != null && configuration.getAcceptClasses().length() > 0) {
			acceptClasses = Collections.unmodifiableCollection(Arrays.asList(configuration.getAcceptClasses().split(",")));
		} else {
			acceptClasses = Collections.emptyList();
		}
		if (configuration.getAcceptPackages() != null && configuration.getAcceptPackages().length() > 0) {
			acceptPackages = Collections.unmodifiableCollection(Arrays.asList(configuration.getAcceptPackages().split(",")));
		} else {
			acceptPackages = Collections.emptyList();
		}
		classpathScanningReplaceBuiltIn = configuration.getClasspathScanningReplaceBuiltIn();
		enableClasspathScanning = configuration.getEnableClasspathScanning();
		if (configuration.getRejectClasses() != null && configuration.getRejectClasses().length() > 0) {
			rejectClasses = Collections.unmodifiableCollection(Arrays.asList(configuration.getRejectClasses().split(",")));
		} else {
			rejectClasses = Collections.emptyList();
		}
		if (configuration.getRejectPackages() != null && configuration.getRejectPackages().length() > 0) {
			rejectPackages = Collections.unmodifiableCollection(Arrays.asList(configuration.getRejectPackages().split(",")));
		} else {
			rejectPackages = Collections.emptyList();
		}
		initBuiltInPolicyMap();
		initBuiltInDefaultConverterMap();
		initBuiltInDefaultCollectionConverterMap();
		if (enableClasspathScanning) {
			initDynamicPolicyMap();
			initDynamicDefaultConverterMap();
			initDynamicDefaultCollectionConverterMap();
		} else {
			dynamicPolicyMap = new HashMap<>();
			dynamicDefaultConverterMap = new HashMap<>();
			dynamicDefaultCollectionConverterMap = new HashMap<>();
		}
	}
	
	/**
	 * Returns actual type declared by class implementing a parameterised interface for a specific generic type.  
	 * genericTypeIndex starts at 0.
	 */
	protected Class<?> actualTypeFromClassForInterface(Class<?> candidateClass, Class<?> interfaceClass, int genericTypeIndex) {
    	GenericTypeDeclarationReflection genericTypeDeclaration;
    	ParameterizedType workingSuperType;
    	List<GenericTypeDeclarationReflection> genericTypeDeclarations;
    	Type[] directInterfaceTypes, genericTypes;
		Class<?> result, workingClass;
		Type genericType, typeArgument;
		String workingGenericName;
		int workingTypeIndex, declarationIndex, j;
		boolean genericTypeFound, interfaceFound;
		
		interfaceFound = false;
    	result = null;
    	genericTypeDeclarations = new ArrayList<>();
    	workingTypeIndex = -1;
    	genericType = null;
    	// Traces superclasses till expected interface found
    	workingClass = candidateClass;
		while (workingClass != null && !interfaceFound) {
			if (workingClass.getGenericSuperclass() instanceof ParameterizedType) {
				workingSuperType = (ParameterizedType)workingClass.getGenericSuperclass();
			} else {
				// Is top level class
				workingSuperType = null;
			}
			genericTypeDeclaration = new GenericTypeDeclarationReflection();
			genericTypeDeclaration.setGenericsDeclaration(workingClass.getTypeParameters());
			genericTypeDeclaration.setSuperclassGenericsDeclaration((workingSuperType != null)?workingSuperType.getActualTypeArguments():null);
			genericTypeDeclarations.add(genericTypeDeclaration);
			
	    	directInterfaceTypes = workingClass.getGenericInterfaces();
			for (Type directInterfaceType: directInterfaceTypes) {
	    		if (directInterfaceType instanceof ParameterizedType && ((ParameterizedType)directInterfaceType).getRawType().equals(interfaceClass)) {
	    			genericTypes = ((ParameterizedType)directInterfaceType).getActualTypeArguments();
	    			workingGenericName = genericTypes[genericTypeIndex].getTypeName();
	    			interfaceFound = true;
	    			for (int i = 0; i < genericTypes.length; i++) {
	    				genericType = genericTypes[i];
	    				if (workingGenericName.equals(genericType.getTypeName())) {
	    					workingTypeIndex = genericTypeIndex;
	    					break;
	    				}
	    			}
	    			if (workingTypeIndex < 0) {
	    				// Doesn't match any generic parameter, so name is actual class
			        	return (Class<?>)genericType;
	    			} else if (interfaceFound) {
	    				// Sets interface actual arguments of first to be read entry
	    				genericTypeDeclaration.setSuperclassGenericsDeclaration(genericTypes);
	    				break;
	    			}
	    		}
			}
			workingClass = workingClass.getSuperclass();
		}
		if (interfaceFound && genericTypeDeclarations.size() > 0) {
			// Starts with required generic type of support class and traces type declarations of subclasses
			workingTypeIndex = genericTypeIndex;
			for (declarationIndex = genericTypeDeclarations.size() - 1; declarationIndex >= 0; declarationIndex--) {
				typeArgument = genericTypeDeclarations.get(declarationIndex).getSuperclassGenericsDeclaration()[workingTypeIndex];
				if (typeArgument == null) {
					break;
				}
				workingGenericName = typeArgument.getTypeName();
				j = 0;
				genericTypeFound = false;
				for (Type genericTypeDeclaration2:  genericTypeDeclarations.get(declarationIndex).getGenericsDeclaration()) {
					if (genericTypeDeclaration2.getTypeName().equals(workingGenericName)) {
						workingTypeIndex = j;
						genericTypeFound = true;
						break;
					}
					j++;
				}
				if (!genericTypeFound) {
					break;
				}
			}
			if (declarationIndex >= 0) { // If -1, class is generic and does not define annotation or recipient type
				result = (Class<?>)genericTypeDeclarations.get(declarationIndex).getSuperclassGenericsDeclaration()[workingTypeIndex];
			}
		}
		return result;
	}
	
	/**
	 * Returns name of actual type declared by class implementing a parameterised interface for a specific generic type, 
	 * according to ClassGraph into.  genericTypeIndex starts at 0.
	 */
	protected String actualTypeFromClassInfoForInterface(ClassInfo candidateClassInfo, Class<?> interfaceClass, int genericTypeIndex) {
    	GenericTypeDeclarationClassGraph genericTypeDeclaration;
    	ClassInfo workingClassInfo;
    	ClassRefTypeSignature workingSignature;
    	ClassTypeSignature workingTypeSignature;
    	TypeArgument typeArgument;
    	TypeParameter typeParameter;
    	List<ClassRefTypeSignature> directInterfaceSignatures;
    	List<GenericTypeDeclarationClassGraph> genericTypeDeclarations;
    	List<TypeArgument> typeArguments;
		String result, workingGenericName;
		int workingTypeIndex, declarationIndex, j;
		boolean interfaceFound, genericTypeFound;
		
		interfaceFound = false;
		result = null;
    	genericTypeDeclarations = new ArrayList<>();
    	workingTypeIndex = -1;
    	// Traces superclasses till expected interface found
		workingClassInfo = candidateClassInfo;
		while (workingClassInfo != null && !interfaceFound) {
			workingSignature = workingClassInfo.getTypeSignature().getSuperclassSignature();
			genericTypeDeclaration = new GenericTypeDeclarationClassGraph();
			genericTypeDeclaration.setGenericsDeclaration(workingClassInfo.getTypeSignature().getTypeParameters());
			genericTypeDeclaration.setSuperclassGenericsDeclaration(workingSignature.getTypeArguments());
			genericTypeDeclarations.add(genericTypeDeclaration);
			
			workingTypeSignature = workingClassInfo.getTypeSignature();
			directInterfaceSignatures = workingTypeSignature.getSuperinterfaceSignatures();
			for (ClassRefTypeSignature directInterfaceSignature: directInterfaceSignatures) {
	    		if (directInterfaceSignature.getBaseClassName().equals(interfaceClass.getName())) {
			        typeArguments = directInterfaceSignature.getTypeArguments();
	    			workingGenericName = typeArguments.get(genericTypeIndex).getTypeSignature().toString();
	    			interfaceFound = true;
	    			for (int i = 0; i < genericTypeDeclaration.getGenericsDeclaration().size(); i++) {
	    				typeParameter = genericTypeDeclaration.getGenericsDeclaration().get(i);
	    				if (workingGenericName.equals(typeParameter.getName())) {
	    					workingTypeIndex = genericTypeIndex;
	    					break;
	    				}
	    			}
	    			if (workingTypeIndex < 0) {
	    				// Doesn't match any generic parameter, so name is actual classname
			        	return workingGenericName;
	    			} else if (interfaceFound) {
	    				// Sets interface actual arguments of first to be read entry
	    				genericTypeDeclaration.setSuperclassGenericsDeclaration(typeArguments);
	    				break;
	    			}
	    		}
			}
			workingClassInfo = workingClassInfo.getSuperclass();
		}
		
		if (interfaceFound && genericTypeDeclarations.size() > 0) {
			// Starts with required generic type of declaring class and traces type declarations of subclasses
			for (declarationIndex = genericTypeDeclarations.size() - 1; declarationIndex >= 0; declarationIndex--) {
				typeArgument = genericTypeDeclarations.get(declarationIndex).getSuperclassGenericsDeclaration().get(workingTypeIndex);
				if (typeArgument.getTypeSignature() == null) {
					break;
				}
				workingGenericName = typeArgument.getTypeSignature().toString();
				j = 0;
				genericTypeFound = false;
				for (TypeParameter genericTypeDeclaration2:  genericTypeDeclarations.get(declarationIndex).getGenericsDeclaration()) {
					if (genericTypeDeclaration2.getName().equals(workingGenericName)) {
						workingTypeIndex = j;
						genericTypeFound = true;
						break;
					}
					j++;
				}
				if (!genericTypeFound) {
					break;
				}
			}
			if (declarationIndex >= 0) { // If -1, class is generic and does not define annotation or recipient type
				result = genericTypeDeclarations.get(declarationIndex).getSuperclassGenericsDeclaration().get(workingTypeIndex).toString();
			}
		}
		return result;
	}
	
	protected <A extends Annotation, P extends Policy<A>,T> void addDynamicPolicyInternalAutomatic(PolicyEntry<A,P,T> policyEntry) {
		PolicyEntry<?,?,?> builtInPolicyEntry, existingPolicyEntry;
		
		if (InterceptorCommonLibrary.isCustomAnnotationClass(policyEntry.getAnnotationClass())) {
			return;
		}
		builtInPolicyEntry = builtInPolicyMap.get(policyEntry.getAnnotationClass());
		if (builtInPolicyEntry == null) {
			existingPolicyEntry = dynamicPolicyMap.get(policyEntry.getAnnotationClass());
			if (existingPolicyEntry == null) {
				dynamicPolicyMap.put(policyEntry.getAnnotationClass(), policyEntry);
			} else if (policyEntry.getPolicyClass() != existingPolicyEntry.getPolicyClass()) {
			    LOG.error("Classpath scanning found bespoke policies with clashing annotation" + 
			        "  first policy class=" + policyEntry.getPolicyClass() + "  second policy class=" + existingPolicyEntry.getPolicyClass() + 
					"  annotation=" + policyEntry.getAnnotationClass());
			}
		} else if (!builtInPolicyClasses.contains(policyEntry.getPolicyClass())) {
		    LOG.error("Classpath scanning found bespoke policy using built-in annotation" + "  policy class=" + policyEntry.getPolicyClass() + 
				    "  built-in annotation=" + policyEntry.getAnnotationClass());
		} 
	}
	
	@SuppressWarnings("unchecked")
	protected <A extends Annotation, P extends Policy<A>,T> PolicyEntry<A,?,?> addDynamicPolicyInternalManual(PolicyEntry<A,P,T> policyEntry) throws PolicyLookupRejectionException {
		PolicyEntry<A,?,?> result;
		String message;
		
		if (InterceptorCommonLibrary.isCustomAnnotationClass(policyEntry.getAnnotationClass())) {
		    message = "Adding validation policy rejected as it's custom and doesn't need to be added" + 
		        "  policy class=" + policyEntry.getPolicyClass() + "  built-in annotation=" + policyEntry.getAnnotationClass();
		    LOG.error(message);
			throw new CustomPolicyNotApplicableException(policyEntry.getPolicyClass());
		}
		
		result = (PolicyEntry<A,?,?>)dynamicPolicyMap.put(policyEntry.getAnnotationClass(), policyEntry);
		if (result == null) {
			result = (PolicyEntry<A,?,?>)builtInPolicyMap.get(policyEntry.getAnnotationClass());
		}
		return result;
	}
	
	protected boolean defaultPolicyFromClass(Class<?> candidateClass) {
		return DefaultPolicy.class.isAssignableFrom(candidateClass);
	}

	protected boolean defaultPolicyFromClassInfo(ClassInfo candidateClassInfo) {
		return candidateClassInfo.implementsInterface(DefaultPolicy.class);
	}
	
	protected List<PolicyEntry<?,?,?>> getScannedPoliciesByType(ScanResult scanResult, Class<?> interfaceClass, Class<?> supportClass, 
			AnnotationUsage annotationUsage) {
		PolicyEntry<?,?,?> policyEntry;
    	ClassInfo interfaceInfo;
    	ClassInfoList candidateInfoList;
    	List<ClassInfo> candidateClassInfos;
    	List<PolicyEntry<?,?,?>> result;
    	String annotationName, recipientTypeName, message;
    	boolean recipientTypeNeeded, defaultPolicy;
    	
		interfaceInfo = scanResult.getClassInfo(interfaceClass.getName());
		candidateInfoList = interfaceInfo.getClassesImplementing();
		candidateClassInfos = new ArrayList<>();
		result = new ArrayList<>();
        for (ClassInfo candidateInfo: candidateInfoList) {
        	if (!candidateInfo.isAbstract()) {
        		candidateClassInfos.add(candidateInfo);
        	}
        }
        for (ClassInfo candidateClassInfo: candidateClassInfos) {
        	recipientTypeName = null;
        	recipientTypeNeeded = annotationUsage.getDirectInterfaceRecipientTypeIndex() != null;
        	if (recipientTypeNeeded) {	// If policy is a converter, collection converter or post conversion
        		recipientTypeName = actualTypeFromClassInfoForInterface(candidateClassInfo, interfaceClass, annotationUsage.getDirectInterfaceRecipientTypeIndex());
        	}
        	defaultPolicy = defaultPolicyFromClassInfo(candidateClassInfo);
        	annotationName = null;
    		annotationName = actualTypeFromClassInfoForInterface(candidateClassInfo, interfaceClass, annotationUsage.getDirectInterfaceAnnotationIndex());
    		if (annotationName != null && (!recipientTypeNeeded || recipientTypeName != null)) {
    			try {
			        policyEntry = makePolicyEntryFromClassNames(annotationName, candidateClassInfo.getName(), recipientTypeName, annotationUsage, false, defaultPolicy);
			        result.add(policyEntry);
				}
    			catch (ClassNotFoundException e) {
    				LOG.error("getScannedPoliciesByType: " + annotationName + " or " + candidateClassInfo.getName() + 
    				    " found by class scanner but cannot be loaded");
				}
    		} else {
    			// To get here, candidate is subclass of implementation of interface but not of standard support class.
    			// Possible to trace recipient type but I can't be bothered to work it out.  
        		message = "Policy cannot be added as its recipient type cannot be found; it must directly implement " + 
        				annotationUsage.getDirectInterface() + " or inherit from " + annotationUsage.getSupportClass() + 
        				"  policy=" + candidateClassInfo;
        		LOG.error(message);
    		}
        }
        return result;
	}
    
	/**
     * Returns unmodifiable map of all built-in default converters.
     */
	protected void initBuiltInDefaultCollectionConverterMap() {
		List<DefaultCollectionConverterEntry<?,?>> converterEntries;
    	Map<Class<?>,DefaultCollectionConverterEntry<?,?>> working;
    	Set<Class<?>> converterClasses;
    	
        if (builtInDefaultCollectionConverterMap == null) {
        	working = new HashMap<>();
        	converterClasses = new HashSet<>();
        	converterEntries = InterceptorCommonLibrary.getBuiltInDefaultCollectionConverters();
        	for (DefaultCollectionConverterEntry<?,?> converterEntry: converterEntries) {
            	working.put(converterEntry.getItemClass(), converterEntry);
            	converterClasses.add(converterEntry.getCollectionConverterClass());
        	}
        	builtInDefaultCollectionConverterMap = Collections.unmodifiableMap(working);
        	builtInDefaultCollectionConverterClasses = Collections.unmodifiableSet(converterClasses);
        }
	}
    
	/**
     * Returns unmodifiable map of all built-in default converters.
     */
	protected void initBuiltInDefaultConverterMap() {
		List<DefaultConverterEntry<?,?>> converterEntries;
    	Map<Class<?>,DefaultConverterEntry<?,?>> working;
    	Set<Class<?>> converterClasses;
    	
        if (builtInDefaultConverterMap == null) {
        	working = new HashMap<>();
        	converterClasses = new HashSet<>();
        	converterEntries = InterceptorCommonLibrary.getBuiltInDefaultConverters();
        	for (DefaultConverterEntry<?,?> converterEntry: converterEntries) {
            	working.put(converterEntry.getFieldClass(), converterEntry);
            	converterClasses.add(converterEntry.getConverterClass());
        	}
        	builtInDefaultConverterMap = Collections.unmodifiableMap(working);
        	builtInDefaultConverterClasses = Collections.unmodifiableSet(converterClasses);
        }
	}

    /**
     * Returns unmodifiable map of all built-in adjusters, converters, and validators. 
     */
	protected void initBuiltInPolicyMap() {
		List<PolicyEntry<?,?,?>> policyEntries;
    	Map<Class<? extends Annotation>,PolicyEntry<?,?,?>> working;
    	Set<Class<?>> policyClasses;
    	
        if (builtInPolicyMap == null) {
        	working = new HashMap<>();
        	policyClasses = new HashSet<>();
        	policyEntries = InterceptorCommonLibrary.getBuiltInPolicies();
        	for (PolicyEntry<?,?,?> policyEntry: policyEntries) {
            	working.put(policyEntry.getAnnotationClass(), policyEntry);
            	policyClasses.add(policyEntry.getPolicyClass());
        	}
        	builtInPolicyMap = Collections.unmodifiableMap(working);
        	builtInPolicyClasses = Collections.unmodifiableSet(policyClasses);
        }
	}
	
    @SuppressWarnings({"rawtypes", "unchecked"})
	protected void initDynamicDefaultCollectionConverterMap() {
    	DefaultCollectionConverterEntry<?,?> defaultCollectionConverterEntry;
    	
    	dynamicDefaultCollectionConverterMap = new HashMap<>();
    	for (PolicyEntry<?,?,?> policyEntry: dynamicPolicyMap.values()) {
    		if (policyEntry.getAnnotationUsage() == AnnotationUsage.COLLECTION_CONVERT && policyEntry.getDefaultPolicy()) {
    			if (!classpathScanningReplaceBuiltIn && builtInDefaultCollectionConverterMap.containsKey(policyEntry.getRecipientType())) {
    				continue;
    			}
    			defaultCollectionConverterEntry = new DefaultCollectionConverterEntry(policyEntry.getRecipientType(), policyEntry.getPolicyClass(), false);
    			dynamicDefaultCollectionConverterMap.put(policyEntry.getRecipientType(), defaultCollectionConverterEntry);
    		}
    	}
	}
	
    @SuppressWarnings({"rawtypes", "unchecked"})
	protected void initDynamicDefaultConverterMap() {
    	DefaultConverterEntry<?,?> defaultConverterEntry;
    	
    	dynamicDefaultConverterMap = new HashMap<>();
    	for (PolicyEntry<?,?,?> policyEntry: dynamicPolicyMap.values()) {
    		if (policyEntry.getAnnotationUsage() == AnnotationUsage.CONVERT && policyEntry.getDefaultPolicy()) {
    			if (!classpathScanningReplaceBuiltIn && builtInDefaultConverterMap.containsKey(policyEntry.getRecipientType())) {
    				continue;
    			}
    			defaultConverterEntry = new DefaultConverterEntry(policyEntry.getRecipientType(), policyEntry.getPolicyClass(), false);
    			dynamicDefaultConverterMap.put(policyEntry.getRecipientType(), defaultConverterEntry);
    		}
    	}
	}
	
	protected void initDynamicPolicyMap() {
		ClassGraph classGraph;
    	List<PolicyEntry<?,?,?>> policyEntries;
    	Collection<String> workingAcceptPackages;

		dynamicPolicyMap = new HashMap<>();
		
		workingAcceptPackages = new ArrayList<>();
		if (acceptPackages.size() > 0 || acceptClasses.size() > 0) {
			// Classgraph needs to find policy interfaces and support classes to find classes implementing them
			workingAcceptPackages.add(BASE_POLICY_PACKAGES);
			workingAcceptPackages.addAll(acceptPackages);
		}
		
		classGraph = new ClassGraph();
		classGraph.enableClassInfo();
		classGraph.acceptClasses(acceptClasses.toArray(new String[0]));
		classGraph.acceptPackages(workingAcceptPackages.toArray(new String[0]));
		classGraph.rejectClasses(rejectClasses.toArray(new String[0]));
		classGraph.rejectPackages(rejectPackages.toArray(new String[0]));
		try (ScanResult scanResult = classGraph.scan()) {
			policyEntries = getScannedPoliciesByType(scanResult, Adjuster.class, AbstractAdjusterSupport.class, AnnotationUsage.ADJUSTER);
			for (PolicyEntry<?,?,?> policyEntry: policyEntries) {
				addDynamicPolicyInternalAutomatic(policyEntry);
			}
			policyEntries = getScannedPoliciesByType(scanResult, CollectionConverter.class, AbstractCollectionConverterSupport.class, AnnotationUsage.COLLECTION_CONVERT);
			for (PolicyEntry<?,?,?> policyEntry: policyEntries) {
				addDynamicPolicyInternalAutomatic(policyEntry);
			}
			policyEntries = getScannedPoliciesByType(scanResult, CollectionPostConversionAdjuster.class, AbstractCollectionPostConversionAdjusterSupport.class, AnnotationUsage.COLLECTION_POST_ADJUSTER);
			for (PolicyEntry<?,?,?> policyEntry: policyEntries) {
				addDynamicPolicyInternalAutomatic(policyEntry);
			}
			policyEntries = getScannedPoliciesByType(scanResult, CollectionPostConversionValidator.class, AbstractCollectionPostConversionValidatorSupport.class, AnnotationUsage.COLLECTION_POST_VALIDATION);
			for (PolicyEntry<?,?,?> policyEntry: policyEntries) {
				addDynamicPolicyInternalAutomatic(policyEntry);
			}
			policyEntries = getScannedPoliciesByType(scanResult, Converter.class, AbstractConverterSupport.class, AnnotationUsage.CONVERT);
			for (PolicyEntry<?,?,?> policyEntry: policyEntries) {
				addDynamicPolicyInternalAutomatic(policyEntry);
			}
			policyEntries = getScannedPoliciesByType(scanResult, NonConversionValidator.class, AbstractNonConversionValidatorSupport.class, AnnotationUsage.NON_CONVERT_VALIDATION);
			for (PolicyEntry<?,?,?> policyEntry: policyEntries) {
				addDynamicPolicyInternalAutomatic(policyEntry);
			}
			policyEntries = getScannedPoliciesByType(scanResult, PostConversionAdjuster.class, AbstractPostConversionAdjusterSupport.class, AnnotationUsage.POST_ADJUSTER);
			for (PolicyEntry<?,?,?> policyEntry: policyEntries) {
				addDynamicPolicyInternalAutomatic(policyEntry);
			}
			policyEntries = getScannedPoliciesByType(scanResult, PostConversionValidator.class, AbstractPostConversionValidatorSupport.class, AnnotationUsage.POST_VALIDATION);
			for (PolicyEntry<?,?,?> policyEntry: policyEntries) {
				addDynamicPolicyInternalAutomatic(policyEntry);
			}
	    }
	}

	/**
	 * Translates form field class or item type of collection form field into a class used by default converter and 
	 * collection converter lookups. 
	 */
	protected Class<?> translateDefaultConverterRecipientClass(Class<?> recipientClass) {
		if (Enum.class.isAssignableFrom(recipientClass)) {
			return Enum.class;
		} else {
			return InterceptorCommonLibrary.getConverterClassLookupFromFormFieldClass(recipientClass);
		}
	}

	public Collection<String> getAcceptClasses() {
		return acceptClasses;
	}

	public Collection<String> getAcceptPackages() {
		return acceptPackages;
	}

	/**
	 * Returns whether instance was created allowing classpath scanning, if run, to replace built-in default converters 
	 * and collection converters with bespoke one. 
	 */
	public boolean getClasspathScanningReplaceBuiltIn() {
		return classpathScanningReplaceBuiltIn;
	}

	@Override
	public Collection<DefaultCollectionConverterEntry<?,?>> getDefaultCollectionConverterEntries() {
		Collection<DefaultCollectionConverterEntry<?,?>> result;
		Set<Class<?>> dynamicClassesInUse;
		
		dynamicClassesInUse = dynamicDefaultCollectionConverterMap.keySet();
		result = new ArrayList<>();
		result.addAll(dynamicDefaultCollectionConverterMap.values());
		for (DefaultCollectionConverterEntry<?,?> builtInEntry: builtInDefaultCollectionConverterMap.values()) {
			if (!dynamicClassesInUse.contains(builtInEntry.getItemClass())) {
				result.add(builtInEntry);
			}
		}
		return result;
	}

	/**
	 * Returns default collection converter configuration for the collection item's type, or null if none set.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <C extends CollectionConverter<?,T>,T> DefaultCollectionConverterEntry<T,C> getDefaultCollectionConverterEntry(Class<? extends T> itemClass) {
		DefaultCollectionConverterEntry<T,C> result;
		Class<? extends T> workingItemClass;
		
		workingItemClass = (Class<? extends T>)translateDefaultConverterRecipientClass(itemClass);
		result = (DefaultCollectionConverterEntry<T,C>)dynamicDefaultCollectionConverterMap.get(workingItemClass);
		if (result == null) {
			result = (DefaultCollectionConverterEntry<T,C>)builtInDefaultCollectionConverterMap.get(workingItemClass);
		}
		return result;
	}

	@Override
	public Collection<DefaultConverterEntry<?,?>> getDefaultConverterEntries() {
		Collection<DefaultConverterEntry<?,?>> result;
		Set<Class<?>> dynamicClassesInUse;
		
		dynamicClassesInUse = dynamicDefaultConverterMap.keySet();
		result = new ArrayList<>();
		result.addAll(dynamicDefaultConverterMap.values());
		for (DefaultConverterEntry<?,?> builtInEntry: builtInDefaultConverterMap.values()) {
			if (!dynamicClassesInUse.contains(builtInEntry.getFieldClass())) {
				result.add(builtInEntry);
			}
		}
		return result;
	}

	/**
	 * Returns default converter configuration for the form field type, or null if none set.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <C extends Converter<?,T>,T> DefaultConverterEntry<T,C> getDefaultConverterEntry(Class<? extends T> fieldClass) {
		DefaultConverterEntry<T,C> result;
		Class<? extends T> workingFieldClass;
		
		workingFieldClass = (Class<? extends T>)translateDefaultConverterRecipientClass(fieldClass);
		result = (DefaultConverterEntry<T,C>)dynamicDefaultConverterMap.get(workingFieldClass);
		if (result == null) {
			result = (DefaultConverterEntry<T,C>)builtInDefaultConverterMap.get(workingFieldClass);
		}
		return result;
	}

	/**
	 * Returns whether instance was created with classpath scanning. 
	 */
	public boolean getEnableClasspathScanning() {
		return enableClasspathScanning;
	}

	@Override
	public Collection<PolicyEntry<?,?,?>> getPolicyEntries() {
		Collection<PolicyEntry<?,?,?>> result;
		
		result = new ArrayList<>();
		result.addAll(builtInPolicyMap.values());
		result.addAll(dynamicPolicyMap.values());
		return result;
	}
	
	/**
	 * Returns policy implementation configuration for a form field annotation, or null if not set.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public <A extends Annotation, P extends Policy<A>,T> PolicyEntry<A,P,T> getPolicyEntry(Class<? extends A> annotationClass) {
		PolicyEntry<A,P,T> result;
		
		result = (PolicyEntry<A,P,T>)dynamicPolicyMap.get(annotationClass);
		if (result == null) {
			result = (PolicyEntry<A,P,T>)builtInPolicyMap.get(annotationClass);
		}
		return result;
	}
	
	public Collection<String> getRejectClasses() {
		return rejectClasses;
	}

	public Collection<String> getRejectPackages() {
		return rejectPackages;
	}
	
	/**
	 * Sets form field collection converter as the default collection converter for its recipient item type, replacing 
	 * any existing one.  Class must directly implement {@link Converter} or inherit {@link AbstractConverterSupport}.  
	 * Cannot replace a built-in collection converter.
	 */
	@SuppressWarnings("unchecked")
	public <C extends CollectionConverter<?,T>,T> DefaultCollectionConverterEntry<T,?> putDefaultCollectionConverter(Class<C> collectionConverterClass) 
			throws PolicyLookupRejectionException {
    	DefaultCollectionConverterEntry<T,C> defaultCollectionConverterEntry;
    	DefaultCollectionConverterEntry<T,?> result;
		Class<? extends Annotation> annotationClass;
    	Class<T> recipientItemType;
    	String message;
    	
		// Sanity checks
		if (collectionConverterClass.isInterface() || Modifier.isAbstract(collectionConverterClass.getModifiers())) {
			throw new NotPolicyImplementationException(collectionConverterClass);
		}
		
		annotationClass = null;
		annotationClass = (Class<? extends Annotation>)actualTypeFromClassForInterface(collectionConverterClass, AnnotationUsage.COLLECTION_CONVERT.getDirectInterface(), AnnotationUsage.CONVERT.getDirectInterfaceAnnotationIndex());
		if (InterceptorCommonLibrary.isCustomAnnotationClass(annotationClass)) {
			throw new CustomPolicyNotApplicableException(collectionConverterClass);
		}
    	if (!DefaultPolicy.class.isAssignableFrom(collectionConverterClass)) {
    		throw new ConverterNotDefaultException(collectionConverterClass);
    	}
    	
    	recipientItemType = null;
		recipientItemType = (Class<T>)actualTypeFromClassForInterface(collectionConverterClass, AnnotationUsage.COLLECTION_CONVERT.getDirectInterface(),
    			AnnotationUsage.COLLECTION_CONVERT.getDirectInterfaceRecipientTypeIndex());
    	if (recipientItemType != null) {
			defaultCollectionConverterEntry = new DefaultCollectionConverterEntry<T,C>(recipientItemType, collectionConverterClass, false);
			result = (DefaultCollectionConverterEntry<T,?>)dynamicDefaultCollectionConverterMap.put(recipientItemType, defaultCollectionConverterEntry);
			if (result == null) {
				result = (DefaultCollectionConverterEntry<T,?>)builtInDefaultCollectionConverterMap.get(recipientItemType);
			}
			return result;
    	} else {
			// To get here, collection converter is subclass of implementation of interface.  Possible to trace recipient type but I can't be bothered to work it out.  
    		message = "Collection converter class cannot be added as default collection converter as it does not directly implement " + 
        	        AnnotationUsage.COLLECTION_CONVERT.getDirectInterface() + " nor inherit from " + AnnotationUsage.COLLECTION_CONVERT.getSupportClass();
    		LOG.error(message);
    		throw new PolicyLookupRejectionException(message, collectionConverterClass);
    	}
	}
	
	/**
	 * Sets form field converter as the default converter for its recipient type, replacing any existing converter.
	 * Class must directly implement {@link Converter} or inherit {@link AbstractConverterSupport}.  Ignores any attempt to replace a built-in or custom converter. 
	 */
	@SuppressWarnings("unchecked")
	public <C extends Converter<?,T>,T> DefaultConverterEntry<T,?> putDefaultConverter(Class<C> converterClass) 
			throws PolicyLookupRejectionException {
    	DefaultConverterEntry<T,C> defaultConverterEntry;
    	DefaultConverterEntry<T,?> result;
		Class<? extends Annotation> annotationClass;
    	Class<T> recipientType;
    	String message;

		// Sanity checks
		if (converterClass.isInterface() || Modifier.isAbstract(converterClass.getModifiers())) {
			throw new NotPolicyImplementationException(converterClass);
		}
		
		annotationClass = null;
		annotationClass = (Class<? extends Annotation>)actualTypeFromClassForInterface(converterClass, AnnotationUsage.CONVERT.getDirectInterface(), AnnotationUsage.CONVERT.getDirectInterfaceAnnotationIndex());
		if (InterceptorCommonLibrary.isCustomAnnotationClass(annotationClass)) {
			throw new CustomPolicyNotApplicableException(converterClass);
		}
    	if (!DefaultPolicy.class.isAssignableFrom(converterClass)) {
    		throw new ConverterNotDefaultException(converterClass);
    	}
    	
    	recipientType = null;
		recipientType = (Class<T>)actualTypeFromClassForInterface(converterClass, AnnotationUsage.CONVERT.getDirectInterface(),
    			AnnotationUsage.CONVERT.getDirectInterfaceRecipientTypeIndex());
    	if (recipientType != null) {
			defaultConverterEntry = new DefaultConverterEntry<T,C>(recipientType, converterClass, false);
			result = (DefaultConverterEntry<T,?>)dynamicDefaultConverterMap.put(recipientType, defaultConverterEntry);
			if (result == null) {
				result = (DefaultConverterEntry<T,?>)builtInDefaultConverterMap.get(recipientType);
			}
			return result;
    	} else {
			// To get here, converter is subclass of implementation of interface.  Possible to trace recipient type but I can't be bothered to work it out.  
    		message = "Converter class cannot be added as default converter as it does not directly implement " + 
        	        AnnotationUsage.CONVERT.getDirectInterface() + " nor inherit from " + AnnotationUsage.CONVERT.getSupportClass();
    		LOG.error(message);
    		throw new PolicyLookupRejectionException(message, converterClass);
    	}
	}
	
	/**
	 * Adds a form field policy not found by classpath scanning, so it can be found from the annotation that configures 
	 * it, replacing any existing using the same annotation.  Ignores polices referring to built-in annotations or 
	 * custom annotations ({@link CustomAdjuster}, {@link CustomConversion}, {@link CustomValidation} etc.).  
	 */
	@SuppressWarnings("unchecked")
	public <A extends Annotation, P extends Policy<A>,T> PolicyEntry<A,?,?> putPolicy(Class<P> policyClass) throws PolicyLookupRejectionException {
		AnnotationUsage annotationUsage;
		PolicyEntry<A,P,T> policyEntry;
		PolicyEntry<A,?,?> result;
		Class<?> directInterfaceClass;
		Class<A> annotationClass;
		Class<T> recipientType;
		boolean defaultPolicy;

		// Sanity checks
		if (policyClass.isInterface() || Modifier.isAbstract(policyClass.getModifiers())) {
			throw new NotPolicyImplementationException(policyClass);
		}
		
		result = null;
		annotationUsage = InterceptorCommonLibrary.getAnnotationUsageFromPolicyClass(policyClass);
		directInterfaceClass = annotationUsage.getDirectInterface();
		if (directInterfaceClass != null) {
			annotationClass = null;
			annotationClass = (Class<A>)actualTypeFromClassForInterface(policyClass, directInterfaceClass, annotationUsage.getDirectInterfaceAnnotationIndex());
			if (annotationClass != null && InterceptorCommonLibrary.isCustomAnnotationClass(annotationClass)) {
				throw new CustomPolicyNotApplicableException(policyClass);
			}
			defaultPolicy = defaultPolicyFromClass(policyClass);
			recipientType = null;
			if (annotationUsage.getDirectInterfaceRecipientTypeIndex() != null) {	// If policy is a converter, collection converter or post conversion
        		recipientType = (Class<T>)actualTypeFromClassForInterface(policyClass, directInterfaceClass, annotationUsage.getDirectInterfaceRecipientTypeIndex());
			}
			if (annotationClass != null) {
				policyEntry = new PolicyEntry<>(annotationClass, policyClass, recipientType, annotationUsage, false, defaultPolicy);
				result = addDynamicPolicyInternalManual(policyEntry);
			}
		} else {
			throw new NotPolicyImplementationException(policyClass);
		}
		return result;
	}

	/**
	 * Removes default collection converter for an item type for collection form fields and returns it, or null if not 
	 * found.  Cannot remove a built-in collection converter.  
	 */
	@SuppressWarnings("unchecked")
	public <C extends CollectionConverter<?,T>,T> DefaultCollectionConverterEntry<T,C> removeDefaultCollectionConverter(Class<? extends T> itemClass) throws IllegalArgumentException {
    	DefaultCollectionConverterEntry<T,C> defaultCollectionConverterEntry;
    	String message;
    	
    	defaultCollectionConverterEntry = (DefaultCollectionConverterEntry<T,C>)builtInDefaultCollectionConverterMap.get(itemClass);
		if (defaultCollectionConverterEntry != null) {
			message = "Built-in converters cannot be removed as default converters  itemClass=" + itemClass;
			LOG.error(message);
			throw new IllegalArgumentException(message);
		}
		
		defaultCollectionConverterEntry = (DefaultCollectionConverterEntry<T,C>)dynamicDefaultCollectionConverterMap.remove(itemClass);
		return defaultCollectionConverterEntry;
	}

	/**
	 * Removes default converter for a form field type and returns it, or null if not found.  Cannot remove a built-in 
	 * converter.  
	 */
	@SuppressWarnings("unchecked")
	public <C extends Converter<?,T>,T> DefaultConverterEntry<T,C> removeDefaultConverter(Class<? extends T> fieldClass) throws IllegalArgumentException {
    	DefaultConverterEntry<T,C> defaultConverterEntry;
    	String message;
    	
    	defaultConverterEntry = (DefaultConverterEntry<T,C>)builtInDefaultConverterMap.get(fieldClass);
		if (defaultConverterEntry != null) {
			message = "Built-in converters cannot be removed as default converters  fieldClass=" + fieldClass;
			LOG.error(message);
			throw new IllegalArgumentException(message);
		}
		
		defaultConverterEntry = (DefaultConverterEntry<T,C>)dynamicDefaultConverterMap.remove(fieldClass);
		return defaultConverterEntry;
	}

	/**
	 * Removes form field policy found from the class of annotation that configures it and returns policy config, or 
	 * null if not found.  Cannot remove for built-in annotations.  
	 */
	@SuppressWarnings("unchecked")
	public <A extends Annotation, P extends Policy<A>,T> PolicyEntry<A,P,T> removePolicy(Class<? extends A> annotationClass) throws IllegalArgumentException {
		PolicyEntry<A,P,T> policyEntry;
		String message;
		
		policyEntry = (PolicyEntry<A,P,T>)builtInPolicyMap.get(annotationClass);
		if (policyEntry != null) {
			message = "Built-in policies cannot be removed  annotation=" + annotationClass;
			LOG.error(message);
			throw new IllegalArgumentException(message);
		}
		
		policyEntry = (PolicyEntry<A,P,T>)dynamicPolicyMap.remove(annotationClass);
		return policyEntry;
	}

}
