package name.matthewgreet.strutscommons.util;

import java.lang.annotation.Annotation;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import name.matthewgreet.strutscommons.annotation.DisableFormatting;
import name.matthewgreet.strutscommons.annotation.IntegerConversion;
import name.matthewgreet.strutscommons.policy.AbstractCustomCollectionFormatterSupport;
import name.matthewgreet.strutscommons.policy.AbstractCustomFormatterSupport;
import name.matthewgreet.strutscommons.policy.CollectionConverter;
import name.matthewgreet.strutscommons.policy.Converter;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.AnnotationEntries;

/**
     * <P>Default library for formatting a record to a view helper in Struts 2.  Sets string, string collection or 
     * string array display fields from record fields with the same name but formatted according to the conversion 
     * annotations, such as {@link IntegerConversion}, on <CODE>display</CODE> fields, or default converter if no 
     * annotation.</P>
     * 
     * <P>For each display field, the exact behaviour depends on it, the matching record field, and conversion 
     * annotation.  Combinations not shown here aren't allowed.</P>
     * <TABLE CLASS="main">
     *   <CAPTION>Display formatting modes</CAPTION>
     *   <TR>
     *     <TH STYLE="text-align: left; width: 140px;">Record field type</TH>
     *     <TH STYLE="text-align: left; width: 140px;">Display field type</TH>
     *     <TH STYLE="text-align: left; width: 100px;">Converter annotation</TH>
     *     <TH STYLE="text-align: left;">Notes</TH>
     *   </TR>
     *   <TR CLASS="row_odd">
     *     <TD>Single value</TD>
     *     <TD>Single value string</TD>
     *     <TD>Converter or none</TD>
     *     <TD>If no converter set, uses default single value converter</TD>
     *   </TR>
     *   <TR CLASS="row_even">
     *     <TD>Collection</TD>
     *     <TD>Single value string</TD>
     *     <TD>Collection converter or none</TD>
     *     <TD>If no collection converter set, uses default collection converter</TD>
     *   </TR>
     *   <TR CLASS="row_odd">
     *     <TD>Array</TD>
     *     <TD>String array</TD>
     *     <TD>Converter or none</TD>
     *     <TD>If no converter set, uses default converter</TD>
     *   </TR>
     *   <TR CLASS="row_even">
     *     <TD>Array</TD>
     *     <TD>String collection</TD>
     *     <TD>Converter or none</TD>
     *     <TD>If no converter set, uses default converter</TD>
     *   </TR>
     *   <TR CLASS="row_odd">
     *     <TD>Collection</TD>
     *     <TD>String array</TD>
     *     <TD>Converter or none</TD>
     *     <TD>If no converter set, uses default converter</TD>
     *   </TR>
     *   <TR CLASS="row_even">
     *     <TD>Collection</TD>
     *     <TD>String collection</TD>
     *     <TD>Converter or none</TD>
     *     <TD>If no converter set, uses default converter</TD>
     *   </TR>
     * </TABLE>
     * 
     * <P>Miscellaneous notes.</P>
     * <UL>
     *   <LI>Source field type must apply to converter's type.</LI>
     *   <LI>Custom converters only for formatting are best derived from {@link AbstractCustomFormatterSupport} or 
     *       {@link AbstractCustomCollectionFormatterSupport}.</LI>
     *   <LI>The default converters for booleans and enumerations don't format to user friendly text, so custom 
     *       converters are better.</LI>
     *   <LI>Does not follow associations.</LI>
     * </UL> 
 */
public class DefaultDisplayFormatter implements DisplayFormatter {
	public static class DisplayFieldSource {
    	private Field displayField;
    	private Field recordField;
    	
    	
		public Field getDisplayField() {
			return displayField;
		}
		public Field getRecordField() {
			return recordField;
		}
		public void setDisplayField(Field displayField) {
			this.displayField = displayField;
		}
		public void setRecordField(Field recordField) {
			this.recordField = recordField;
		}
    }
    
    public static class DisplayFieldUsage<T> {
    	private AnnotationEntries<T> annountationEntries;
    	private ConverterCategory converterCategory;
    	private Class<T> fieldType;
    	private FieldCategory recordFieldCategory;
    	private Field recordField;
    	private String recordName;
    	private FieldCategory displayFieldCategory;
    	private Field displayField;
    	private String displayName;
    	
    	
		public AnnotationEntries<T> getAnnountationEntries() {
			return annountationEntries;
		}
		public void setAnnountationEntries(AnnotationEntries<T> annountationEntries) {
			this.annountationEntries = annountationEntries;
		}
		
		public Class<T> getFieldType() {
			return fieldType;
		}
		public void setFieldType(Class<T> fieldType) {
			this.fieldType = fieldType;
		}

		public ConverterCategory getConverterCategory() {
			return converterCategory;
		}
		public void setConverterCategory(ConverterCategory converterCategory) {
			this.converterCategory = converterCategory;
		}

		public FieldCategory getRecordFieldCategory() {
			return recordFieldCategory;
		}
		public void setRecordFieldCategory(FieldCategory recordFieldCategory) {
			this.recordFieldCategory = recordFieldCategory;
		}
		
		public Field getRecordField() {
			return recordField;
		}
		public void setRecordField(Field recordField) {
			this.recordField = recordField;
		}

		public String getRecordName() {
			return recordName;
		}
		public void setRecordName(String recordName) {
			this.recordName = recordName;
		}


		public FieldCategory getDisplayFieldCategory() {
			return displayFieldCategory;
		}
		public void setDisplayFieldCategory(FieldCategory displayFieldCategory) {
			this.displayFieldCategory = displayFieldCategory;
		}
		
		public Field getDisplayField() {
			return displayField;
		}
		public void setDisplayField(Field displayField) {
			this.displayField = displayField;
		}

		public String getDisplayName() {
			return displayName;
		}
		public void setDisplayName(String displayName) {
			this.displayName = displayName;
		}

    }
    
    /**
     * Whether a converter annotation is explicitly set and whether it's single value or collection.
     */
    public enum ConverterCategory {COLLECTION, DEFAULT, SINGLE}
    
    /**
	 *  Modes of how an unformatted, record field is converted to a formatted, display field, each with their own 
	 *  processing function.  See {@link #getDisplayFormatMode}.
	 */
    public enum DisplayFormatMode {SINGLE_TO_SINGLE, SINGLE_TO_ARRAY, SINGLE_TO_COLLECTION, ARRAY_TO_SINGLE,	
    	ARRAY_TO_ARRAY, ARRAY_TO_COLLECTION, COLLECTION_TO_SINGLE, COLLECTION_TO_ARRAY, COLLECTION_TO_COLLECTION, 
    	OTHER1, OTHER2, OTHER3, OTHER4, OTHER5, OTHER6, OTHER7, OTHER8, OTHER9, OTHER10, UNSUPPORTED}
    

    /**
     * Whether a record or display field is a single value, array, or collection.
     */
    public enum FieldCategory {ARRAY, COLLECTION, SINGLE}
    
    private Logger LOG = LogManager.getLogger(DefaultDisplayFormatter.class);
    
    private PolicyLookup policyLookup = null;

    
    public DefaultDisplayFormatter() {
		super();
	}

	/**
     * Returns whether a converter's target data type can convert to the form field's data type. 
     */
	protected <T> boolean checkFieldClass(Class<?> conversionClass, Class<?> fieldClass) {
		return InterceptorCommonLibrary.checkFieldClass(conversionClass, fieldClass);
	}

	/**
	 * Returns whether display field is allowed to receive a formatted value, which is not annotated with 
	 * {@link DisableFormatting}.  
	 */
	protected boolean filterDisplayField(Field displayField) {
		boolean disableFound;
		
		disableFound = false;
        for (Annotation annotation: displayField.getAnnotations()) {
            if (annotation instanceof DisableFormatting) {
            	disableFound = true;
            	break;
            }
        }
		return !disableFound;
	}

    /**
     * Initialisation that needs to be done before formatting function is run.
     */
    protected void formatInit() {
        // Not set at construction so client can set different PolicyLookup and avoid DefaultPolicyLookup being created.
        if (policyLookup == null) {
        	policyLookup = DefaultPolicyLookup.getInstance();
        }
    }

	/**
	 * Returns all recognised annotations and their linked policies.
	 */
    protected <T> AnnotationEntries<T> getAnnotationEntries(Field displayField, PolicyLookup policyLookup) {
		return InterceptorCommonLibrary.getAnnotationEntries(displayField, policyLookup);
	}
    
	/**
	 * Returns default collection converter for a collection display field, or null if none available. 
	 */
    @SuppressWarnings("unchecked")
	protected <T> CollectionConverter<?,T> getDefaultCollectionConverter(DisplayFieldUsage<T> displayFieldUsage) {
		return (CollectionConverter<?,T>)InterceptorCommonLibrary.getDefaultCollectionConverter(displayFieldUsage.getRecordField(), policyLookup);
	}
    
	/**
	 * Returns default converter for a single value display field, or null if none available. 
	 */
    @SuppressWarnings("unchecked")
	protected <T> Converter<?,T> getDefaultConverter(DisplayFieldUsage<T> displayFieldUsage) {
		return (Converter<?,T>)InterceptorCommonLibrary.getDefaultConverter(displayFieldUsage.getRecordField(), policyLookup);
	}
	
	/**
	 * Returns default converter for each item of an array display field, or null if none available. 
	 */
    @SuppressWarnings("unchecked")
	protected <T> Converter<?,T> getDefaultConverterFromArray(DisplayFieldUsage<T> displayFieldUsage) {
		return (Converter<?,T>)InterceptorCommonLibrary.getDefaultConverterFromArray(displayFieldUsage.getRecordField(), policyLookup);
	}
    
	/**
	 * Returns default converter for each item of a collection display field, or null if none available. 
	 */
    @SuppressWarnings("unchecked")
	protected <T> Converter<?,T> getDefaultConverterFromCollection(DisplayFieldUsage<T> displayFieldUsage) {
		return (Converter<?,T>)InterceptorCommonLibrary.getDefaultConverterFromCollection(displayFieldUsage.getRecordField(), policyLookup);
	}
    
	/**
     * Returns every display field that will receive a formatted value and the record field containing the value to be 
     * formatted. 
     */
    protected Collection<DisplayFieldSource> getDisplayFieldSources(Collection<Field> displayFields, Collection<Field> recordFields) {
    	DisplayFieldSource displayFieldSource;
    	Field recordField;
		Map<String,Field> recordFieldLookup;
		Collection<DisplayFieldSource> result;
		
        recordFieldLookup = recordFields.stream().collect(Collectors.toMap(f -> f.getName(), f -> f));
        result = new ArrayList<>();
        for (Field displayField: displayFields) {
        	recordField = recordFieldLookup.get(displayField.getName());
        	if (recordField != null) {
        		displayFieldSource = new DisplayFieldSource();
        		displayFieldSource.setDisplayField(displayField);
        		displayFieldSource.setRecordField(recordField);
        		result.add(displayFieldSource);
        	}
        }
        return result;
    }
	
	/**
     * Analyses converter annotations on display field and returns configured formatter to formatted from source field 
     * display field, or null if not applicable.
     */
	@SuppressWarnings("unchecked")
	protected <T> DisplayFieldUsage<T> getDisplayFieldUsage(Field recordField, Field displayField, PolicyLookup policyLookup) {
   	    AnnotationEntries<T> annotationEntries;
	    CollectionConverter<?,T> collectionConverter;
    	Converter<?,T> converter;
    	ConverterCategory converterCategory;
    	DisplayFieldUsage<T> result;
    	FieldCategory displayFieldCategory, recordFieldCategory; 
    	Class<?> displayType;
    	Class<T> recordType;
    	boolean compatible;
        
    	converterCategory = ConverterCategory.DEFAULT;
    	annotationEntries = getAnnotationEntries(displayField, policyLookup);
    	if (annotationEntries.getCollectionConverter() != null) {
    		collectionConverter = annotationEntries.getCollectionConverter().getCollectionConverter();
    		converterCategory = ConverterCategory.COLLECTION;
    	} else {
    		collectionConverter = null;
    	}
    	if (annotationEntries.getConverter() != null) {
    		converter = annotationEntries.getConverter().getConverter();
    		converterCategory = ConverterCategory.SINGLE;
    	} else {
    		converter = null;
    	}
    	
    	if (displayField.getType().isArray()) {
    		displayFieldCategory = FieldCategory.ARRAY;
    		displayType = displayField.getType().getComponentType();
    	} else if (Collection.class.isAssignableFrom(displayField.getType())) {
    		displayFieldCategory = FieldCategory.COLLECTION;
    		displayType = getTypeFromCollectionField(displayField);
    	} else {
    		displayFieldCategory = FieldCategory.SINGLE;
    		displayType = displayField.getType();
    	}
    	if (!String.class.isAssignableFrom(displayType)) {
    		if (converterCategory == ConverterCategory.SINGLE || converterCategory == ConverterCategory.COLLECTION) {
                LOG.error("Annotated display fields must be a string, string array or string collection" + 
    		        "  display=" + displayField.getDeclaringClass().getName() + 
                    "  field=" + displayField.getName());
    		}
    		return null;
    	}
    	if (recordField.getType().isArray()) {
    		recordFieldCategory = FieldCategory.ARRAY;
    		recordType = (Class<T>)recordField.getType().getComponentType();
    	} else if (Collection.class.isAssignableFrom(recordField.getType())) {
    		recordFieldCategory = FieldCategory.COLLECTION;
    		recordType = getTypeFromCollectionField(recordField);
    	} else {
    		recordFieldCategory = FieldCategory.SINGLE;
    		recordType = (Class<T>)recordField.getType();
    	}
    	
    	switch (converterCategory) {
		case COLLECTION:
	    	compatible = checkFieldClass(collectionConverter.getRecipientClass(), recordType);
    	    if (!compatible) {
                LOG.error("Annotation on display field does not match type of record field  display=" + displayField.getDeclaringClass().getName() + 
                    "  field=" + displayField.getName() + "  annotation=" + collectionConverter.getAnnotation().annotationType() + 
                  	"  record=" + recordField.getDeclaringClass() + "  field=" + recordField.getName());
                return null;
    	    }
			break;
		case DEFAULT:
			// Empty
			break;
		case SINGLE:
    	    compatible = checkFieldClass(converter.getRecipientClass(), recordType);
    	    if (!compatible) {
                LOG.error("Annotation on display field does not match type of record field  display=" + displayField.getDeclaringClass().getName() + 
                    "  field=" + displayField.getName() + "  annotation=" + converter.getAnnotation().annotationType() + 
                 	"  record=" + recordField.getDeclaringClass() + "  field=" + recordField.getName());
                return null;
    	    }
			break;
    	}
    	
    	result = new DisplayFieldUsage<T>();
    	result.setAnnountationEntries(annotationEntries);
    	result.setConverterCategory(converterCategory);
    	result.setDisplayFieldCategory(displayFieldCategory);
    	result.setDisplayField(displayField);
    	result.setDisplayName(displayField.getName());
    	result.setFieldType(recordType);
    	result.setRecordFieldCategory(recordFieldCategory);
    	result.setRecordField(recordField);
    	result.setRecordName(recordField.getName());
    	return result;
	}

	/**
     * Returns mode for formatting a record field to the matching display field.  The following combinations are 
     * recognised, the rest unsupported.   
     * <TABLE>
     *   <CAPTION>Recognised record and display field types and converters</CAPTION>
     *   <TR>
     *     <TH>Record field</TH>
     *     <TH>Display field</TH>
     *     <TH>Converters</TH>
     *     <TH>Mode</TH>
     *   </TR>
     *   <TR>
     *     <TD>Array</TD>
     *     <TD>Array</TD>
     *     <TD>Single value or default</TD>
     *     <TD>ARRAY_TO_ARRAY</TD>
     *   </TR>
     *   <TR>
     *     <TD>Array</TD>
     *     <TD>Collection</TD>
     *     <TD>Single value or default</TD>
     *     <TD>ARRAY_TO_COLLECTION</TD>
     *   </TR>
     *   <TR>
     *     <TD>Collection</TD>
     *     <TD>Array</TD>
     *     <TD>Single value or default</TD>
     *     <TD>COLLECTION_TO_ARRAY</TD>
     *   </TR>
     *   <TR>
     *     <TD>Collection</TD>
     *     <TD>Collection</TD>
     *     <TD>Single value or default</TD>
     *     <TD>COLLECTION_TO_COLLECTION</TD>
     *   </TR>
     *   <TR>
     *     <TD>Collection</TD>
     *     <TD>Single value</TD>
     *     <TD>Collection or default</TD>
     *     <TD>COLLECTION_TO_SINGLE</TD>
     *   </TR>
     *   <TR>
     *     <TD>Single value</TD>
     *     <TD>Single value</TD>
     *     <TD>Single value or default</TD>
     *     <TD>SINGLE_TO_SINGLE</TD>
     *   </TR>
     * </TABLE>
     */
    protected DisplayFormatMode getDisplayFormatMode(DisplayFieldUsage<?> displayFieldUsage) {
    	switch (displayFieldUsage.getRecordFieldCategory()) {
		case ARRAY:
			switch (displayFieldUsage.getDisplayFieldCategory()) {
			case ARRAY:
				switch (displayFieldUsage.getConverterCategory()) {
				case COLLECTION:
		    		return DisplayFormatMode.UNSUPPORTED;
				case DEFAULT:
				case SINGLE:
		    		return DisplayFormatMode.ARRAY_TO_ARRAY;
				}
			case COLLECTION:
				switch (displayFieldUsage.getConverterCategory()) {
				case COLLECTION:
		    		return DisplayFormatMode.UNSUPPORTED;
				case DEFAULT:
				case SINGLE:
		    		return DisplayFormatMode.ARRAY_TO_COLLECTION;
				}
			case SINGLE:
	    		return DisplayFormatMode.UNSUPPORTED;
			}
			break;
		case COLLECTION:
			switch (displayFieldUsage.getDisplayFieldCategory()) {
			case ARRAY:
				switch (displayFieldUsage.getConverterCategory()) {
				case COLLECTION:
		    		return DisplayFormatMode.UNSUPPORTED;
				case DEFAULT:
				case SINGLE:
		    		return DisplayFormatMode.COLLECTION_TO_ARRAY;
				}
			case COLLECTION:
				switch (displayFieldUsage.getConverterCategory()) {
				case COLLECTION:
		    		return DisplayFormatMode.UNSUPPORTED;
				case DEFAULT:
				case SINGLE:
		    		return DisplayFormatMode.COLLECTION_TO_COLLECTION;
				}
			case SINGLE:
				switch (displayFieldUsage.getConverterCategory()) {
				case COLLECTION:
				case DEFAULT:
		    		return DisplayFormatMode.COLLECTION_TO_SINGLE;
				case SINGLE:
		    		return DisplayFormatMode.UNSUPPORTED;
				}
			}
			break;
		case SINGLE:
			switch (displayFieldUsage.getDisplayFieldCategory()) {
			case ARRAY:
			case COLLECTION:
	    		return DisplayFormatMode.UNSUPPORTED;
			case SINGLE:
				switch (displayFieldUsage.getConverterCategory()) {
				case COLLECTION:
		    		return DisplayFormatMode.UNSUPPORTED;
				case DEFAULT:
				case SINGLE:
		    		return DisplayFormatMode.SINGLE_TO_SINGLE;
				}
			}
			break;
    	}
		return DisplayFormatMode.UNSUPPORTED;
    }

	/**
     * Returns all properties of a class, even private ones, whether directly declared or inherited. 
     */
    protected Collection<Field> getProperties(Class<?> type) {
    	return InterceptorCommonLibrary.getProperties(type);
    }
	
    /**
     * Returns element type of a collection-based form field. 
     */
    protected <T> Class<T> getTypeFromCollectionField(Field displayField) {
		return InterceptorCommonLibrary.getTypeFromCollectionField(displayField);
	}

	/**
     * Returns an empty, modifiable collection from a collection type. 
     */
    protected <T> Collection<T> makeCollectionForRecipient(Class<?> recipientClass) {
    	return InterceptorCommonLibrary.makeCollectionForRecipient(recipientClass);
	}
	
	/**
	 * <P>Sets array display field from each member of an array record field formatted using converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string 
	 * collection, copies source values, otherwise sets empty array.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageArrayToArray(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		Collection<String> formattedValues;
		List<T> unformattedValues;
		String[] finalValue;
		Object rawValue;
		String formattedValue;
		int i, length;
		
	    unformattedValues = new ArrayList<>();
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			// Extracts value from array, whether primitives or objects
			rawValue = displayFieldUsage.getRecordField().get(record);
			if (rawValue != null) {
			    length = Array.getLength(rawValue);
			    for (i = 0; i < length; i++) {
			    	unformattedValues.add((T)Array.get(rawValue, i));
			    }
			}
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverterFromArray(displayFieldUsage);
		}
		formattedValues = new ArrayList<>();
		if (converter != null) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null || converter.getProcessNoValue()) {
					formattedValue = converter.format(unformattedValue);
					formattedValues.add(formattedValue);
				} else {
					formattedValues.add("");
				}
			}
		} else if (String.class.isAssignableFrom(displayFieldUsage.getFieldType())) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null) {
					formattedValue = (String)unformattedValue;
					formattedValues.add(formattedValue);
				} else {
					formattedValues.add("");
				}
			}
		}
		finalValue = new String[formattedValues.size()];
		i = 0;
		for (String formattedValue2: formattedValues) {
			finalValue[i] = formattedValue2;
			i++;
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * <P>Sets collection display field from each member of an array record field formatted using converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string 
	 * collection, copies source values, otherwise sets empty collection.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageArrayToCollection(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		Collection<String> finalValue;
		List<T> unformattedValues;
		Object rawValue;
		String formattedValue;
		int i, length;
		
	    unformattedValues = new ArrayList<>();
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			// Extracts value from array, whether primitives or objects
			rawValue = displayFieldUsage.getRecordField().get(record);
			if (rawValue != null) {
			    length = Array.getLength(rawValue);
			    for (i = 0; i < length; i++) {
			    	unformattedValues.add((T)Array.get(rawValue, i));
			    }
			}
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverterFromArray(displayFieldUsage);
		}
		finalValue = makeCollectionForRecipient(displayFieldUsage.getDisplayField().getType());
		if (converter != null) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null || converter.getProcessNoValue()) {
					formattedValue = converter.format(unformattedValue);
					finalValue.add(formattedValue);
				} else {
					finalValue.add("");
				}
			}
		} else if (String.class.isAssignableFrom(displayFieldUsage.getFieldType())) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null) {
					formattedValue = (String)unformattedValue;
					finalValue.add(formattedValue);
				} else {
					finalValue.add("");
				}
			}
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * <P>Sets single display field from first member of an array record field formatted using converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string 
	 * collection, copies source values, otherwise sets empty string.</P>
	 * 
	 * <P>This is not actually used, not considered useful, and exists for the sake of completeness.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageArrayToSingle(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		List<T> unformattedValues;
		T unformattedValue;
		Object rawValue;
		String finalValue;
		int i, length;
		
	    unformattedValues = new ArrayList<>();
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			// Extracts value from array, whether primitives or objects
			rawValue = displayFieldUsage.getRecordField().get(record);
			if (rawValue != null) {
			    length = Array.getLength(rawValue);
			    for (i = 0; i < length; i++) {
			    	unformattedValues.add((T)Array.get(rawValue, i));
			    }
			}
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		finalValue = "";
		if (converter != null && unformattedValues.size() > 0) {
			unformattedValue = unformattedValues.get(0);
			if (unformattedValue != null || converter.getProcessNoValue()) {
				finalValue = converter.format(unformattedValue);
			}
		} else if (String.class.isAssignableFrom(displayFieldUsage.getFieldType())) {
			unformattedValue = unformattedValues.get(0);
			if (unformattedValue != null) {
				finalValue = (String)unformattedValue;
			}
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * <P>Sets array display field from each member of a collection record field formatted using converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string 
	 * collection, copies source values, otherwise sets empty array.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageCollectionToArray(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		Collection<T> unformattedValues;
		Collection<String> formattedValues;
		String[] finalValue;
		String formattedValue;
		int i;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValues = (Collection<T>)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValues = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverterFromCollection(displayFieldUsage);
		}
		formattedValues = new ArrayList<>();
		if (converter != null && unformattedValues != null) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null || converter.getProcessNoValue()) {
					formattedValue = converter.format(unformattedValue);
					formattedValues.add(formattedValue);
				} else {
					formattedValues.add("");
				}
			}
		} else if (unformattedValues != null && String.class.isAssignableFrom(displayFieldUsage.getFieldType())) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null) {
					formattedValue = (String)unformattedValue;
					formattedValues.add(formattedValue);
				} else {
					formattedValues.add("");
				}
			}
		}
		finalValue = new String[formattedValues.size()];
		i = 0;
		for (String formattedValue2: formattedValues) {
			finalValue[i] = formattedValue2;
			i++;
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * <P>Sets collection display field from each member of a collection record field formatted using converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string 
	 * collection, copies source values, otherwise sets empty collection.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageCollectionToCollection(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		Collection<T> unformattedValues;
		Collection<String> finalValue;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValues = (Collection<T>)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValues = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverterFromCollection(displayFieldUsage);
		}
		finalValue = makeCollectionForRecipient(displayFieldUsage.getDisplayField().getType());
		if (converter != null && unformattedValues != null) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null || converter.getProcessNoValue()) {
					formattedValue = converter.format(unformattedValue);
					finalValue.add(formattedValue);
				} else {
					finalValue.add("");
				}
			}
		} else if (unformattedValues != null && String.class.isAssignableFrom(displayFieldUsage.getFieldType())) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null) {
					formattedValue = (String)unformattedValue;
					finalValue.add(formattedValue);
				} else {
					finalValue.add("");
				}
			}
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * Sets single display field from collection record field formatted using collection converter.  If no collection 
	 * converter is set, uses default collection converter for source field.  If no default converter, sets empty 
	 * string.
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageCollectionToSingle(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		CollectionConverter<?,T> collectionconverter;
		Collection<T> unformattedValue;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValue = (Collection<T>)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValue = null;
		}
		collectionconverter = null;
		if (displayFieldUsage.getAnnountationEntries().getCollectionConverter() != null) {
			collectionconverter = displayFieldUsage.getAnnountationEntries().getCollectionConverter().getCollectionConverter();
		}
		if (collectionconverter == null) {
			collectionconverter = (CollectionConverter<?,T>)getDefaultCollectionConverter(displayFieldUsage);
		}
		if (collectionconverter != null) {
			if ((unformattedValue != null && unformattedValue.size() > 0) || collectionconverter.getProcessNoValue()) {
				formattedValue = collectionconverter.format(unformattedValue);
			} else {
				formattedValue = "";
			}
		} else {
			formattedValue = "";
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, formattedValue);
	}
	
	/**
	 * <P>Called for OTHER1 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode #getDisplayFormatMode}, to define OTHER1 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther1(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER10 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode #getDisplayFormatMode}, to define OTHER10 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther10(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER2 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode #getDisplayFormatMode}, to define OTHER2 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther2(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER3 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode #getDisplayFormatMode}, to define OTHER3 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther3(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER4 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode #getDisplayFormatMode}, to define OTHER4 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther4(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER5 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode #getDisplayFormatMode}, to define OTHER5 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther5(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER6 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode #getDisplayFormatMode}, to define OTHER6 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther6(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER7 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode #getDisplayFormatMode}, to define OTHER7 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther7(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER8 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode #getDisplayFormatMode}, to define OTHER8 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther8(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER9 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode #getDisplayFormatMode}, to define OTHER9 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther9(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Sets array display field from from single record field formatted using configured converter.  If no converter 
	 * is set, uses default converter for source field.  If no default converter and source is a string, copies source 
	 * value, otherwise uses empty string.  If the formatted value is not an empty string, sets an array of just that 
	 * value, otherwise sets an empty array.</P>
	 * 
	 * <P>This is not actually used, not considered useful and exists for the sake of completeness.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageSingleToArray(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		String[] finalValue;
		T unformattedValue;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValue = (T)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValue = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverterFromArray(displayFieldUsage);
		}
		if (converter != null) {
			if (unformattedValue != null || converter.getProcessNoValue()) {
				formattedValue = converter.format(unformattedValue);
			} else {
				formattedValue = "";
			}
		} else if (unformattedValue != null && String.class.isAssignableFrom(unformattedValue.getClass())) {
			formattedValue = (String)unformattedValue;
		} else {
			formattedValue = "";
		}
		if (formattedValue.length() > 0) {
			finalValue = new String[1];
			finalValue[0] = formattedValue;
		} else {
			finalValue = new String[0];
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * <P>Sets collection display field from from single record field formatted using configured converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string, 
	 * copies source value, otherwise uses empty string.  If the formatted value is not an empty string, sets a 
	 * collection of just that value, otherwise sets an empty collection.</P>
	 * 
	 * <P>This is not actually used, not considered useful and exists for the sake of completeness.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageSingleToCollection(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		Collection<String> finalValue;
		T unformattedValue;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValue = (T)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValue = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		if (converter != null) {
			if (unformattedValue != null || converter.getProcessNoValue()) {
				formattedValue = converter.format(unformattedValue);
			} else {
				formattedValue = "";
			}
		} else if (unformattedValue != null && String.class.isAssignableFrom(unformattedValue.getClass())) {
			formattedValue = (String)unformattedValue;
		} else {
			formattedValue = "";
		}
		finalValue = makeCollectionForRecipient(displayFieldUsage.getDisplayField().getType());
		if (formattedValue.length() > 0) {
			finalValue.add(formattedValue);
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * Sets single display field from from single record field formatted using configured converter.  If no converter is 
	 * set, uses default converter for source field.  If no default converter and source is a string, copies source 
	 * value, otherwise sets empty string.
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageSingleToSingle(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		T unformattedValue;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValue = (T)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValue = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		if (converter != null) {
			if (unformattedValue != null || converter.getProcessNoValue()) {
				formattedValue = converter.format(unformattedValue);
			} else {
				formattedValue = "";
			}
		} else if (unformattedValue != null && String.class.isAssignableFrom(displayFieldUsage.getFieldType())) {
			formattedValue = (String)unformattedValue;
		} else {
			formattedValue = "";
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, formattedValue);
	}

	/**
	 * <P>Called for UNSUPPORTED formatting mode and does nothing.  This exists so subclasses can override it.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageUnsupported(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * Returns policy lookup by annotation helper used in validation. 
	 */
	public PolicyLookup getPolicyLookup() {
		return policyLookup;
	}
	public void setPolicyLookup(PolicyLookup policyLookup) {
		this.policyLookup = policyLookup;
	}

	/**
     * <P>Sets string fields of <CODE>display</CODE> from fields of <CODE>record</CODE> with the same name but formatted 
     * according to the conversion annotations on <CODE>display</CODE> fields, or default converter if no annotation.  
     * If <CODE>record</CODE> is null, recipient <CODE>display</CODE> fields are set to empty string.  This function is 
     * useful for displaying database records.</P>
     * 
     * <P>Miscellaneous notes.</P>
     * <UL>
     *   <LI>Source field type must apply to converter's type.</LI>
     *   <LI>Recipient field must be a string, not even string array or string collection.</LI>
     *   <LI>Custom converters only for formatting are best derived from {@link AbstractCustomFormatterSupport} or 
     *       {@link AbstractCustomCollectionFormatterSupport}.</LI>
     *   <LI>The default converters for booleans and enumerations don't format to user friendly text, so custom 
     *       converters are better.</LI>
     *   <LI>Does not follow associations.</LI>
     * </UL> 
     */
	@Override
	public void updateDisplay(Class<?> recordClass, Object record, Object display) {
		DisplayFormatMode conversionMode;
		DisplayFieldUsage<?> displayFieldUsage;
        Field displayField, recordField;
		Collection<DisplayFieldSource> displayFieldSources;
        Collection<Field> recordFields, displayFields;
        Object unformattedValue;
        
        formatInit();
        recordFields = getProperties(recordClass);
        displayFields = getProperties(display.getClass());
        displayFields = displayFields.stream().filter(this::filterDisplayField).collect(Collectors.toList());
        displayFieldSources = getDisplayFieldSources(displayFields, recordFields);
        
        for (DisplayFieldSource displayFieldSource: displayFieldSources) {
        	displayField = displayFieldSource.getDisplayField();
        	recordField = displayFieldSource.getRecordField();
    		unformattedValue = null;
    		try {
				displayFieldUsage = getDisplayFieldUsage(recordField, displayField, policyLookup);
				if (displayFieldUsage != null) {
					if (record != null) {
						displayFieldUsage.getRecordField().setAccessible(true);
	        			unformattedValue = displayFieldUsage.getRecordField().get(record) ;
					} else {
						unformattedValue = "";
					}
					conversionMode = getDisplayFormatMode(displayFieldUsage);
		        	switch (conversionMode) {
					case ARRAY_TO_ARRAY:
	        			processDisplayFieldUsageArrayToArray(record, display, displayFieldUsage);
						break;
					case ARRAY_TO_COLLECTION:
	        			processDisplayFieldUsageArrayToCollection(record, display, displayFieldUsage);
						break;
					case ARRAY_TO_SINGLE:
	        			processDisplayFieldUsageArrayToSingle(record, display, displayFieldUsage);
						break;
					case COLLECTION_TO_ARRAY:
	        			processDisplayFieldUsageCollectionToArray(record, display, displayFieldUsage);
						break;
					case COLLECTION_TO_COLLECTION:
	        			processDisplayFieldUsageCollectionToCollection(record, display, displayFieldUsage);
						break;
					case COLLECTION_TO_SINGLE:
	        			processDisplayFieldUsageCollectionToSingle(record, display, displayFieldUsage);
						break;
					case OTHER1:
	        			processDisplayFieldUsageOther1(record, display, displayFieldUsage);
						break;
					case OTHER2:
	        			processDisplayFieldUsageOther2(record, display, displayFieldUsage);
						break;
					case OTHER3:
	        			processDisplayFieldUsageOther3(record, display, displayFieldUsage);
						break;
					case OTHER4:
	        			processDisplayFieldUsageOther4(record, display, displayFieldUsage);
						break;
					case OTHER5:
	        			processDisplayFieldUsageOther5(record, display, displayFieldUsage);
						break;
					case OTHER6:
	        			processDisplayFieldUsageOther6(record, display, displayFieldUsage);
						break;
					case OTHER7:
	        			processDisplayFieldUsageOther7(record, display, displayFieldUsage);
						break;
					case OTHER8:
	        			processDisplayFieldUsageOther8(record, display, displayFieldUsage);
						break;
					case OTHER9:
	        			processDisplayFieldUsageOther9(record, display, displayFieldUsage);
						break;
					case OTHER10:
	        			processDisplayFieldUsageOther10(record, display, displayFieldUsage);
						break;
					case SINGLE_TO_ARRAY:
	        			processDisplayFieldUsageSingleToArray(record, display, displayFieldUsage);
						break;
					case SINGLE_TO_COLLECTION:
	        			processDisplayFieldUsageSingleToCollection(record, display, displayFieldUsage);
						break;
					case SINGLE_TO_SINGLE:
	        			processDisplayFieldUsageSingleToSingle(record, display, displayFieldUsage);
						break;
					case UNSUPPORTED:
						break;
		        	}
				}
    		}
    		catch (IllegalArgumentException | IllegalAccessException e) {
                LOG.error("Security violation when accessing record or setting display   display=" + display.getClass() + 
                        "  field=" + displayField.getName() + "  record=" + record.getClass() + "  field=" + recordField.getName(), e);
			}
    		catch (Exception e) {
                LOG.error("Formatter failed   display=" + display.getClass() + "  field=" + displayField.getName() +
                	"  value=" + unformattedValue, e);
			}
        }
	}

}
