package name.matthewgreet.strutscommons.util;

import java.io.File;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.struts2.dispatcher.HttpParameters;
import org.apache.struts2.dispatcher.Parameter;
import org.apache.struts2.dispatcher.Parameter.Empty;
import org.apache.struts2.dispatcher.multipart.UploadedFile;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.ModelDriven;
import com.opensymphony.xwork2.TextProvider;
import com.opensymphony.xwork2.conversion.impl.ConversionData;
import com.opensymphony.xwork2.interceptor.ParameterNameAware;
import com.opensymphony.xwork2.interceptor.ValidationAware;

import name.matthewgreet.strutscommons.action.AbstractActionSupport;
import name.matthewgreet.strutscommons.action.LoggingAware;
import name.matthewgreet.strutscommons.action.ValidationAware2;
import name.matthewgreet.strutscommons.annotation.Required.MessageType;
import name.matthewgreet.strutscommons.form.Form;
import name.matthewgreet.strutscommons.form.ManualParameterConversionResult;
import name.matthewgreet.strutscommons.interceptor.AnnotationValidationInterceptor2;
import name.matthewgreet.strutscommons.policy.Adjuster;
import name.matthewgreet.strutscommons.policy.CollectionConverter;
import name.matthewgreet.strutscommons.policy.CollectionPostConversionAdjuster;
import name.matthewgreet.strutscommons.policy.CollectionPostConversionValidator;
import name.matthewgreet.strutscommons.policy.ConversionResult;
import name.matthewgreet.strutscommons.policy.Converter;
import name.matthewgreet.strutscommons.policy.NonConversionValidator;
import name.matthewgreet.strutscommons.policy.PostConversionAdjuster;
import name.matthewgreet.strutscommons.policy.PostConversionValidator;
import name.matthewgreet.strutscommons.policy.ValidationResult;
import name.matthewgreet.strutscommons.util.DefaultAnnotationValidator.FieldContext.ConversionMode;
import name.matthewgreet.strutscommons.util.DefaultAnnotationValidator.FieldContext.FieldStep;
import name.matthewgreet.strutscommons.util.DefaultAnnotationValidator.InterceptorContext.InterceptorStep;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.AnnotationEntries;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.CategoriseFieldResult;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.ConfiguredPolicy;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.FieldUsage;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.PairFieldUsage;

/**
 * <P>Implementation of {@link AnnotationValidatior} used by {@link AnnotationValidationInterceptor2}.</P>
 */
@SuppressWarnings("deprecation")
public class DefaultAnnotationValidator implements AnnotationValidatior {
	/**
     * Describes the result of a attempting to convert from a string to the generic type, and the field to receive it, 
     * successful or not.
     */
    public static class ConversionFieldResult<T> {
        private ConversionResult<T> conversionResult;
        private Field recipeintField;
        
        public ConversionFieldResult(Field recipeintField, ConversionResult<T> conversionResult) {
            super();
            this.recipeintField = recipeintField;
            this.conversionResult = conversionResult;
        }
        
        
        public ConversionResult<T> getConversionResult() {
            return conversionResult;
        }
        public Field getRecipeintField() {
            return recipeintField;
        }
    }
    
    /**
     * Thread local execution state of algorithm about a field.  Use {@link #getInstance} to obtain the current state.  
     * Properties are set as the algorithm progresses and these are divided into sets, shown below.  Documentation of 
     * each library function defines the sets in use as well as its purpose. 
     * 
     * <P>Property set 1</P>
     * <UL>
     *   <LI>{@link #getStep step}</LI>
     * </UL>
     * 
     * <P>Property set 2</P>
     * <UL>
     *   <LI>{@link #getConversionMode conversionMode}</LI>
     *   <LI>{@link #getParameter parameter}</LI>
     *   <LI>{@link #getFieldUsage fieldUsage}</LI>
     *   <LI>{@link #getPairFieldUsage pairFieldUsage}</LI>
     *   <LI>{@link #getAnnountationEntries annountationEntries}</LI>
     *   <LI>{@link #getAnnotatedFieldName annotatedFieldName}</LI>
     * </UL>
     * 
     * <P>Property set 3</P>
     * <UL>
     *   <LI>{@link #getFormattedValue formattedValue}</LI>
     *   <LI>{@link #getConfiguredPolicy configuredPolicy}</LI>
     * </UL>
     * 
     * <P>Property set 4</P>
     * <UL>
     *   <LI>{@link #getValidationResult validationResult}</LI>
     * </UL>
     * 
     */
    public static class FieldContext<T> {
    	public enum ConversionMode {AUTO, DEFAULT, FILE, NO_CONVERSION, PAIR, SET_ONLY}
    	public enum FieldStep {START, ADJUSTING, NON_CONVERSION_VALIDATING, CONVERTING, POST_CONVERSION_ADJUSTING, POST_CONVERSION_VALIDATING, END}
    	
    	private static ThreadLocal<FieldContext<?>> instance = new ThreadLocal<>();
    	
    	@SuppressWarnings("unchecked")
		public static <T> FieldContext<T> getInstance() {
    		return (FieldContext<T>)instance.get();
    	}
    	public static <T> void setInstance(FieldContext<T> value) {
    		instance.set(value);
    	}

    	

    	// Set 1
    	private FieldStep step;
    	// Set 2
    	private ConversionMode conversionMode;
    	private Parameter parameter;
    	private FieldUsage<T> fieldUsage;
    	private PairFieldUsage<T> pairFieldUsage;
    	private AnnotationEntries<T> annountationEntries;
    	private String annotatedFieldName;
    	// Set 3
    	private String formattedValue;
    	private ConfiguredPolicy<T> configuredPolicy;
    	// Set 4
    	private ValidationResult validationResult;
    	// Set 5
    	private ConversionResult<T> conversionResult;
        private T parsedValue;
    	private Collection<T> parsedCollectionValue;

		protected FieldContext() {
			super();
			reset();
		}

		public void reset() {
			step = null;
			conversionMode = null;
	    	parameter = null;
	    	fieldUsage = null;
	    	pairFieldUsage = null;
	    	annountationEntries = null;
	    	annotatedFieldName = null;
	    	
	    	formattedValue = null;
	    	configuredPolicy = null;
	    	validationResult = null;
	    	conversionResult = null;
	        parsedValue = null;
	    	parsedCollectionValue = null;
		}
		
		
		/**
		 * Returns current, field-level processing step.
		 */
		public FieldStep getStep() {
			return step;
		}
		public void setStep(FieldStep step) {
			this.step = step;
		}

		/**
		 * Returns conversion mode, defining how form field is set from a request parameter. 
		 */
		public ConversionMode getConversionMode() {
			return conversionMode;
		}
		public void setConversionMode(ConversionMode conversionMode) {
			this.conversionMode = conversionMode;
		}
		
		/**
		 * Returns the request parameter applying to the form field, which can be an {@link Empty} instance. 
		 */
		public Parameter getParameter() {
			return parameter;
		}
		public void setParameter(Parameter parameter) {
			this.parameter = parameter;
		}

		/**
		 * Returns field and configured policies to apply except for pair conversion mode.
		 */
		public FieldUsage<T> getFieldUsage() {
			return fieldUsage;
		}
		public void setFieldUsage(FieldUsage<T> fieldUsage) {
			this.fieldUsage = fieldUsage;
		}

		/**
		 * Returns field and configured policies to apply for pair conversion mode.
		 */
		public PairFieldUsage<T> getPairFieldUsage() {
			return pairFieldUsage;
		}
		public void setPairFieldUsage(PairFieldUsage<T> pairFieldUsage) {
			this.pairFieldUsage = pairFieldUsage;
		}

		/**
		 * Returns configured policies to apply.
		 */
		public AnnotationEntries<T> getAnnountationEntries() {
			return annountationEntries;
		}
		public void setAnnountationEntries(AnnotationEntries<T> annountationEntries) {
			this.annountationEntries = annountationEntries;
		}
		
		/**
		 * Returns name of field, which is the string field in a pair conversion.
		 */
		public String getAnnotatedFieldName() {
			return annotatedFieldName;
		}
		public void setAnnotatedFieldName(String annotatedFieldName) {
			this.annotatedFieldName = annotatedFieldName;
		}

		/**
		 * Returns the request parameter value, which is adjusted by adjuster policies after the ADJUSTING step. 
		 */
		public String getFormattedValue() {
			return formattedValue;
		}
		public void setFormattedValue(String formattedValue) {
			this.formattedValue = formattedValue;
		}

		/**
		 * Returns the policy being processed. 
		 */
		public ConfiguredPolicy<T> getConfiguredPolicy() {
			return configuredPolicy;
		}
		public void setConfiguredPolicy(ConfiguredPolicy<T> configuredPolicy) {
			this.configuredPolicy = configuredPolicy;
		}

		/**
		 * Returns result of a non-conversion or post-conversion validation, if any. 
		 */
		public ValidationResult getValidationResult() {
			return validationResult;
		}
		public void setValidationResult(ValidationResult validationResult) {
			this.validationResult = validationResult;
		}
		
		/**
		 * Returns result of a conversion, if any.
		 */
		public ConversionResult<T> getConversionResult() {
			return conversionResult;
		}
		public void setConversionResult(ConversionResult<T> conversionResult) {
			this.conversionResult = conversionResult;
		}
    	
		/**
		 * Returns successfully converted value, for a single value field. 
		 */
		public T getParsedValue() {
			return parsedValue;
		}
		public void setParsedValue(T parsedValue) {
			this.parsedValue = parsedValue;
		}

		/**
		 * Returns successfully converted value, for a collection field. 
		 */
		public Collection<T> getParsedCollectionValue() {
			return parsedCollectionValue;
		}
		public void setParsedCollectionValue(Collection<T> parsedCollectionValue) {
			this.parsedCollectionValue = parsedCollectionValue;
		}

    }

    /**
     * Thread local execution state of algorithm.  Use {@link #getInstance} to obtain the current state.  Properties 
     * are set as the algorithm progresses and these are divided into sets, shown below.  Documentation of each library 
     * function defines the sets in use as well as its purpose. 
     * 
     * <P>Property set 1</P>
     * <UL>
     *   <LI>{@link #getStep step}</LI>
     * </UL>
     * 
     * <P>Property set 2</P>
     * <UL>
     *   <LI>{@link #getInvocation invocation}</LI>
     *   <LI>{@link #getAction action}</LI>
     *   <LI>{@link #getForm form}</LI>
     *   <LI>{@link #getModelDriven modelDriven}</LI>
     *   <LI>{@link #getTextProvider textProvider}</LI>
     *   <LI>{@link #getValidationAware validationAware}</LI>
     * </UL>
     * 
     * <P>Property set 3</P>
     * <UL>
     *   <LI>{@link #getAllowedRequestParameters allowedRequestParameters}</LI>
     * </UL>
     * 
     * <P>Property set 4</P>
     * <UL>
     *   <LI>{@link #getAutoParameterConversionFields autoParameterConversionFields}</LI>
     *   <LI>{@link #getDefaultParameterConversionFields defaultParameterConversionFields}</LI>
     *   <LI>{@link #getManualParameterConversionFields manualParameterConversionFields}</LI>
     *   <LI>{@link #getNoConversionFields noConversionFields}</LI>
     *   <LI>{@link #getPairConversionFields pairConversionFields}</LI>
     *   <LI>{@link #getSetOnlyFields setOnlyFields}</LI>
     * </UL>
     * 
     */
    public static class InterceptorContext {
    	/**
    	 * Describes processing step.
    	 */
    	public enum InterceptorStep {START, PARAMETER_FILTERING, ANNOTATION_PARSING, FIELD_PROCESSING, MANUAL_VALIDATION, END}

    	private static ThreadLocal<InterceptorContext> instance = new ThreadLocal<>();

    	/**
    	 * Retrieves processing state.
    	 */
    	public static InterceptorContext getInstance() {
    		return instance.get();
    	}
    	public static void setInstance(InterceptorContext value) {
    		instance.set(value);
    	}

    	// Set 1
    	private InterceptorStep step;
    	// Set 2
    	private ActionInvocation invocation;
    	private Object action;
    	private Object form;
    	private boolean modelDriven;
    	private TextProvider textProvider;
    	private ValidationAware validationAware;
    	private Logger validationLogger;
    	// Set 3
    	private HttpParameters allowedRequestParameters;
    	// Set 4
    	private Collection<FieldUsage<?>> autoParameterConversionFields;	// Converted from parameter using annotated converter
    	private Collection<FieldUsage<?>> defaultParameterConversionFields;	// Converted from parameter using default converter
    	private Collection<FieldUsage<?>> fileFields;                       // Set from special parameter created by FileUploadIntereceptor
    	private Collection<FieldUsage<?>> manualParameterConversionFields;	// Calls form to convert from parameter
    	private Collection<FieldUsage<?>> noConversionFields;				// Set from parameter with no conversion and other policies apply 
    	private Collection<PairFieldUsage<?>> pairConversionFields;			// Sets string field from parameter, than converts that to non-string field
    	private Collection<FieldUsage<?>> setOnlyFields;				    // Set from parameter with no conversion and no policies apply 
    	
    	
		protected InterceptorContext() {
			super();
			reset();
		}
		
		public void reset() {
	    	step = null;
	    	invocation = null;
	    	action = null;
	    	form = null;
	    	modelDriven = false;
	    	textProvider = null;
	    	validationAware = null;
	    	
	    	allowedRequestParameters = null;
	    	autoParameterConversionFields = new ArrayList<>();
	    	defaultParameterConversionFields = new ArrayList<>();
	    	fileFields = new ArrayList<>();
	    	manualParameterConversionFields = new ArrayList<>();
	    	noConversionFields = new ArrayList<>(); 
	    	pairConversionFields = new ArrayList<>();
	    	setOnlyFields = new ArrayList<>();
		}

		/**
		 * Returns current, top-level processing step.
		 */
		public InterceptorStep getStep() {
			return step;
		}
		public void setStep(InterceptorStep step) {
			this.step = step;
		}

		/**
		 * Returns Action's execution state.
		 */
		public ActionInvocation getInvocation() {
			return invocation;
		}
		public void setInvocation(ActionInvocation invocation) {
			this.invocation = invocation;
		}

		/**
		 * Returns current Action.
		 */
		public Object getAction() {
			return action;
		}
		public void setAction(Object action) {
			this.action = action;
		}
		
		/**
		 * Returns whatever's the Action's form, whether it's the Action itself, or the model for a ModelDriven Action. 
		 */
		public Object getForm() {
			return form;
		}
		public void setForm(Object form) {
			this.form = form;
		}
		
		/**
		 * Returns whether Action is ModelDriven. 
		 */
		public boolean getModelDriven() {
			return modelDriven;
		}
		public void setModelDriven(boolean modelDriven) {
			this.modelDriven = modelDriven;
		}

		/**
		 * Returns message key translator, which can be null if not provided by Action. 
		 */
		public TextProvider getTextProvider() {
			return textProvider;
		}
		public void setTextProvider(TextProvider textProvider) {
			this.textProvider = textProvider;
		}

		/**
		 * Returns recipient of error messages, which can be null if not provided by Action.
		 */
		public ValidationAware getValidationAware() {
			return validationAware;
		}
		public void setValidationAware(ValidationAware validationAware) {
			this.validationAware = validationAware;
		}

		/**
		 * Returns Log4j2 logger for logging conversion and validation failures. 
		 */
		public Logger getValidationLogger() {
			return validationLogger;
		}
		public void setValidationLogger(Logger validationLogger) {
			this.validationLogger = validationLogger;
		}
		
		/**
		 * Returns request parameters that have passed parameter filtering step.
		 */
		public HttpParameters getAllowedRequestParameters() {
			return allowedRequestParameters;
		}
		public void setAllowedRequestParameters(HttpParameters allowedRequestParameters) {
			this.allowedRequestParameters = allowedRequestParameters;
		}
		
		/**
		 * Returns policies for fields where a converted request parameter is directly set according to a converter 
		 * annotation.   
		 */
		public Collection<FieldUsage<?>> getAutoParameterConversionFields() {
			return autoParameterConversionFields;
		}
		public void setAutoParameterConversionFields(Collection<FieldUsage<?>> autoParameterConversionFields) {
			this.autoParameterConversionFields = autoParameterConversionFields;
		}

		/**
		 * Returns policies for fields where a converted request parameter is directly set according to the default 
		 * converter for the data type.   
		 */
		public Collection<FieldUsage<?>> getDefaultParameterConversionFields() {
			return defaultParameterConversionFields;
		}
		public void setDefaultParameterConversionFields(Collection<FieldUsage<?>> defaultParameterConversionFields) {
			this.defaultParameterConversionFields = defaultParameterConversionFields;
		}

		/**
		 * Returns fields that receive file parameters created by the FileUploadInterceptor. 
		 */
		public Collection<FieldUsage<?>> getFileFields() {
			return fileFields;
		}
		public void setFileFields(Collection<FieldUsage<?>> fileFields) {
			this.fileFields = fileFields;
		}
		
		/**
		 * Returns fields that will be validated and converted from request parameters using form code. 
		 */
		public Collection<FieldUsage<?>> getManualParameterConversionFields() {
			return manualParameterConversionFields;
		}
		public void setManualParameterConversionFields(Collection<FieldUsage<?>> manualParameterConversionFields) {
			this.manualParameterConversionFields = manualParameterConversionFields;
		}
		
		/**
		 * Returns policies for string fields set from a request parameter without conversion.  
		 */
		public Collection<FieldUsage<?>> getNoConversionFields() {
			return noConversionFields;
		}
		public void setNoConversionFields(Collection<FieldUsage<?>> noConversionFields) {
			this.noConversionFields = noConversionFields;
		}
		
		/**
		 * Returns policies for string/non-string field pairs where the string field stores the unconverted request 
		 * parameter and non-string field has the converted value. 
		 */
		public Collection<PairFieldUsage<?>> getPairConversionFields() {
			return pairConversionFields;
		}
		public void setPairConversionFields(Collection<PairFieldUsage<?>> pairConversionFields) {
			this.pairConversionFields = pairConversionFields;
		}
		
		/**
		 * Returns string fields that are set from a request parameter with no validation or other policies. 
		 */
		public Collection<FieldUsage<?>> getSetOnlyFields() {
			return setOnlyFields;
		}
		public void setSetOnlyFields(Collection<FieldUsage<?>> setOnlyFields) {
			this.setOnlyFields = setOnlyFields;
		}
		
    }
    
    private Logger ownLogger = LogManager.getLogger(DefaultAnnotationValidator.class);

    
    private MessageType defaultMessageType = MessageType.ERROR;
    private boolean ignoreNonAnnotatedFields = false;
    private int paramNameMaxLength = 100;
    private PolicyLookup policyLookup = null;
    
    
    public DefaultAnnotationValidator() {
    	super();
    }

    
    /**
     * <P>Sets how form fields should be converted (InterceptorContext setting property set 4).</P>
     * 
     * <P>Can access InterceptorContext property set 3 and lower.</P>
     */
    protected void categoriseFields(Collection<Field> formFields) {
    	CategoriseFieldResult categoriseFieldResult;
    	InterceptorContext interceptorContext;
    	
    	interceptorContext = InterceptorContext.getInstance();
    	
    	categoriseFieldResult = InterceptorCommonLibrary.categoriseFormFields(formFields, policyLookup, ignoreNonAnnotatedFields);
    	interceptorContext.getAutoParameterConversionFields().addAll(categoriseFieldResult.getAutoParameterConversionFields());
    	interceptorContext.getDefaultParameterConversionFields().addAll(categoriseFieldResult.getDefaultParameterConversionFields());
    	interceptorContext.getFileFields().addAll(categoriseFieldResult.getFileFields());
    	interceptorContext.getManualParameterConversionFields().addAll(categoriseFieldResult.getManualParameterConversionFields());
    	interceptorContext.getNoConversionFields().addAll(categoriseFieldResult.getNoConversionFields()); 
    	interceptorContext.getPairConversionFields().addAll(categoriseFieldResult.getPairConversionFields());
    	interceptorContext.getSetOnlyFields().addAll(categoriseFieldResult.getSetOnlyFields());
	}

    /**
     * <P>Checks if collection conversion results means a message must be written and writes appropriate type if needed.</P>  
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
    protected <T> void checkCollectionConversionMessage() {
    	ConversionResult<T> conversionResult;
    	CollectionConverter<?,T> collectionConverter;
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
        MessageType messageType;
        String message, messageKey, finalMessage;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        conversionResult = fieldContext.getConversionResult();
        collectionConverter = fieldContext.getConfiguredPolicy().getCollectionConverter();
        if (!conversionResult.getSuccess()) {
            if (conversionResult.getMessage() != null) {
                message = conversionResult.getMessage();
            } else {
                message = collectionConverter.getMessage();
            }
            if (conversionResult.getMessageKey() != null) {
                messageKey = conversionResult.getMessageKey();
            } else {
                messageKey = collectionConverter.getMessageKey();
            }
            finalMessage = getMessage(interceptorContext.getTextProvider(), messageKey, message);
            if (conversionResult.getMessageType() != null) {
                messageType = conversionResult.getMessageType();
            } else {
                messageType = collectionConverter.getMessageType();
            }
            if (messageType == MessageType.DEFAULT) {
            	messageType = getMessageTypeForDefault();
            }
            switch (messageType) {
            case ERROR:
                writeGeneralError(interceptorContext.getValidationAware(), finalMessage);
                break;
            case FIELD:
                writeFieldError(interceptorContext.getValidationAware(), fieldContext.getAnnotatedFieldName(), finalMessage);
                break;
            case DEFAULT:
            case IGNORE:
            	ignoreMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_DEBUG:
            	logDebugMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_ERROR:
            	logErrorMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_INFO:
            	logInfoMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_TRACE:
            	logTraceMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_WARN:
            	logWarnMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case MESSAGE:
                writeInfoMessage(interceptorContext.getValidationAware(), finalMessage);
                break;
            case WARNING:
                writeWarningMessage(interceptorContext.getValidationAware(), finalMessage);
                break;
            }
        }
    }


	/**
     * Checks if conversion results means a message must be written and writes appropriate type if needed.  
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
    protected <T> void checkCollectionPostConversionMessage() {
        MessageType messageType;
        CollectionPostConversionValidator<?,T> validator;
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
        ValidationResult validationResult;
        String message, messageKey, finalMessage;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        validationResult = fieldContext.getValidationResult();
        if (!validationResult.getSuccess()) {
        	validator = fieldContext.getConfiguredPolicy().getCollectionPostConversionValidator();
        	
            if (validationResult.getMessage() != null) {
                message = validationResult.getMessage();
            } else {
                message = validator.getMessage();
            }
            if (validationResult.getMessageKey() != null) {
                messageKey = validationResult.getMessageKey();
            } else {
                messageKey = validator.getMessageKey();
            }
            finalMessage = getMessage(interceptorContext.getTextProvider(), messageKey, message);
            if (validationResult.getMessageType() != null) {
                messageType = validationResult.getMessageType();
            } else {
                messageType = validator.getMessageType();
            }
            if (messageType == MessageType.DEFAULT) {
            	messageType = getMessageTypeForDefault();
            }
            switch (messageType) {
            case ERROR:
                writeGeneralError(interceptorContext.getValidationAware(), finalMessage);
                break;
            case FIELD:
                writeFieldError(interceptorContext.getValidationAware(), fieldContext.getAnnotatedFieldName(), finalMessage);
                break;
            case DEFAULT:
            case IGNORE:
            	ignoreMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_DEBUG:
            	logDebugMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_ERROR:
            	logErrorMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_INFO:
            	logInfoMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_TRACE:
            	logTraceMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_WARN:
            	logWarnMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case MESSAGE:
                writeInfoMessage(interceptorContext.getValidationAware(), finalMessage);
                break;
            case WARNING:
                writeWarningMessage(interceptorContext.getValidationAware(), finalMessage);
                break;
            }
        }
    }

    /**
     * <P>Checks if conversion results means a message must be written and writes appropriate type if needed.</P>  
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
    protected <T> void checkConversionMessage() {
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
    	ConversionResult<T> conversionResult;
    	Converter<?,T> converter;
        MessageType messageType;
        String message, messageKey, finalMessage;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        conversionResult = fieldContext.getConversionResult();
        converter = fieldContext.getConfiguredPolicy().getConverter();
        if (!conversionResult.getSuccess()) {
            if (conversionResult.getMessage() != null) {
                message = conversionResult.getMessage();
            } else {
                message = converter.getMessage();
            }
            if (conversionResult.getMessageKey() != null) {
                messageKey = conversionResult.getMessageKey();
            } else {
                messageKey = converter.getMessageKey();
            }
            finalMessage = getMessage(interceptorContext.getTextProvider(), messageKey, message);
            if (finalMessage.length() == 0) {
            	finalMessage = getDefaultMessage(fieldContext.getAnnotatedFieldName(), converter.getRecipientClass());
            }
            if (conversionResult.getMessageType() != null) {
                messageType = conversionResult.getMessageType();
            } else {
                messageType = converter.getMessageType();
            }
            if (messageType == MessageType.DEFAULT) {
            	messageType = getMessageTypeForDefault();
            }
            switch (messageType) {
            case ERROR:
                writeGeneralError(interceptorContext.getValidationAware(), finalMessage);
                break;
            case FIELD:
                writeFieldError(interceptorContext.getValidationAware(), fieldContext.getAnnotatedFieldName(), finalMessage);
                break;
            case DEFAULT:
            case IGNORE:
            	ignoreMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_DEBUG:
            	logDebugMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_ERROR:
            	logErrorMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_INFO:
            	logInfoMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_TRACE:
            	logTraceMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_WARN:
            	logWarnMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case MESSAGE:
                writeInfoMessage(interceptorContext.getValidationAware(), finalMessage);
                break;
            case WARNING:
                writeWarningMessage(interceptorContext.getValidationAware(), finalMessage);
                break;
            }
        }
    }

	/**
     * <P>Checks if non-conversion validation results means a message must be written and writes appropriate type if 
     * needed.</P>  
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
    protected <T> void checkNonConversionMessage() {
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
        MessageType messageType;
        NonConversionValidator<?> validator;
        TextProvider textProvider;
        ValidationAware validationAware;
        ValidationResult validationResult;
        String message, messageKey, finalMessage;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        validationResult = fieldContext.getValidationResult();
        if (!validationResult.getSuccess()) {
        	validator = fieldContext.getConfiguredPolicy().getNonConversionValidator();
        	textProvider = interceptorContext.getTextProvider();
        	validationAware = interceptorContext.getValidationAware();
        	
            if (validationResult.getMessage() != null) {
                message = validationResult.getMessage();
            } else {
                message = validator.getMessage();
            }
            if (validationResult.getMessageKey() != null) {
                messageKey = validationResult.getMessageKey();
            } else {
                messageKey = validator.getMessageKey();
            }
            finalMessage = getMessage(textProvider, messageKey, message);
            if (validationResult.getMessageType() != null) {
                messageType = validationResult.getMessageType();
            } else {
                messageType = validator.getMessageType();
            }
            if (messageType == MessageType.DEFAULT) {
            	messageType = getMessageTypeForDefault();
            }
            switch (messageType) {
            case ERROR:
                writeGeneralError(validationAware, finalMessage);
                break;
            case FIELD:
                writeFieldError(validationAware, fieldContext.getAnnotatedFieldName(), finalMessage);
                break;
            case DEFAULT:
            case IGNORE:
            	ignoreMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_DEBUG:
            	logDebugMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_ERROR:
            	logErrorMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_INFO:
            	logInfoMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_TRACE:
            	logTraceMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_WARN:
            	logWarnMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case MESSAGE:
                writeInfoMessage(validationAware, finalMessage);
                break;
            case WARNING:
                writeWarningMessage(validationAware, finalMessage);
                break;
            }
        }
    }

    /**
     * Checks if conversion results means a message must be written and writes appropriate type if needed.  
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
    protected <T> void checkPostConversionMessage() {
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
        MessageType messageType;
        PostConversionValidator<?,T> validator;
        ValidationResult validationResult;
        String message, messageKey, finalMessage;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        validationResult = fieldContext.getValidationResult();
        if (!validationResult.getSuccess()) {
        	validator = fieldContext.getConfiguredPolicy().getPostConversionValidator();
        	
            if (validationResult.getMessage() != null) {
                message = validationResult.getMessage();
            } else {
                message = validator.getMessage();
            }
            if (validationResult.getMessageKey() != null) {
                messageKey = validationResult.getMessageKey();
            } else {
                messageKey = validator.getMessageKey();
            }
            finalMessage = getMessage(interceptorContext.getTextProvider(), messageKey, message);
            if (validationResult.getMessageType() != null) {
                messageType = validationResult.getMessageType();
            } else {
                messageType = validator.getMessageType();
            }
            if (messageType == MessageType.DEFAULT) {
            	messageType = getMessageTypeForDefault();
            }
            switch (messageType) {
            case ERROR:
                writeGeneralError(interceptorContext.getValidationAware(), finalMessage);
                break;
            case FIELD:
                writeFieldError(interceptorContext.getValidationAware(), fieldContext.getAnnotatedFieldName(), finalMessage);
                break;
            case DEFAULT:
            case IGNORE:
            	ignoreMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_DEBUG:
            	logDebugMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_ERROR:
            	logErrorMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_INFO:
            	logInfoMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_TRACE:
            	logTraceMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case LOG_WARN:
            	logWarnMessage(interceptorContext.getValidationLogger(), finalMessage);
            	break;
            case MESSAGE:
                writeInfoMessage(interceptorContext.getValidationAware(), finalMessage);
                break;
            case WARNING:
                writeWarningMessage(interceptorContext.getValidationAware(), finalMessage);
                break;
            }
        }
    }

	/**
     * Called when processing of a form field is complete.  Just sets end state and really exists for extending 
     * subclasses. 
     */
    protected <T> void endFieldContext() {
    	FieldContext<T> fieldContext;
    	
    	fieldContext = FieldContext.getInstance();
    	fieldContext.setStep(FieldStep.END);
	}

	/**
     * Called when this interceptor completes (and is not disabled).  Does nothing and really exists for extending subclasses. 
     */
	protected void endInterceptorContext() {
		// Empty
	}
	
	/**
     * <P>Returns form fields except those that should be skipped before any policy annotations are analysed.  Skips 
     * none and exists for extending subclasses.</P> 
     * 
     * <P>Can access InterceptorContext property set 3 and lower.</P>
     */
	protected Collection<Field> filterFormFieldsPreCategorise(Collection<Field> formFields) {
		return formFields;
	}

	/**
     * <P>Does not nothing and is a placeholder called after {@link #categoriseFields categoriseFields}.</P> 
     * 
     * <P>Can access InterceptorContext property set 4 and lower.</P>
     */
	protected void filterFormFieldsPostCategorise() {
		// Empty
	}

	/**
     * <P>Removes rejected request parameters from the allowed request parameters (property set 3), which are those that 
     * exceed max parameter length or fail filter if Action implements ParameterNameAware.</P> 
     * 
     * <P>Can access InterceptorContext property set 3 and lower.</P>
     */
	protected void filterRequestParameters() {
		InterceptorContext interceptorContext;
		HttpParameters allowedRequestParameters;
		ParameterNameAware parameterNameAware;
		Map<String, Parameter> filteredParameters;
		Set<String> parameterNames;
		Object action;
    	
		interceptorContext = InterceptorContext.getInstance();
    	allowedRequestParameters = interceptorContext.getAllowedRequestParameters();
    	filteredParameters = new HashMap<>();
		parameterNames = allowedRequestParameters.keySet();
		
		action = interceptorContext.getAction();
		if (action instanceof ParameterNameAware) {
			parameterNameAware = (ParameterNameAware)action;
		} else {
			parameterNameAware = null;
		}
        
		for (String parameterName: parameterNames) {
			if (parameterName.length() > paramNameMaxLength) {
				ownLogger.warn("Parameter name " + parameterName + " exceeds maxium length of " + paramNameMaxLength);
				continue;
			}
			if (parameterNameAware != null && parameterNameAware.acceptableParameterName(parameterName)) {
				continue;
			}
			filteredParameters.put(parameterName, allowedRequestParameters.get(parameterName));
		}
		
		interceptorContext.setAllowedRequestParameters(HttpParameters.create(filteredParameters).build());
	}

    /**
     * Returns error message if a default converter fails.
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
    protected String getDefaultMessage(String recipientFieldName, Class<?> recipientClass) {
		return "Value for " + recipientFieldName + " is invalid";
	}

    /**
	 * <P>Returns form member fields that can plausibly receive form data.  This excludes static and final fields.</P>
     * 
     * <P>Can access InterceptorContext property set 3 and lower.</P>
     */
    protected Collection<Field> getFormFields(Collection<Field> allFields) {
    	return InterceptorCommonLibrary.filterPlausibleFormFields(allFields);
    }
    
    /**
     * Returns message to use, extracted from text provider if required.  textProvider can be null if Action does not 
     * implement it.
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
    protected String getMessage(TextProvider textProvider, String messageKey, String message) {
        String result;
        
        if (textProvider != null && messageKey != null && messageKey.length() > 0 && textProvider.hasKey(messageKey)) {
            result = textProvider.getText(messageKey);
        } else {
            result = message;
        }
        return result;
    }
    
    /**
     * <P>Returns target type for conversion and validation failures if annotation uses default type.</P>  
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
	protected MessageType getMessageTypeForDefault() {
		return this.defaultMessageType;
	}


    /**
     * <P>Returns all form fields of defined by form class.</P> 
     * 
     * <P>Can access InterceptorContext property set 3 and lower.</P>
     */
    protected Collection<Field> getProperties(Class<?> type) {
    	return InterceptorCommonLibrary.getProperties(type);
    }
    
    protected Field getRecipientFieldForConversionMode() {
        FieldContext<?> fieldContext;
        InterceptorContext interceptorContext;
        Field result;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        result = null;
        switch (fieldContext.getConversionMode()) {
        case AUTO:
        case DEFAULT:
        case FILE:
            result = fieldContext.getFieldUsage().getField();
            break;
        case NO_CONVERSION:
        case SET_ONLY:
        	ownLogger.error("processConverter inexplicably called for no conversion or set only" + "  Struts action=" + interceptorContext.getAction().getClass() + 
                "  form class=" + interceptorContext.getForm().getClass() + "  field=" + fieldContext.getAnnotatedFieldName());
            throw new IllegalArgumentException("processConverter inexplicably called for no conversion");
        case PAIR:
            result = fieldContext.getPairFieldUsage().getUnformattedField();
            break;
        }
        return result;
    }

    /**
     * <P>Returns Log4j2 logger to use for conversion and validation failures.  Uses Action's 
     * {@link AbstractActionSupport#getLogger} if it exists, otherwise this classes logger.  Cannot return null.</P>
     * 
     * <P>Can access InterceptorContext property set 1.</P>
     */
    protected Logger getValidationLogger(ActionInvocation invocation, Object action) {
		Logger result;
		
		result = null;
		if (action instanceof LoggingAware) {
			result = ((LoggingAware)action).getLogger();
		}
		if (result == null) {
			result = ownLogger;
		}
		return result;
	}

    /**
     * Called when an error message is to be ignored.
     */
    protected void ignoreMessage(Logger logger, String message) {
		// Does nothing
	}

    /**
     * Logs message to Action's logger at DEBUG level.
     */
	protected void logDebugMessage(Logger validationLogger, String message) {
		if (validationLogger.isDebugEnabled()) {
			validationLogger.debug(message);
		}
	}

    /**
     * Logs message to Action's logger at ERROR level.
     */
	protected void logErrorMessage(Logger validationLogger, String message) {
		if (validationLogger.isErrorEnabled()) {
			validationLogger.error(message);
		}
	}

    /**
     * Logs message to Action's logger at INFO level.
     */
	protected void logInfoMessage(Logger validationLogger, String message) {
		if (validationLogger.isInfoEnabled()) {
			validationLogger.info(message);
		}
	}

    /**
     * Logs message to Action's logger at TRACE level.
     */
	protected void logTraceMessage(Logger validationLogger, String message) {
		if (validationLogger.isTraceEnabled()) {
			validationLogger.trace(message);
		}
	}

    /**
     * Logs message to Action's logger at WARN level.
     */
	protected void logWarnMessage(Logger validationLogger, String message) {
		if (validationLogger.isWarnEnabled()) {
			validationLogger.warn(message);
		}
	}

    /**
     * Returns an empty collection to set the recipient field and will accept converted values. 
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
    protected <T> Collection<T> makeCollectionForRecipient(Class<?> recipientFieldClass) {
        return InterceptorCommonLibrary.makeCollectionForRecipient(recipientFieldClass);
    }

	/**
	 * Returns an initial execution state about a field.
	 */
	protected <T> FieldContext<T> makeFieldContext() {
    	return new FieldContext<T>();
    }

	/**
	 * Returns an initial execution state.
	 */
    protected InterceptorContext makeInterceptorContext() {
    	return new InterceptorContext();
    }
    
    /**
     * <P>Invokes adjuster, updates FieldContext formattedValue property, and updates form field for no conversion and 
     * pair conversion modes.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 3 and lower.</P>
     */
    protected <T> void processAdjuster() throws IllegalArgumentException, IllegalAccessException {
        Adjuster<?> adjuster;
        FieldContext<T> fieldContext;
        InterceptorContext interceptorContext;
        Field recipientField;
        String formattedValue, newValue;
        boolean failed;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        formattedValue = fieldContext.getFormattedValue();
        adjuster = fieldContext.getConfiguredPolicy().getAdjuster();
        if (formattedValue.length() > 0 || adjuster.getProcessNoValue()) {
        	failed = false;
        	try {
        		newValue = adjuster.adjust(formattedValue);
	        }
	        catch (Exception e) {
	        	ownLogger.error("Adjuster failed" + 
	                "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
	                "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
	                "  formatted value=" + formattedValue, e);
	            newValue = formattedValue;
	        	failed = true;
	        }
        	if (!failed) {
		        fieldContext.setFormattedValue(newValue);
		        recipientField = null;
		        switch (fieldContext.getConversionMode()) {
				case AUTO:
				case DEFAULT:
					// No formatted field
					break;
				case NO_CONVERSION:
					recipientField = fieldContext.getFieldUsage().getField();
					fieldContext.getFieldUsage().getField().set(interceptorContext.getForm(), fieldContext.getFormattedValue());
					break;
				case PAIR:
					recipientField = fieldContext.getPairFieldUsage().getFormattedField();
					fieldContext.getPairFieldUsage().getFormattedField().set(interceptorContext.getForm(), fieldContext.getFormattedValue());
					break;
				case FILE:
				case SET_ONLY:
					// Not applicable
					break;
		        }
		        if (recipientField != null) {
	        		try {
	        			recipientField.set(interceptorContext.getForm(), newValue);
			        }
			        catch (Exception e) {
			        	ownLogger.error("Setting recipient field failed" + 
			                "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
			                "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
			                "  formatted value=" + fieldContext.getFormattedValue() + "  recipient field=" + recipientField.getName() + 
			                "  value to set=" + newValue, e);
			        }
		        }
        	}
        }
    }

    /**
     * <P>Invokes collection converter, sets field (non-string collection for pair conversion), sets FieldContext 
     * conversionResult property and writes error message if needed; or sets FieldContext conversionResult property to 
     * skipped collection if the adjusted formatted value is empty string.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
    protected <T> void processCollectionConverter() {
        CollectionConverter<?,T> collectionConverter;
        ConversionResult<T> conversionResult;
        FieldContext<T> fieldContext;
        InterceptorContext interceptorContext;
        Field recipientField;
        Class<?> recipientFieldClass;
        String formattedValue;
        Collection<T> parsedCollectionValue;

        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        formattedValue = fieldContext.getFormattedValue();
        collectionConverter = fieldContext.getConfiguredPolicy().getCollectionConverter();
        recipientField = null;
        switch (fieldContext.getConversionMode()) {
		case AUTO:
		case DEFAULT:
			recipientField = fieldContext.getFieldUsage().getField();
			break;
		case FILE:
		case NO_CONVERSION:
		case SET_ONLY:
			ownLogger.error("processConverter inexplicably called for file, no conversion, or set only" + 
		        "  Struts action=" + interceptorContext.getAction().getClass() + 
            	"  form class=" + interceptorContext.getForm().getClass() + "  field=" + fieldContext.getAnnotatedFieldName());
			throw new IllegalArgumentException("processConverter inexplicably called for no conversion");
		case PAIR:
			recipientField = fieldContext.getPairFieldUsage().getUnformattedField();
			break;
        }
        recipientFieldClass = recipientField.getType();
        
        if (formattedValue.length() > 0 || collectionConverter.getProcessNoValue()) {
	        parsedCollectionValue = null;
            if (formattedValue != null && formattedValue.trim().length() > 0) {
            	try {
            	conversionResult = collectionConverter.convert(formattedValue, recipientFieldClass, collectionConverter.getRecipientClass());
		        }
		        catch (Exception e) {
		        	ownLogger.error("Converter failed" + 
		                "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
		                "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
		                "  formatted value=" + formattedValue, e);
		            conversionResult = ConversionResult.makeFailureWithMessageResult("System failure with converter", MessageType.ERROR);
		        }
            } else {
            	conversionResult = ConversionResult.makeSkippedCollectionResult(makeCollectionForRecipient(recipientFieldClass));
            }
            fieldContext.setConversionResult(conversionResult);
            parsedCollectionValue = conversionResult.getParsedCollectionValue();
            fieldContext.setParsedCollectionValue(parsedCollectionValue);
            if (conversionResult.getSuccess()) {
    	        try {
            	recipientField.set(interceptorContext.getForm(), parsedCollectionValue);
    	        }
    	        catch (Exception e) {
    	        	ownLogger.error("Setting recipient field failed" + 
    	                "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
    	                "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
    	                "  formatted value=" + fieldContext.getFormattedValue() + "  recipient field=" + recipientField.getName() + 
    	                "  value to set=" + parsedCollectionValue, e);
    	            conversionResult = ConversionResult.makeFailureResult();
    	        }
            }
	        fieldContext.setConversionResult(conversionResult);
	        
	        checkCollectionConversionMessage();
	        if (!conversionResult.getSuccess()) {
	        	processCollectionConversionRejection();
	        }
        } else {
        	conversionResult = ConversionResult.makeSkippedCollectionResult(makeCollectionForRecipient(recipientFieldClass));
	        fieldContext.setConversionResult(conversionResult);
        }
	}

    /**
     * <P>Handles aftermath of a form field failing the conversion step, which is writing a conversion error for auto 
     * and default conversion mode, or leaving the field as is for no conversion and pair conversion mode.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
	protected <T> void processCollectionConversionRejection() {
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
    	Map<String, ConversionData> conversionErrors;
        Class<T> recipientClass;
    	String formattedValue;
    	
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        formattedValue = fieldContext.getFormattedValue();
        recipientClass = fieldContext.getConfiguredPolicy().getCollectionConverter().getRecipientClass();
    	switch (fieldContext.getConversionMode()) {
		case AUTO:
		case DEFAULT:
			conversionErrors = interceptorContext.getInvocation().getInvocationContext().getConversionErrors();
			conversionErrors.put(fieldContext.getAnnotatedFieldName(), new ConversionData(formattedValue, recipientClass));
			break;
		case NO_CONVERSION:
		case PAIR:
			// Formatted value in form field
			break;
		case FILE:
		case SET_ONLY:
			// Not applicable
			break;
    	}
	}

    /**
     * <P>Invokes post-conversion adjuster, sets field (non-string for pair conversion), and sets FieldContext 
     * parsedCollectionValue property; or does nothing if the converted value is null and adjuster skips them.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 5 and lower.</P>
     */
    protected <T> void processCollectionPostConversionAdjuster() {
        ConversionResult<T> conversionResult;
        Field recipientField;
        FieldContext<T> fieldContext;
        InterceptorContext interceptorContext;
        CollectionPostConversionAdjuster<?,T> collectionPostConversionAdjuster;
        Collection<T> newValue;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        conversionResult = fieldContext.getConversionResult();
        collectionPostConversionAdjuster = fieldContext.getConfiguredPolicy().getCollectionPostConversionAdjuster();
        if ((conversionResult.getSuccess() && conversionResult.getParsedCollectionValue() != null) || collectionPostConversionAdjuster.getProcessNoValue()) {
            try {
                newValue = collectionPostConversionAdjuster.adjust(fieldContext.getParsedCollectionValue());
                fieldContext.setParsedCollectionValue(newValue);
                recipientField = getRecipientFieldForConversionMode();
                recipientField.set(interceptorContext.getForm(), newValue);
            }
            catch (Exception e) {
            	ownLogger.error("Collection post-conversion adjuster failed" + 
                    "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
                    "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
                    "  unformatted value=" + fieldContext.getParsedCollectionValue(), e);
            }
        }
    }

    /**
     * <P>Invokes collection post-conversion validator, sets FieldContext validationResult property and writes error 
     * message if needed; or sets FieldContext validationResult property to success if the converted value is null and 
     * validator skips them.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 5 and lower.</P>
     */
    protected <T> void processCollectionPostConversionValidator() {
    	ConversionResult<T> conversionResult;
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
        CollectionPostConversionValidator<?,T> collectionPostConversionValidator;
        ValidationResult validationResult;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        conversionResult = fieldContext.getConversionResult();
        collectionPostConversionValidator = fieldContext.getConfiguredPolicy().getCollectionPostConversionValidator();
        if ((conversionResult.getSuccess() && conversionResult.getParsedCollectionValue() != null) || collectionPostConversionValidator.getProcessNoValue()) {
        	try {
        		validationResult = collectionPostConversionValidator.validate(fieldContext.getParsedCollectionValue());
	        }
	        catch (Exception e) {
	        	ownLogger.error("Post-conversion validator failed" + 
	                "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
	                "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
	                "  unformatted value=" + fieldContext.getParsedValue(), e);
	            validationResult = ValidationResult.makeFailureWithMessageResult("System failure with validator", MessageType.ERROR);
	        }
	        fieldContext.setValidationResult(validationResult);
	        validationResult.setShortCircuit(collectionPostConversionValidator.getShortCircuit());
	        checkCollectionPostConversionMessage();
        } else {
	        fieldContext.setValidationResult(ValidationResult.makeSuccessResult());
        }
    }

    /**
     * <P>Handles aftermath of a form field that failed the post-conversion validation step, which is nothing as error 
     * messages are already written.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
	protected <T> void processCollectionPostConversionRejection() {
		// Does nothing
	}

    /**
     * <P>Invokes converter, sets field (non-string for pair conversion), sets FieldContext conversionResult property 
     * and writes error message if needed; or sets FieldContext conversionResult property to skipped if the adjusted formatted value is empty string.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
	@SuppressWarnings("unchecked")
	protected <T> void processConverter() {
        Converter<?,T> converter;
        ConversionResult<T> conversionResult;
        Field recipientField;
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
        Class<T> recipientClass;
        String formattedValue;
        T parsedValue;

        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        formattedValue = fieldContext.getFormattedValue();
        converter = fieldContext.getConfiguredPolicy().getConverter();
        if (formattedValue.length() > 0 || converter.getProcessNoValue()) {
	        recipientField = getRecipientFieldForConversionMode();
	        recipientClass = (Class<T>)recipientField.getType();
	        
	        parsedValue = null;
            if (formattedValue.trim().length() > 0 || converter.getProcessNoValue()) {
		        try {
	            	conversionResult = converter.convert(formattedValue, recipientClass);
		        }
		        catch (Exception e) {
		        	ownLogger.error("Converter failed" + 
		                "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
		                "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
		                "  formatted value=" + formattedValue, e);
		            conversionResult = ConversionResult.makeFailureWithMessageResult("System failure with converter", MessageType.ERROR);
		        }
            } else {
            	conversionResult = ConversionResult.makeSkippedResult();
            }
            fieldContext.setConversionResult(conversionResult);
            parsedValue = conversionResult.getParsedValue();
            fieldContext.setParsedValue(parsedValue);
            if (conversionResult.getSuccess()) {
		        try {
		            	recipientField.set(interceptorContext.getForm(), parsedValue);
		        }
		        catch (Exception e) {
		        	ownLogger.error("Setting recipient field failed" + 
		                "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
		                "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
		                "  formatted value=" + fieldContext.getFormattedValue() + "  recipient field=" + recipientField.getName() + 
		                "  value to set=" + parsedValue, e);
		            conversionResult = ConversionResult.makeFailureResult();
		        }
            }
	        fieldContext.setConversionResult(conversionResult);
	        
	        checkConversionMessage();
	        if (!conversionResult.getSuccess()) {
	        	processConversionRejection();
	        }
        } else {
            conversionResult = ConversionResult.makeSkippedResult();
	        fieldContext.setConversionResult(conversionResult);
        }
	}

    /**
     * <P>Handles aftermath of a form field failing the conversion step, which is writing a conversion error for auto 
     * and default conversion mode, or leaving the field as is for no conversion and pair conversion mode.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
	protected <T> void processConversionRejection() {
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
    	Map<String, ConversionData> conversionErrors;
        Class<T> recipientClass;
    	String formattedValue;
    	
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        formattedValue = fieldContext.getFormattedValue();
        recipientClass = fieldContext.getConfiguredPolicy().getConverter().getRecipientClass();
    	switch (fieldContext.getConversionMode()) {
		case AUTO:
		case DEFAULT:
			conversionErrors = interceptorContext.getInvocation().getInvocationContext().getConversionErrors();
			conversionErrors.put(fieldContext.getAnnotatedFieldName(), new ConversionData(formattedValue, recipientClass));
			break;
		case NO_CONVERSION:
		case PAIR:
			// Formatted value in form field
			break;
		case FILE:
		case SET_ONLY:
			// Not applicable
			break;
    	}
	}

	/**
     * <P>Sets File form field from special file parameter created by FileUplodIntereceptor, for file conversion mode.</P> 
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 2 and lower.</P>
     */
    protected void processFileFormField() throws IllegalArgumentException, IllegalAccessException {
        FieldContext<?> fieldContext;
        FieldUsage<?> fieldUsage;
        InterceptorContext interceptorContext;
        UploadedFile[] uploadedFiles;
		File[] fileArrayValue;
		File fileValue;
        Collection<File> fileCollectionValue;
        int i;
		
		interceptorContext = InterceptorContext.getInstance();
		fieldContext = FieldContext.getInstance();
		fieldUsage = fieldContext.getFieldUsage();

		if (fieldUsage.getArray()) {
			fileArrayValue = new File[] {};
			if (fieldContext.getParameter() != null && fieldContext.getParameter().getObject() instanceof UploadedFile[]) {
				uploadedFiles = (UploadedFile[])fieldContext.getParameter().getObject();
				fileArrayValue = new File[uploadedFiles.length];
				i = 0;
				for (UploadedFile uploadedFile: uploadedFiles) {
					fileArrayValue[i] = (File)uploadedFile.getContent();
					i++;
				}
			}
			fieldContext.getFieldUsage().getField().set(interceptorContext.getForm(), fileArrayValue);
		} else if (fieldUsage.getCollection()) {
			fileCollectionValue = makeCollectionForRecipient(fieldUsage.getField().getType());
			if (fieldContext.getParameter() != null && fieldContext.getParameter().getObject() instanceof UploadedFile[]) {
				uploadedFiles = (UploadedFile[])fieldContext.getParameter().getObject();
				for (UploadedFile uploadedFile: uploadedFiles) {
					fileCollectionValue.add((File)uploadedFile.getContent());
				}
			}
			fieldContext.getFieldUsage().getField().set(interceptorContext.getForm(), fileCollectionValue);
		} else {
			fileValue = null;
			if (fieldContext.getParameter() != null && fieldContext.getParameter().getObject() instanceof UploadedFile[]) {
				uploadedFiles = (UploadedFile[])fieldContext.getParameter().getObject();
				if (uploadedFiles.length > 0) {
					fileValue = (File)(uploadedFiles[0]).getContent();
				}
			}
			fieldContext.getFieldUsage().getField().set(interceptorContext.getForm(), fileValue);
		}
	}

	/**
     * <P>Sets, adjusts, converts and validates a form field from a request parameter according to its conversion mode.
     * Not called for set only conversion mode.</P> 
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 2 and lower.</P>
     */
    protected <T> void processFormField() throws IllegalArgumentException, IllegalAccessException {
        AnnotationEntries<T> annountationEntries;
        ConversionResult<T> conversionResult;
        FieldContext<T> fieldContext;
        InterceptorContext interceptorContext;
        ValidationResult validationResult;
		String formattedValue;
		boolean rejected;
		
		interceptorContext = InterceptorContext.getInstance();
		fieldContext = FieldContext.getInstance();
		if (fieldContext.getParameter() != null && fieldContext.getParameter().getValue() != null) {
			formattedValue = fieldContext.getParameter().getValue();
		} else {
			formattedValue = "";
		}
		
		fieldContext.setFormattedValue(formattedValue);
		switch (fieldContext.getConversionMode()) {
		case AUTO:
		case DEFAULT:
			break;
		case NO_CONVERSION:
			fieldContext.getFieldUsage().getField().set(interceptorContext.getForm(), formattedValue);
			break;
		case PAIR:
			fieldContext.getPairFieldUsage().getFormattedField().set(interceptorContext.getForm(), formattedValue);
			break;
		case FILE:
		case SET_ONLY:
			// Not applicable
			break;
		}
		
		annountationEntries = fieldContext.getAnnountationEntries();
        // Adjusters
		fieldContext.setStep(FieldStep.ADJUSTING);
        for (ConfiguredPolicy<T> configuredPolicy: annountationEntries.getAdjusters()) {
        	fieldContext.setConfiguredPolicy(configuredPolicy);
        	processAdjuster();
        }
        // Non-conversion validators
		fieldContext.setStep(FieldStep.NON_CONVERSION_VALIDATING);
        rejected = false;
        for (ConfiguredPolicy<T> annotationUsageResult: annountationEntries.getValidators()) {
        	fieldContext.setConfiguredPolicy(annotationUsageResult);
        	processNonConversionValidator();
        	validationResult = fieldContext.getValidationResult();
        	if (!validationResult.getSuccess()) {
        		rejected = true;
        	}
            if (rejected && validationResult.getShortCircuit()) {
                break;
            }
        }
        
        if (fieldContext.getConversionMode() == ConversionMode.NO_CONVERSION) {
        	return;
        }
        if (rejected) {
	    	processNonConversionRejection();
        	return;
        }
        if (annountationEntries.getConverter() != null) {
        	// Conversion (single value)
    		fieldContext.setStep(FieldStep.CONVERTING);
        	fieldContext.setConfiguredPolicy(annountationEntries.getConverter());
        	processConverter();
            conversionResult = fieldContext.getConversionResult();
            rejected = !conversionResult.getSuccess();
        } else if (annountationEntries.getCollectionConverter() != null) {
        	// Conversion (collection)
    		fieldContext.setStep(FieldStep.CONVERTING);
        	fieldContext.setConfiguredPolicy(annountationEntries.getCollectionConverter());
        	processCollectionConverter();
            conversionResult = fieldContext.getConversionResult();
            rejected = !conversionResult.getSuccess();
        }
        if (rejected) {
        	return;
        }
        fieldContext.setStep(FieldStep.POST_CONVERSION_ADJUSTING);
        if (annountationEntries.getConverter() != null) {
            // Post-conversion adjusters (single value)
            for (ConfiguredPolicy<T> configuredPolicy: annountationEntries.getPostConversionAdjusters()) {
                fieldContext.setConfiguredPolicy(configuredPolicy);
                processPostConversionAdjuster();
                validationResult = fieldContext.getValidationResult();
            }
        } else if (annountationEntries.getCollectionConverter() != null) {
            // Post-conversion adjusters (collection)
            for (ConfiguredPolicy<T> configuredPolicy: annountationEntries.getCollectionPostConversionAdjusters()) {
                fieldContext.setConfiguredPolicy(configuredPolicy);
                processCollectionPostConversionAdjuster();
            }
        }
		fieldContext.setStep(FieldStep.POST_CONVERSION_VALIDATING);
        if (annountationEntries.getConverter() != null) {
        	// Post-conversion validation (single value)
            for (ConfiguredPolicy<T> configuredPolicy: annountationEntries.getPostConversionValidators()) {
            	fieldContext.setConfiguredPolicy(configuredPolicy);
                processPostConversionValidator();
                validationResult = fieldContext.getValidationResult();
            	if (!validationResult.getSuccess()) {
            		rejected = true;
            	}
                if (rejected && validationResult.getShortCircuit()) {
                    break;
                }
            }
            if (rejected) {
    	    	processPostConversionRejection();
            }
        } else if (annountationEntries.getCollectionConverter() != null) {
        	// Post-conversion validation (collection)
            for (ConfiguredPolicy<T> configuredPolicy: annountationEntries.getCollectionPostConversionValidators()) {
            	fieldContext.setConfiguredPolicy(configuredPolicy);
                processCollectionPostConversionValidator();
                validationResult = fieldContext.getValidationResult();
            	if (!validationResult.getSuccess()) {
            		rejected = true;
            	}
                if (rejected && validationResult.getShortCircuit()) {
                    break;
                }
            }
            if (rejected) {
    	    	processCollectionPostConversionRejection();
            }
        }
	}

    /**
     * <P>Calls form to process request parameters not handled by other form fields and writes parameters that failed as 
     * conversion errors.  Does nothing if the form doesn't implement {@link Form}.</P>
     * 
     * <P>Can access all InterceptorContext properties.</P>
     */
	protected void processManualConversionFormFields() {
		Form manualFormFieldConverter;
		InterceptorContext interceptorContext;
		ManualParameterConversionResult manualParameterConversionResults;
		HttpParameters httpParameters;
    	Map<String, ConversionData> conversionErrors;
    	Map<String,String> unprocessedParameters;
		Object form;
		
		interceptorContext = InterceptorContext.getInstance();
		httpParameters = interceptorContext.getAllowedRequestParameters();
		if (!httpParameters.isEmpty()) {
			form = interceptorContext.getForm();
			if (form instanceof Form) {
				unprocessedParameters = new HashMap<>();
				for (Entry<String, Parameter> entry: httpParameters.entrySet()) {
					unprocessedParameters.put(entry.getKey(), entry.getValue().getValue());
				}
				manualFormFieldConverter = (Form)form;
				manualParameterConversionResults = manualFormFieldConverter.manualParameterConvert(unprocessedParameters, interceptorContext.getValidationAware(),
					interceptorContext.getTextProvider());
				for (String parameter: manualParameterConversionResults.getProcessedParameterNames()) {
					httpParameters.remove(parameter);
				}
				conversionErrors = new HashMap<>();
				for (String parameter: manualParameterConversionResults.getFailedParameterNames()) {
					httpParameters.remove(parameter);
					conversionErrors.put(parameter, new ConversionData(unprocessedParameters.get(parameter), String.class));
				}
				interceptorContext.setAllowedRequestParameters(httpParameters);
				interceptorContext.getInvocation().getInvocationContext().getConversionErrors().putAll(conversionErrors);
			} else {
				ownLogger.debug("Unprocessed request parameters found but form does not support Form" + 
	                "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
	                "  unprocessed parameters=" + httpParameters);
			}
		}
	}
	
    /**
     * <P>Handles aftermath of a form field failing the non-conversion validation step, which is writing a conversion 
     * error for auto and default conversion mode, or leaving the field as is for no conversion and pair conversion 
     * mode.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
	protected <T> void processNonConversionRejection() {
		InterceptorContext interceptorContext;
    	FieldContext<T> fieldContext;
    	Map<String, ConversionData> conversionErrors;
    	
		interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
    	switch (fieldContext.getConversionMode()) {
		case AUTO:
		case DEFAULT:
			conversionErrors = interceptorContext.getInvocation().getInvocationContext().getConversionErrors();
			conversionErrors.put(fieldContext.getAnnotatedFieldName(), new ConversionData(fieldContext.getFormattedValue(), 
					fieldContext.getFieldUsage().getField().getType()));
			break;
		case NO_CONVERSION:
		case PAIR:
			// Formatted value in form field
			break;
		case FILE:
		case SET_ONLY:
			// Not applicable
			break;
    	}
	}

    /**
     * <P>Invokes non-conversion validator, sets FieldContext validationResult property and writes error message if 
     * needed; or sets FieldContext validationResult property to success if the adjusted formatted value is empty string 
     * and validator skips them.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 3 and lower.</P>
     */
	protected <T> void processNonConversionValidator() {
		ConfiguredPolicy<T> nonConversionConfiguredPolicy;
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
        NonConversionValidator<?> nonConversionValidator;
        ValidationResult validationResult;
        String formattedValue;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        nonConversionConfiguredPolicy = fieldContext.getConfiguredPolicy();
        nonConversionValidator = nonConversionConfiguredPolicy.getNonConversionValidator();
        formattedValue = fieldContext.getFormattedValue();
        if (formattedValue.length() > 0 || nonConversionValidator.getProcessNoValue()) {
        	try {
        		validationResult = nonConversionValidator.validate(formattedValue);
	        }
	        catch (Exception e) {
	        	ownLogger.error("Non-conversion validator failed" + 
	                "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
	                "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
	                "  formatted value=" + formattedValue, e);
	            validationResult = ValidationResult.makeFailureWithMessageResult("System failure with validator", MessageType.ERROR);
	        }
	        fieldContext.setValidationResult(validationResult);
	        validationResult.setShortCircuit(nonConversionValidator.getShortCircuit());
	        checkNonConversionMessage();
        } else {
	        fieldContext.setValidationResult(ValidationResult.makeSuccessResult());
        }
    }

    /**
     * <P>Handles aftermath of a form field that failed the post-conversion validation step, which is nothing as error 
     * messages are already written.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 4 and lower.</P>
     */
	protected <T> void processPostConversionRejection() {
		// Does nothing
	}

    /**
     * <P>Invokes post-conversion adjuster, sets field (non-string for pair conversion), and sets FieldContext 
     * parsedValue property; or does nothing if the converted value is null and adjuster skips them.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 5 and lower.</P>
     */
    protected <T> void processPostConversionAdjuster() {
        ConversionResult<T> conversionResult;
        Field recipientField;
        FieldContext<T> fieldContext;
        InterceptorContext interceptorContext;
        PostConversionAdjuster<?,T> postConversionAdjuster;
        T newValue;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        conversionResult = fieldContext.getConversionResult();
        postConversionAdjuster = fieldContext.getConfiguredPolicy().getPostConversionAdjuster();
        if ((conversionResult.getSuccess() && conversionResult.getParsedValue() != null) || postConversionAdjuster.getProcessNoValue()) {
            try {
                newValue = postConversionAdjuster.adjust(fieldContext.getParsedValue());
                fieldContext.setParsedValue(newValue);
                recipientField = getRecipientFieldForConversionMode();
                recipientField.set(interceptorContext.getForm(), newValue);
            }
            catch (Exception e) {
            	ownLogger.error("Post-conversion adjuster failed" + 
                    "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
                    "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
                    "  unformatted value=" + fieldContext.getParsedValue(), e);
            }
        }
    }

    /**
     * <P>Invokes post-conversion validator, sets FieldContext validationResult property and writes error message if 
     * needed; or sets FieldContext validationResult property to success if the converted value is null and validator 
     * skips them.</P>
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 5 and lower.</P>
     */
    protected <T> void processPostConversionValidator() {
    	ConversionResult<T> conversionResult;
    	FieldContext<T> fieldContext;
    	InterceptorContext interceptorContext;
        PostConversionValidator<?,T> postConversionValidator;
        ValidationResult validationResult;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        conversionResult = fieldContext.getConversionResult();
        postConversionValidator = fieldContext.getConfiguredPolicy().getPostConversionValidator();
        if ((conversionResult.getSuccess() && conversionResult.getParsedValue() != null) || postConversionValidator.getProcessNoValue()) {
        	try {
	            validationResult = postConversionValidator.validate(fieldContext.getParsedValue());
	        }
	        catch (Exception e) {
	        	ownLogger.error("Post-conversion validator failed" + 
	                "  Struts action=" + interceptorContext.getAction().getClass() + "  form class=" + interceptorContext.getForm().getClass() + 
	                "  field=" + fieldContext.getAnnotatedFieldName() + "  annotation=" + fieldContext.getConfiguredPolicy().getAnnotation().annotationType() +
	                "  unformatted value=" + fieldContext.getParsedValue(), e);
	            validationResult = ValidationResult.makeFailureWithMessageResult("System failure with validator", MessageType.ERROR);
	        }
	        fieldContext.setValidationResult(validationResult);
	        validationResult.setShortCircuit(postConversionValidator.getShortCircuit());
	        checkPostConversionMessage();
        } else {
	        fieldContext.setValidationResult(ValidationResult.makeSuccessResult());
        }
    }
    
    
    
	/**
     * <P>Sets string field, string array or string collection form field for set only conversion mode.</P> 
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 2 and lower.</P>
     */
    protected void processSetOnlyFormField() throws IllegalArgumentException, IllegalAccessException {
        FieldContext<?> fieldContext;
        FieldUsage<?> fieldUsage;
        InterceptorContext interceptorContext;
        Collection<String> formattedCollectionValue;
		String[] formattedArrayValue;
		String formattedValue;
		
		interceptorContext = InterceptorContext.getInstance();
		fieldContext = FieldContext.getInstance();
		fieldUsage = fieldContext.getFieldUsage();
		
		if (fieldUsage.getArray()) {
			if (fieldContext.getParameter() != null && fieldContext.getParameter().getMultipleValues() != null) {
				formattedArrayValue = fieldContext.getParameter().getMultipleValues();
			} else {
				formattedArrayValue = new String[] {};
			}
			fieldContext.getFieldUsage().getField().set(interceptorContext.getForm(), formattedArrayValue);
		} else if (fieldUsage.getCollection()) {
			formattedCollectionValue = makeCollectionForRecipient(fieldUsage.getField().getType());
			if (fieldContext.getParameter() != null && fieldContext.getParameter().getMultipleValues() != null) {
				for (String formattedValue2: fieldContext.getParameter().getMultipleValues()) {
					formattedCollectionValue.add(formattedValue2);
				}
			}
			fieldContext.getFieldUsage().getField().set(interceptorContext.getForm(), formattedCollectionValue);
		} else {
			if (fieldContext.getParameter() != null && fieldContext.getParameter().getValue() != null) {
				formattedValue = fieldContext.getParameter().getValue();
			} else {
				formattedValue = "";
			}
			fieldContext.getFieldUsage().getField().set(interceptorContext.getForm(), formattedValue);
		}
	}

	/**
	 * Called when algorithm is about to start for a field in earnest and sets FieldContext property set 2. 
	 */
	protected <T> void startFieldContext(ConversionMode conversionMode, Parameter parameter, FieldUsage<T> fieldUsage, PairFieldUsage<T> pairFieldUsage) {
		FieldContext<T> fieldContext;
		
		fieldContext = FieldContext.getInstance();
		fieldContext.setConversionMode(conversionMode);
		fieldContext.setParameter(parameter);
		fieldContext.setFieldUsage(fieldUsage);
		fieldContext.setPairFieldUsage(pairFieldUsage);
		switch (conversionMode) {
		case AUTO:
		case DEFAULT:
		case FILE:
		case NO_CONVERSION:
		case SET_ONLY:
			fieldContext.setAnnountationEntries(fieldUsage.getAnnountationEntries());
			fieldContext.setAnnotatedFieldName(fieldUsage.getName());
			fieldUsage.getField().setAccessible(true);
			break;
		case PAIR:
			fieldContext.setAnnountationEntries(pairFieldUsage.getAnnountationEntries());
			fieldContext.setAnnotatedFieldName(pairFieldUsage.getFormattedName());
			pairFieldUsage.getUnformattedField().setAccessible(true);
			pairFieldUsage.getFormattedField().setAccessible(true);
			break;
		}
	}

	/**
	 * Called when algorithm is about to start in earnest and sets InterceptorContext property set 2. 
	 */
    protected void startInterceptorContext(ActionInvocation invocation) {
    	InterceptorContext interceptorContext;
    	Logger validationLogger;
    	Object action;
    	
    	interceptorContext = InterceptorContext.getInstance();
    	interceptorContext.setInvocation(invocation);
        action = invocation.getAction();
        interceptorContext.setAction(action);
        validationLogger = getValidationLogger(invocation, action);
        interceptorContext.setValidationLogger(validationLogger);
        if (action instanceof TextProvider) {
        	interceptorContext.setTextProvider((TextProvider)action);
        } else {
        	interceptorContext.setTextProvider(null);
        }
        if (action instanceof ValidationAware) {
        	interceptorContext.setValidationAware((ValidationAware)action);
        } else {
        	interceptorContext.setValidationAware(null);
        }
        interceptorContext.setModelDriven(action instanceof ModelDriven);
        if (interceptorContext.getModelDriven()) {
        	interceptorContext.setForm(((ModelDriven<?>)action).getModel());
        } else {
        	interceptorContext.setForm(action);
        }
    }


	/**
     * <P>Sets interceptor context's request parameters as a mutable copy.</P>
     * 
     * <P>Can access InterceptorContext property set 2 and lower.</P>
     */
    protected void updateRequestParameters() {
    	InterceptorContext interceptorContext;
    	ActionInvocation invocation;
    	
    	interceptorContext = InterceptorContext.getInstance();
    	invocation = interceptorContext.getInvocation();
    	if (invocation.getInvocationContext().getParameters() != null) {
    		interceptorContext.setAllowedRequestParameters(HttpParameters.create(invocation.getInvocationContext().getParameters()).build());
    	} else {
    		interceptorContext.setAllowedRequestParameters(HttpParameters.create().build());
    	}
	}

    /**
     * Initialisation that needs to be done before main logic of validate is run.
     */
    protected void validateInit() {
        // Not set at construction so client can set different PolicyLookup and avoid DefaultPolicyLookup being created.
        if (policyLookup == null) {
        	policyLookup = DefaultPolicyLookup.getInstance();
        }
    }

	/**
     * Writes message to Action's field errors for field.  ValidationAware can be null if Action does not implement it.  
     */
    protected void writeFieldError(ValidationAware validationAware, String fieldName, String message) {
        if (validationAware != null) {
            validationAware.addFieldError(fieldName, message);
        }
    }

    /**
     * Writes message to Action's action errors.  ValidationAware can be null if Action does not implement it.
     */
    protected void writeGeneralError(ValidationAware validationAware, String message) {
        if (validationAware != null) {
            validationAware.addActionError(message);
        }
    }

    /**
     * Writes message to Action's action messages.  ValidationAware can be null if Action does not implement it.
     */
    protected void writeInfoMessage(ValidationAware validationAware, String message) {
        if (validationAware != null) {
            validationAware.addActionMessage(message);
        }
    }
    
    /**
     * Writes message to Action's action warnings.  ValidationAware can be null if Action does not implement it.
     */
    protected void writeWarningMessage(ValidationAware validationAware, String message) {
        if (validationAware != null && validationAware instanceof ValidationAware2) {
            ((ValidationAware2)validationAware).addActionWarning(message);
        } else if (validationAware != null) {
            validationAware.addActionMessage(message);
        }
    }


    /**
     * Returns default target for writing or logging conversion and validation failures.  Cannot return DEFAULT or null. 
     */
    public MessageType getDefaultMessageType() {
		return defaultMessageType;
	}
	public void setDefaultMessageType(MessageType defaultMessageType) {
		if (defaultMessageType != null && defaultMessageType != MessageType.DEFAULT) {
			this.defaultMessageType = defaultMessageType;
		}
	}


	/**
     * Returns whether form fields with no recognised annotations are ignored.. 
     */
    public boolean getIgnoreNonAnnotatedFields() {
		return ignoreNonAnnotatedFields;
	}
	public void setIgnoreNonAnnotatedFields(boolean ignoreNonAnnotatedFields) {
		this.ignoreNonAnnotatedFields = ignoreNonAnnotatedFields;
	}


	/**
     * Returns maximum length of parameters that can be accepted.  Defaults to 100. 
     */
	public int getParamNameMaxLength() {
		return paramNameMaxLength;
	}
	public void setParamNameMaxLength(int paramNameMaxLength) {
		this.paramNameMaxLength = paramNameMaxLength;
	}

	/**
	 * Returns policy lookup by annotation helper used in validation. 
	 */
	public PolicyLookup getPolicyLookup() {
		return policyLookup;
	}
	public void setPolicyLookup(PolicyLookup policyLookup) {
		this.policyLookup = policyLookup;
	}


	/**
	 * Entry point called by {@link AnnotationValidationInterceptor2}.
	 */
	@Override
	public void validate() {
    	FieldContext<?> fieldContext;
    	InterceptorContext interceptorContext;
    	Parameter parameter;
        Collection<Field> allFields, formFields;
        HttpParameters allowedRequestParameters;
        Object form;

        validateInit();
    	interceptorContext = makeInterceptorContext();
    	try {
        	InterceptorContext.setInstance(interceptorContext);
        	interceptorContext.setStep(InterceptorStep.START);
        	startInterceptorContext(ActionContext.getContext().getActionInvocation());
        	form = interceptorContext.getForm();
        	interceptorContext.setStep(InterceptorStep.PARAMETER_FILTERING);
	        updateRequestParameters();
	        filterRequestParameters();
	        allowedRequestParameters = interceptorContext.getAllowedRequestParameters();
        	interceptorContext.setStep(InterceptorStep.ANNOTATION_PARSING);
        	allFields = getProperties(form.getClass());
        	formFields = getFormFields(allFields);
        	formFields = filterFormFieldsPreCategorise(formFields);
	        categoriseFields(formFields);
        	filterFormFieldsPostCategorise();
	        
        	interceptorContext.setStep(InterceptorStep.FIELD_PROCESSING);
	        for (FieldUsage<?> fieldUsage: interceptorContext.getAutoParameterConversionFields()) {
	        	parameter = allowedRequestParameters.get(fieldUsage.getName());
	        	allowedRequestParameters.remove(fieldUsage.getName());
	        	interceptorContext.setAllowedRequestParameters(allowedRequestParameters);
	        	fieldContext = null;
	        	try {
	        		fieldContext = makeFieldContext();
	        		FieldContext.setInstance(fieldContext);
	        		fieldContext.setStep(FieldStep.START);
	        		startFieldContext(ConversionMode.AUTO, parameter, fieldUsage, null);
	        		processFormField();
	        	}
	        	catch (Exception e) {
	        		ownLogger.error("Processing auto conversion field failed  Struts action=" + ActionContext.getContext().getActionName() + 
	                    "  form=" + form.getClass() + "  field=" + fieldUsage.getName() + "  parameter=" + parameter, e);
	        	}
	        	finally {
	        		if (fieldContext != null) {
		        		fieldContext.setStep(FieldStep.END);
		        		endFieldContext();
	        		}
	        	}
            }
	        for (FieldUsage<?> fieldUsage: interceptorContext.getDefaultParameterConversionFields()) {
	        	parameter = allowedRequestParameters.get(fieldUsage.getName());
	        	allowedRequestParameters.remove(fieldUsage.getName());
	        	interceptorContext.setAllowedRequestParameters(allowedRequestParameters);
	        	try {
	        		fieldContext = makeFieldContext();
	        		FieldContext.setInstance(fieldContext);
	        		startFieldContext(ConversionMode.DEFAULT, parameter, fieldUsage, null);
	        		processFormField();
	        	}
	        	catch (Exception e) {
	        		ownLogger.error("Processing no conversion field failed  Struts action=" + ActionContext.getContext().getActionName() + 
	                    "  form=" + form.getClass() + "  field=" + fieldUsage.getName() + "  parameter=" + parameter, e);
	        	}
	        	finally {
	        		endFieldContext();
	        	}
            }
	        for (FieldUsage<?> fieldUsage: interceptorContext.getFileFields()) {
	        	parameter = allowedRequestParameters.get(fieldUsage.getName());
	        	allowedRequestParameters.remove(fieldUsage.getName());
	        	interceptorContext.setAllowedRequestParameters(allowedRequestParameters);
	        	try {
	        		fieldContext = makeFieldContext();
	        		FieldContext.setInstance(fieldContext);
	        		startFieldContext(ConversionMode.FILE, parameter, fieldUsage, null);
	        		processFileFormField();
	        	}
	        	catch (Exception e) {
	        		ownLogger.error("Processing file field failed  Struts action=" + ActionContext.getContext().getActionName() + 
	                    "  form=" + form.getClass() + "  field=" + fieldUsage.getName() + "  parameter=" + parameter, e);
	        	}
	        	finally {
	        		endFieldContext();
	        	}
            }
	        for (FieldUsage<?> fieldUsage: interceptorContext.getNoConversionFields()) {
	        	parameter = allowedRequestParameters.get(fieldUsage.getName());
	        	allowedRequestParameters.remove(fieldUsage.getName());
	        	interceptorContext.setAllowedRequestParameters(allowedRequestParameters);
	        	try {
	        		fieldContext = makeFieldContext();
	        		FieldContext.setInstance(fieldContext);
	        		startFieldContext(ConversionMode.NO_CONVERSION, parameter, fieldUsage, null);
	        		processFormField();
	        	}
	        	catch (Exception e) {
	        		ownLogger.error("Processing no conversion field failed  Struts action=" + ActionContext.getContext().getActionName() + 
	                    "  form=" + form.getClass() + "  field=" + fieldUsage.getName() + "  parameter=" + parameter, e);
	        	}
            }
	        for (PairFieldUsage<?> pairFieldUsage: interceptorContext.getPairConversionFields()) {
	        	parameter = allowedRequestParameters.get(pairFieldUsage.getFormattedName());
	        	allowedRequestParameters.remove(pairFieldUsage.getFormattedName());
	        	interceptorContext.setAllowedRequestParameters(allowedRequestParameters);
	        	try {
	        		fieldContext = makeFieldContext();
	        		FieldContext.setInstance(fieldContext);
	        		startFieldContext(ConversionMode.PAIR, parameter, null, pairFieldUsage);
	        		processFormField();
	        	}
	        	catch (Exception e) {
	        		ownLogger.error("Processing pair conversion field failed  Struts action=" + ActionContext.getContext().getActionName() + 
	                    "  form=" + form.getClass() + "  formatted field=" + pairFieldUsage.getFormattedName() + 
	                    "  unformatted field=" + pairFieldUsage.getUnformattedName() + "  parameter=" + parameter, e);
	        	}
	        	finally {
	        		endFieldContext();
	        	}
            }
	        for (FieldUsage<?> fieldUsage: interceptorContext.getSetOnlyFields()) {
	        	parameter = allowedRequestParameters.get(fieldUsage.getName());
	        	allowedRequestParameters.remove(fieldUsage.getName());
	        	interceptorContext.setAllowedRequestParameters(allowedRequestParameters);
	        	try {
	        		fieldContext = makeFieldContext();
	        		FieldContext.setInstance(fieldContext);
	        		startFieldContext(ConversionMode.SET_ONLY, parameter, fieldUsage, null);
	        		processSetOnlyFormField();
	        	}
	        	catch (Exception e) {
	        		ownLogger.error("Processing set only field failed  Struts action=" + ActionContext.getContext().getActionName() + 
	                    "  form=" + form.getClass() + "  field=" + fieldUsage.getName() + "  parameter=" + parameter, e);
	        	}
            }
        	interceptorContext.setStep(InterceptorStep.MANUAL_VALIDATION);
        	try {
        		processManualConversionFormFields();
        	}
        	catch (Exception e) {
        		ownLogger.error("Processing manual conversions failed  Struts action=" + ActionContext.getContext().getActionName() + 
                    "  form=" + form.getClass() + "  parameters=" + allowedRequestParameters, e);
        	}
    	}
    	finally {
        	interceptorContext.setStep(InterceptorStep.END);
    		endInterceptorContext();
    	}
    }

}
