package name.matthewgreet.strutscommons.policy;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import name.matthewgreet.strutscommons.annotation.DateConversion;
import name.matthewgreet.strutscommons.annotation.DateConversion.Style;
import name.matthewgreet.strutscommons.annotation.DateConversion.TimeStyle;
import name.matthewgreet.strutscommons.annotation.Required.MessageType;
import com.opensymphony.xwork2.ActionContext;

@SuppressWarnings("deprecation")
public class DateConverter extends AbstractDefaultConverterSupport<DateConversion,Date> {
    private static final Logger LOG = LogManager.getLogger(DateConverter.class);
    
    @DateConversion
    private static boolean annotationPlaceholder;

	@Override
	protected DateConversion makeDefaultAnnotation() {
		try {
			return DateConverter.class.getDeclaredField("annotationPlaceholder").getAnnotation(DateConversion.class);
		}
    	catch (Exception e) {
			LOG.error("Creation of default annotation failed", e);
			return null;
		}
	}

    public static int getDateFormatStyle(Style style) {
        switch (style) {
        case FULL:      return DateFormat.FULL;
        case LONG:      return DateFormat.LONG;
        case MEDIUM:    return DateFormat.MEDIUM;
        case SHORT:     return DateFormat.SHORT;
        }
        return DateFormat.SHORT;
    }
    
    public static int getTimeFormatStyle(TimeStyle style) {
        switch (style) {
        case FULL:      return DateFormat.FULL;
        case LONG:      return DateFormat.LONG;
        case MEDIUM:    return DateFormat.MEDIUM;
        case SHORT:     return DateFormat.SHORT;
		case NONE:		return DateFormat.SHORT;
        }
        return DateFormat.SHORT;
    }
    

    protected TimeZone getTimeZone() {
        TimeZone result;
        
        result = null;
        if (getAnnotation().timezone().length() > 0) {
            result = TimeZone.getTimeZone(getAnnotation().timezone());
        } else if (getAnnotation().actionContextTimeZone().length() > 0) {
            try {
                result = (TimeZone)ActionContext.getContext().get(getAnnotation().actionContextTimeZone());
            }
            catch (ClassCastException e) {
                LOG.warn("DateConversion actionContextTimeZone parameter refers to non-TimeZone" + 
                    "  action=" + ActionContext.getContext().getActionName() + "  actionContextTimeZone=" + getAnnotation().actionContextTimeZone());
            }
        }
        if (result == null) {
            result = TimeZone.getDefault();
        }
        return result;
    }
    
    /**
     * Can return date or datetime format. 
     */
    protected DateFormat makeDateFormat() {
        DateFormat result;
        Locale locale;
        TimeZone timeZone;
        
        if (getAnnotation().format().length() > 0) {
            result = new SimpleDateFormat(getAnnotation().format());
        } else {
        	if (getAnnotation().localeCountry().length() > 0 && getAnnotation().localeLanguage().length() > 0) {
	            locale = new Locale(getAnnotation().localeLanguage(), getAnnotation().localeCountry());
	        } else {
	            locale = ActionContext.getContext().getLocale();
	        }
        	if (getAnnotation().timeStyle() != TimeStyle.NONE) {
        		result = DateFormat.getDateTimeInstance(getDateFormatStyle(getAnnotation().style()), getTimeFormatStyle(getAnnotation().timeStyle()), locale);
        	} else {
        		result = DateFormat.getDateInstance(getDateFormatStyle(getAnnotation().style()), locale);
        	}
        }
        result.setLenient(false);
        timeZone = getTimeZone();
        result.setTimeZone(timeZone);
        return result;
    }

    @Override
    public String format(Date unformattedValue) throws Exception {
        DateFormat formatter;
        
        formatter = makeDateFormat();
        return formatter.format(unformattedValue);
    }

    @Override
    public Class<Date> getRecipientClass() {
        return Date.class;
    }

    @Override
    public MessageType getMessageType() {
        return getAnnotation().messageType();
    }

    @Override
    public String getMessage() {
        return getAnnotation().message();
    }

    @Override
    public String getMessageKey() {
        return getAnnotation().messageKey();
    }

    @Override
    public String getRecipientFieldName() {
        return getAnnotation().parsedFieldName();
    }
    
    @Override
    public ConversionResult<Date> convert(String formValue, Class<? extends Date> recipientClass) throws Exception {
        Date parsedValue;
        DateFormat formatter;
        
        try {
            formatter = makeDateFormat();
            parsedValue = formatter.parse(formValue);
            return ConversionResult.makeSuccessResult(parsedValue);
        }
        catch (ParseException e) {
            return ConversionResult.makeFailureResult();
        }
    }

	@Override
	public boolean getProcessNoValue() {
        return false;
	}

}
