package name.matthewgreet.strutscommons.policy;

import java.util.Collection;

import name.matthewgreet.strutscommons.annotation.Required.MessageType;

/**
 * Describes result of a conversion.  Clients should use static methods starting with 'make'.
 */
public class ConversionResult<T> {
    /**
     * Returns result for when field value is an empty string, skipping collection-based conversion.    
     */
    public static <T> ConversionResult<T> makeSkippedCollectionResult(Collection<T> emptyCollection) {
        ConversionResult<T> result;
        
        result = new ConversionResult<T>();
        result.setSuccess(true);
        result.setParsedCollectionValue(emptyCollection);
        return result;
    }
    
    /**
     * Returns result for when field value is an empty string, skipping single value conversion.    
     */
    public static <T> ConversionResult<T> makeSkippedResult() {
        ConversionResult<T> result;
        
        result = new ConversionResult<T>();
        result.setSuccess(true);
        return result;
    }
    
    /**
	 * Returns result for successful collection-based conversion. 
     */
    public static <T> ConversionResult<T> makeSuccessCollectionResult(Collection<T> parsedValue) {
        ConversionResult<T> result;
        
        result = new ConversionResult<T>();
        result.setSuccess(true);
        result.setParsedCollectionValue(parsedValue);
        return result;
    }
    
	/**
	 * Returns result for successful single value conversion. 
	 */
    public static <T> ConversionResult<T> makeSuccessResult(T parsedValue) {
        ConversionResult<T> result;
        
        result = new ConversionResult<T>();
        result.setSuccess(true);
        result.setParsedValue(parsedValue);
        return result;
    }
    
    /**
     * Returns result for failed conversion and using annotation message configuration.  
     */
    public static <T> ConversionResult<T> makeFailureResult() {
        ConversionResult<T> result;
        
        result = new ConversionResult<T>();
        result.setSuccess(false);
        return result;
    }

    /**
     * Returns result for failed conversion and using supplied message. 
     */
    public static <T> ConversionResult<T> makeFailureWithMessageResult(String message, MessageType messageType) {
        ConversionResult<T> result;
        
        result = new ConversionResult<T>();
        result.setSuccess(false);
        result.setMessage(message);
        result.setMessageType(messageType);
        return result;
    }
    
    /**
     * Returns result for failed conversion and using supplied message key. 
     */
    public static <T> ConversionResult<T> makeFailureWithMessageKeyResult(String messageKey, MessageType messageType) {
        ConversionResult<T> result;
        
        result = new ConversionResult<T>();
        result.setSuccess(false);
        result.setMessageKey(messageKey);
        result.setMessageType(messageType);
        return result;
    }
    
    private boolean success;
    private T parsedValue;
    private Collection<T> parsedCollectionValue;
    public String message;
    public String messageKey;
    public MessageType messageType;
    
    public ConversionResult() {
        super();
        this.success = false;
        this.parsedValue = null;
        this.parsedCollectionValue = null;
        this.message = null;
        this.messageKey = null;
        this.messageType = null;
    }
    
    public ConversionResult(boolean success, T parsedValue, Collection<T> parsedCollectionValue, String message, 
            String messageKey, MessageType messageType) {
        super();
        this.success = success;
        this.parsedValue = parsedValue;
        this.parsedCollectionValue = parsedCollectionValue;
        this.message = message;
        this.messageKey = messageKey;
        this.messageType = messageType;
    }
    
    public boolean getSuccess() {
        return success;
    }
    public void setSuccess(boolean success) {
        this.success = success;
    }

    public T getParsedValue() {
        return parsedValue;
    }
    public void setParsedValue(T parsedValue) {
        this.parsedValue = parsedValue;
    }

    public Collection<T> getParsedCollectionValue() {
        return parsedCollectionValue;
    }
    public void setParsedCollectionValue(Collection<T> parsedCollectionValue) {
        this.parsedCollectionValue = parsedCollectionValue;
    }

    /**
     * If conversion failed, returns error message to use (if messageKey not set), or null to use annotation setting. 
     */
    public String getMessage() {
        return message;
    }
    public void setMessage(String message) {
        this.message = message;
    }
    
    /**
     * If conversion failed, returns error message key to use, or null to use annotation setting. 
     */
    public String getMessageKey() {
        return messageKey;
    }
    public void setMessageKey(String messageKey) {
        this.messageKey = messageKey;
    }
    
    /**
     * If conversion failed, returns message list to use, or null to use annotation setting. 
     */
    public MessageType getMessageType() {
        return messageType;
    }
    public void setMessageType(MessageType messageType) {
        this.messageType = messageType;
    }
    
}