package name.matthewgreet.strutscommons.interceptor;


import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.ValidationAware;

import name.matthewgreet.strutscommons.action.ValidationAware2;

import com.opensymphony.xwork2.interceptor.AbstractInterceptor;


/**
 * <P>If the action implements {@link ValidationAware}, stores the errors and messages in the session so it may be 
 * retrieved by a later action that displays it.  It is usually part of the aflUpdateStack interceptor stack and 
 * retrieved by {@link MessageRetrieveInterceptor} as part of the aflViewStack interceptor stack.</P>
 * 
 * 
 * <P><U>Interceptor parameters</U></P>
 * <DL>
 * <DT>disabled</DT><DD>If true, all processing is disabled.  This is useful for standalone popup windows, especially 
 *                      self-refreshing ones, that never display messages.  Defaults to false.</DD>
 * </DL>
 * 
 * <P><U>Extending the interceptor</U></P>
 * <P>This can't be usefully extended.</P>
 * 
 * <P><U>Example code</U></P>
 * <PRE>
 * &#064;InterceptorRefs({
 *   &#064;InterceptorRef(value="messageStore"),
 *   &#064;InterceptorRef(value="basicStack")
 *})
 * </PRE>
 */
@SuppressWarnings("deprecation")
public class MessageStoreInterceptor extends AbstractInterceptor {
    public static class StoredMessages {
        private Collection<String> actionErrors, actionMessages, actionWarnings;
        private Map<String,List<String>> fieldErrors;
        
        public StoredMessages() {
            actionErrors = new ArrayList<String>();
            actionMessages = new ArrayList<String>();
            actionWarnings = new ArrayList<String>();
            fieldErrors = new HashMap<String,List<String>>();
        }

        public Collection<String> getActionErrors() {
            return actionErrors;
        }
        public void setActionErrors(Collection<String> actionErrors) {
            this.actionErrors = actionErrors;
        }
        
        public Collection<String> getActionMessages() {
            return actionMessages;
        }
        public void setActionMessages(Collection<String> actionMessages) {
            this.actionMessages = actionMessages;
        }
        
        public Collection<String> getActionWarnings() {
            return actionWarnings;
        }
        public void setActionWarnings(Collection<String> actionWarnings) {
            this.actionWarnings = actionWarnings;
        }

        public Map<String,List<String>> getFieldErrors() {
            return fieldErrors;
        }
        public void setFieldErrors(Map<String,List<String>> fieldErrors) {
            this.fieldErrors = fieldErrors;
        }

    }

    private static final long serialVersionUID = -8274766657096416432L;

    public static final String SESSION_STORED_MESSAGES  = MessageStoreInterceptor.class + "_STORED_MESSAGES";
    
    private boolean disabled;

    
    public MessageStoreInterceptor() {
        // Empty
    }

    
    
    @Override
    public String intercept(ActionInvocation invocation) throws Exception {
        ValidationAware validationAware;
        ValidationAware2 aflAware;
        StoredMessages storedMessages;
        String result;
        List<String> fieldErrors;
        Map<String,List<String>> allFieldErrors;
        
        result = invocation.invoke();
        if (!disabled && invocation.getAction() instanceof ValidationAware) {
            validationAware = (ValidationAware)invocation.getAction();
            // Merge with any existing stored messages
            storedMessages = (StoredMessages)ActionContext.getContext().getSession().get(SESSION_STORED_MESSAGES);
            if (storedMessages == null) {
                storedMessages = new StoredMessages();
            }
            storedMessages.getActionErrors().addAll(validationAware.getActionErrors());
            storedMessages.getActionMessages().addAll(validationAware.getActionMessages());
            allFieldErrors = storedMessages.getFieldErrors();
            for (Entry<String,List<String>> entrySet: validationAware.getFieldErrors().entrySet()) {
                fieldErrors = allFieldErrors.get(entrySet.getKey());
                if (fieldErrors == null) {
                    fieldErrors = new ArrayList<String>();
                }
                fieldErrors.addAll(entrySet.getValue());
                allFieldErrors.put(entrySet.getKey(), fieldErrors);
            }
            if (validationAware instanceof ValidationAware2) {
                aflAware = (ValidationAware2)validationAware;
                storedMessages.getActionWarnings().addAll(aflAware.getActionWarnings());
            }
            ActionContext.getContext().getSession().put(SESSION_STORED_MESSAGES, storedMessages);
        }
        return result;
    }

    
	public boolean getDisabled() {
		return disabled;
	}
	public void setDisabled(boolean disabled) {
		this.disabled = disabled;
	}
	
}
