package name.matthewgreet.strutscommons.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import name.matthewgreet.strutscommons.util.StrutsMiscellaneousLibrary;

/**
 * <P>Defines a member variable of a Viewer Action as receiving an injected form processed by a Form Processing Action, 
 * whether successfully or not.  The member variable is only set if the form's type is compatible with its type.</P>
 * 
 * <P>If any <CODE>processors</CODE> are set, the class of the Struts action that processed the form (successfully or 
 * not) must match or be a subclass one of the values.  This is useful if multiple displayed forms happen to be the same
 * type.</P>
 * 
 * <P>By default, the member variable is only set if the form's processor rejected the form.  How <CODE>reception</CODE> 
 * values affect form data reception:</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Form Reception</CAPTION>
 *   <TR CLASS="row_odd">
 *     <TD>NEVER</TD>
 *     <TD>Never receives form.  This option only exists for completeness.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>ERROR</TD>
 *     <TD>Receives form if it was rejected.  Default.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>SUCCESS</TD>
 *     <TD>Receives form if it was rejected.    This option only exists for completeness.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>ALWAYS</TD>
 *     <TD>Always receives form.</TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>Other parameters</P>
 * <TABLE CLASS="main">
 *   <CAPTION>Other Parameters</CAPTION>
 *   <TR CLASS="row_odd">
 *     <TD>disableFormatting</TD>
 *     <TD>If true, disables setting of form fields from their unformatted values (unless 
 *         {@link StrutsMiscellaneousLibrary#formatForm StrutsMiscellaneousLibrary#formatForm} is explicitly called).</TD>
 *   </TR>
 * </TABLE>
 */
@Documented
@Inherited
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface Form {
	/**
	 * Form validation success/fail states accepted by receiving Action.  See {@link Form}.
	 */
    public enum Reception {NEVER, ERROR, SUCCESS, ALWAYS}
    
    public boolean disableFormatting() default false;
    public Class<?>[] processors() default {};
    public Reception reception() default Reception.ERROR;
}
