package name.matthewgreet.strutscommons.action;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import name.matthewgreet.strutscommons.form.Form;
import name.matthewgreet.strutscommons.util.AbstractCompositeCache;
import name.matthewgreet.strutscommons.util.IdListFinder;
import name.matthewgreet.strutscommons.util.ListCache;
import name.matthewgreet.strutscommons.util.ListCache.PaginationMode;
import name.matthewgreet.strutscommons.util.ListCacheRecordComparator;
import name.matthewgreet.strutscommons.util.ListFinder;
import name.matthewgreet.strutscommons.util.ListSizeFinder;
import name.matthewgreet.strutscommons.util.PageByIndexRangeFinder;


/**
 * <P>Template class for creating Command objects that load a list cache in the {@link AbstractCompositeCache} by 
 *    various pagination modes, whether full list, page by id, or page by index.</P>  
 * 
 * <P>Concrete subclasses must implement various template functions:-</P>
 * 
 * <TABLE CLASS="main">
 *   <CAPTION>Required template functions</CAPTION>
 *   <TR CLASS="row_odd">
 *     <TD>{@link #getFindCommand getFindCommand}</TD>
 *     <TD>Returns a Command object used to load list cache, which can be initialized with an input form received by 
 *         this Struts action.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>{@link #getFindListMultiModeConfig getFindListMultiModeConfig}</TD>
 *     <TD>Returns miscellaneous config, such as whether no records found means an input form rejection.</TD>
 *   </TR>
 *   <TR CLASS="row_odd">
 *     <TD>{@link #getListCache getListCache}</TD>
 *     <TD>Returns list cache that stores the retrieved Value Objects.</TD>
 *   </TR>
 *   <TR CLASS="row_even">
 *     <TD>{@link #getMasterListCache getMasterListCache}</TD>
 *     <TD>If the list cache is a slave list, returns master list cache that controls it, otherwise returns null.</TD>
 *   </TR>
 * </TABLE>
 * 
 * @param <M> Record type of master list cache if loading a slave list or NA otherwise.
 * @param <K> Primary key of list cache.
 * @param <T> Record type of list cache.
 * @param <F> Type of form used by this action or NullForm if forms aren't used.
 */
public abstract class AbstractFindListMultiModeActionSupport<M extends Serializable,K extends Serializable,T extends Serializable,F extends Form> extends AbstractFormDrivenActionSupport<F> {
    private static final long serialVersionUID = 7571697364740160930L;

    /**
     * Configures list to be loaded by Struts, finder action and whether empty lists means search parameters must be 
     * rejected.  Configuration is fixed and does not change during the action's lifetime.
     */
    public static class FindListMultiModeConfig {
        private boolean rejectNoMatch;
        private String rejectMessage;
        
        public FindListMultiModeConfig() {
            rejectNoMatch = false;
            rejectMessage = "No results found";
        }
        
        /**
         * Indicates that an empty list means manually entered search parameters
         * must be rejected. 
         */
        public boolean getRejectNoMatch() {
            return rejectNoMatch;
        }
        public void setRejectNoMatch(boolean value) {
            rejectNoMatch = value;
        }
        
        /**
         * Written by subclasses to return name of message to be used if 
         * {@link #getRejectNoMatch} is true.  Defaults to 'No results found'.
         */
        public String getRejectMessage() {
            return rejectMessage;
        }
        public void setRejectMessage(String value) {
            rejectMessage = value;
        }
        
    }

    public static class FindListMultiModeResponse<M extends Serializable,K extends Serializable,T extends Serializable> {
        /**
         * Returns response for a failure to create a finder Command or list was rejected. 
         */
        public static <M extends Serializable,K extends Serializable,T extends Serializable> FindListMultiModeResponse<M,K,T> makeFailureResponse(String message) {
            FindListMultiModeResponse<M,K,T> result;
            
            result = new FindListMultiModeResponse<M,K,T>();
            result.setRejected(true);
            result.addErrorMessage(message);
            return result;
        }
        
        /**
         * Returns response for a successfully created base record list finder Command. 
         */
        public static <M extends Serializable,K extends Serializable,T extends Serializable> FindListMultiModeResponse<M,K,T> makeSuccessBaseRecordListResponse(ListFinder<M,T> finderCommand) {
            FindListMultiModeResponse<M,K,T> result;
            
            result = new FindListMultiModeResponse<M,K,T>();
            result.setFullList(null);
            result.setFullListFinderCommand(finderCommand);
            result.setPaginationMode(PaginationMode.BASE_RECORD_LIST);
            result.setRejected(false);
            return result;
        }
        
        /**
         * Returns response for a successfully created base record list finder Command with results already found. 
         */
        public static <M extends Serializable,K extends Serializable,T extends Serializable> FindListMultiModeResponse<M,K,T> makeSuccessBaseRecordListWithResultsResponse(ListFinder<M,T> finderCommand, List<T> resultList) {
            FindListMultiModeResponse<M,K,T> result;
            
            result = new FindListMultiModeResponse<M,K,T>();
            result.setFullList(resultList);
            result.setFullListFinderCommand(finderCommand);
            result.setPaginationMode(PaginationMode.BASE_RECORD_LIST);
            result.setRejected(false);
            return result;
        }
        
        /**
         * Returns response for a successfully created full list finder Command. 
         */
        public static <M extends Serializable,K extends Serializable,T extends Serializable> FindListMultiModeResponse<M,K,T> makeSuccessFullListResponse(ListFinder<M,T> finderCommand) {
            FindListMultiModeResponse<M,K,T> result;
            
            result = new FindListMultiModeResponse<M,K,T>();
            result.setFullList(null);
            result.setFullListFinderCommand(finderCommand);
            result.setPaginationMode(PaginationMode.FULL_LIST);
            result.setRejected(false);
            return result;
        }
        
        /**
         * Returns response for a successfully created full list finder Command with results already found. 
         */
        public static <M extends Serializable,K extends Serializable,T extends Serializable> FindListMultiModeResponse<M,K,T> makeSuccessFullListWithResultsResponse(ListFinder<M,T> finderCommand, List<T> resultList) {
            FindListMultiModeResponse<M,K,T> result;
            
            result = new FindListMultiModeResponse<M,K,T>();
            result.setFullList(resultList);
            result.setFullListFinderCommand(finderCommand);
            result.setPaginationMode(PaginationMode.FULL_LIST);
            result.setRejected(false);
            return result;
        }
        
        /**
         * Returns response for a successfully created id list finder Command. 
         */
        public static <M extends Serializable,K extends Serializable,T extends Serializable> FindListMultiModeResponse<M,K,T> makeSuccessIdListResponse(IdListFinder<M,K> finderCommand) {
            FindListMultiModeResponse<M,K,T> result;
            
            result = new FindListMultiModeResponse<M,K,T>();
            result.setIdList(null);
            result.setIdListFinderCommand(finderCommand);
            result.setPaginationMode(PaginationMode.PAGE_BY_IDS);
            result.setRejected(false);
            return result;
        }
        
        /**
         * Returns response for a successfully created id list finder Command with results already found. 
         */
        public static <M extends Serializable,K extends Serializable,T extends Serializable> FindListMultiModeResponse<M,K,T> makeSuccessIdListWithResultsResponse(IdListFinder<M,K> finderCommand, List<K> idList) {
            FindListMultiModeResponse<M,K,T> result;
            
            result = new FindListMultiModeResponse<M,K,T>();
            result.setIdList(idList);
            result.setIdListFinderCommand(finderCommand);
            result.setPaginationMode(PaginationMode.PAGE_BY_IDS);
            result.setRejected(false);
            return result;
        }
        
        /**
         * Returns response for a successfully created index list finder Command and page by index Command. 
         */
        public static <M extends Serializable,K extends Serializable,T extends Serializable> FindListMultiModeResponse<M,K,T> makeSuccessIndexListResponse(ListSizeFinder<M> listSizeFinderCommand, PageByIndexRangeFinder<M,T> pageByIndexFinderCommand) {
            FindListMultiModeResponse<M,K,T> result;
            
            result = new FindListMultiModeResponse<M,K,T>();
            result.setListSize(null);
            result.setListSizeFinderCommand(listSizeFinderCommand);
            result.setPageByIndexRangeFinder(pageByIndexFinderCommand);
            result.setPaginationMode(PaginationMode.PAGE_BY_INDEX_RANGE);
            result.setRejected(false);
            return result;
        }
        
        /**
         * Returns response for a successfully created index list finder Command and page by index Command. 
         */
        public static <M extends Serializable,K extends Serializable,T extends Serializable> FindListMultiModeResponse<M,K,T> makeSuccessIndexListWithResultsResponse(ListSizeFinder<M> listSizeFinderCommand, PageByIndexRangeFinder<M,T> pageByIndexFinderCommand, int listSize) {
            FindListMultiModeResponse<M,K,T> result;
            
            result = new FindListMultiModeResponse<M,K,T>();
            result.setListSize(listSize);
            result.setListSizeFinderCommand(listSizeFinderCommand);
            result.setPageByIndexRangeFinder(pageByIndexFinderCommand);
            result.setPaginationMode(PaginationMode.PAGE_BY_INDEX_RANGE);
            result.setRejected(false);
            return result;
        }
        
        
        private boolean rejected = true;
        private String failureResultName = "input";
        private String successResultName = "success";
        private String successSingleForwardName = "successsingle";
        private ListFinder<M,T> fullListFinderCommand;
        private List<T> fullList;
        private IdListFinder<M,K> idListFinderCommand;
        private List<K> idList;
        private ListSizeFinder<M> listSizeFinderCommand;
        private PageByIndexRangeFinder<M,T> pageByIndexRangeFinder;
        private Integer listSize;
        private boolean useSuccessSingle = false;
        private PaginationMode paginationMode;
        private Collection<String> errors = new ArrayList<String>();
        private Collection<String> messages = new ArrayList<String>();
        
        /**
         * Constructor for failure to create a finder Command or list rejected.
         */
        public FindListMultiModeResponse() {
            // Empty
        }
        
        /**
         * Indicates creating the finder Command failed or, if it was retrieved, the list was rejected.  If set, 
         * template base class uses action result named by {@link #getFailureResultName}, otherwise uses action result 
         * named by {@link #getSuccessResultName}.  Defaults to true. 
         */
        public boolean getRejected() {
            return rejected;
        }
        public void setRejected(boolean rejected) {
            this.rejected = rejected;
        }
        
        /**
         * Action result name to use if response is failure.  Defaults to "input".
         */
        public String getFailureResultName() {
            return failureResultName;
        }
        public void setFailureResultName(String value) {
            failureResultName = value;
        }

        /**
         * Action result name to use if response is success.  Defaults to "success".
         */
        public String getSuccessResultName() {
            return successResultName;
        }
        public void setSuccessResultName(String value) {
            successResultName = value;
        }

        /**
         * Action result name to use if response is success, useSuccessSingle is true, and search result contains just 
         * one result.
         */
        public String getSuccessSingleForwardName() {
            return successSingleForwardName;
        }
        public void setSuccessSingleForwardName(String successSingleForwardName) {
            this.successSingleForwardName = successSingleForwardName;
        }

        /**
         * Returns full list finder or base record list finder Command object for full list or base record pagination 
         * mode. 
         */
        public ListFinder<M,T> getFullListFinderCommand() {
            return fullListFinderCommand;
        }

        public void setFullListFinderCommand(ListFinder<M, T> fullListFinderCommand) {
            this.fullListFinderCommand = fullListFinderCommand;
        }

        /**
         * Returns full list or base record list that's already been found for full list or base record pagination 
         * mode, or null otherwise.
         */
        public List<T> getFullList() {
            return fullList;
        }
        public void setFullList(List<T> fullList) {
            this.fullList = fullList;
        }

        /**
         * Returns id list finder for page by ids pagination mode. 
         */
        public IdListFinder<M,K> getIdListFinderCommand() {
            return idListFinderCommand;
        }
        public void setIdListFinderCommand(IdListFinder<M,K> idListFinderCommand) {
            this.idListFinderCommand = idListFinderCommand;
        }

        /**
         * Returns id list that's already been found for page by ids pagination mode, or null otherwise. 
         */
        public List<K> getIdList() {
            return idList;
        }
        public void setIdList(List<K> idList) {
            this.idList = idList;
        }

        /**
         * Returns list size finder for page by index pagination mode.  listSizeFinderCommand and pageByIndexRangeFinder
         * are set in pairs.
         */
        public ListSizeFinder<M> getListSizeFinderCommand() {
            return listSizeFinderCommand;
        }
        public void setListSizeFinderCommand(ListSizeFinder<M> listSizeFinderCommand) {
            this.listSizeFinderCommand = listSizeFinderCommand;
        }

        /**
         * Returns page by index finder for page by index pagination mode.  listSizeFinderCommand and 
         * pageByIndexRangeFinder are set in pairs.
         */
        public PageByIndexRangeFinder<M, T> getPageByIndexRangeFinder() {
            return pageByIndexRangeFinder;
        }
        public void setPageByIndexRangeFinder(PageByIndexRangeFinder<M, T> pageByIndexRangeFinder) {
            this.pageByIndexRangeFinder = pageByIndexRangeFinder;
        }

        /**
         * Returns list size that's already been found for page by index pagination mode, or null otherwise.
         */
        public Integer getListSize() {
            return listSize;
        }
        public void setListSize(Integer listSize) {
            this.listSize = listSize;
        }

        /**
         * If true and search result has just one entry, uses successSingleForwardName Action result. 
         */
        public boolean getUseSuccessSingle() {
            return useSuccessSingle;
        }
        public void setUseSuccessSingle(boolean useSuccessSingle) {
            this.useSuccessSingle = useSuccessSingle;
        }
        
        /**
         * Returns pagination mode. 
         */
        public PaginationMode getPaginationMode() {
            return paginationMode;
        }
        public void setPaginationMode(PaginationMode paginationMode) {
            this.paginationMode = paginationMode;
        }


        /**
         * Returns error messages to display. 
         */
        public Collection<String> getErrors() {
            return errors;
        }
        public void setErrors(Collection<String> errors) {
            this.errors = errors;
        }

        /**
         * Returns info messages to display.  
         */
        public Collection<String> getMessages() {
            return messages;
        }
        public void setMessages(Collection<String> messages) {
            this.messages = messages;
        }

        
        /**
         * Convenience function to add error message.
         */
        public void addErrorMessage(String text) {
            errors.add(text);
        }
        
        /**
         * Convenience function to add info message.
         */
        public void addInfoMessage(String text) {
            messages.add(text);
        }

    }

    public AbstractFindListMultiModeActionSupport() {
        super();
    }
    
    /**
     * Overridden by subclasses to return Command object for retrieving list (configured with any search parameters) and 
     * forwarding in the case of success. 
     * @param selectedMaster Selected record of master list or null if loading master list.
     */
    protected abstract FindListMultiModeResponse<M,K,T> getFindCommand(M selectedMaster) throws Exception;

    /**
     * Written by subclasses to configure behaviour of find action. 
     */
    protected abstract FindListMultiModeConfig getFindListMultiModeConfig();
    
    /**
     * Written by subclasses to return item comparator for sorting list after finding it, or null for no post-find 
     * sorting. 
     */
    protected ListCacheRecordComparator<T> getItemSorter() {
        return null;
    }
 
    /**
     * Written by subclasses to return list cache that is being populated. 
     */
    protected abstract ListCache<M,K,T> getListCache();
 
    /**
     * Written by subclasses to return master list cache or null if the list 
     * cache is the master list. 
     */
    protected abstract ListCache<?,?,M> getMasterListCache();
 
    /**
     * May be overridden by subclasses to translate an exception thrown by a list finder Command, id list finder Command 
     * or list size finder Command into message.  May return null to default to a generic error message. 
     */
    protected String translateFinderException(ListFinder<M,T> listFinderCommand, IdListFinder<M,K> idListFinderCommand, 
            ListSizeFinder<M> listSizeFinderCommand, Exception e) {
        return null;
    }

    /**
     * 
     */
    @Override
    public String execute() throws Exception {
        ListCacheRecordComparator<T> itemSorter;
        ListCache<?,?,M> masterListCache;
        ListCache<M,K,T> listCache;
        FindListMultiModeConfig config;
        FindListMultiModeResponse<M,K,T> findResponse;
        ListFinder<M,T> listFinderCommand;         // Also for base record list finder
        IdListFinder<M,K> idListFinderCommand;
        ListSizeFinder<M> listSizeFinderCommand;
        PageByIndexRangeFinder<M,T> pageByIndexRangeFinderCommand;
        PaginationMode paginationMode;
        List<K> ids;
        List<T> list;
        M selectedMaster;
        String errorMessage;
        int listSize;
        boolean matchFound;
        
        config = getFindListMultiModeConfig();
        findResponse = null;
        list = null;
        listFinderCommand = null;
        ids = null;
        idListFinderCommand = null;
        listSize = 0;
        listSizeFinderCommand = null;
        pageByIndexRangeFinderCommand = null;
        errorMessage = null;
        matchFound = false;
        
        masterListCache = getMasterListCache();
        listCache = getListCache();

        try {
            // Get finder commmand for this list
            if (masterListCache != null && masterListCache != listCache) {
                selectedMaster = masterListCache.getSelected();
            } else {
                selectedMaster = null;
            }
            findResponse = getFindCommand(selectedMaster);
            paginationMode = findResponse.getPaginationMode();
            if (!findResponse.getRejected()) {
                // Run finder command
                switch (paginationMode) {
                case BASE_RECORD_LIST:
                    listFinderCommand = findResponse.getFullListFinderCommand();
                    if (findResponse.getFullList() == null) {
                        list = listFinderCommand.getList(selectedMaster);
                        if (list == null) {
                            list = new ArrayList<>();
                        }
                    } else {
                        list = findResponse.getFullList();
                    }
                    matchFound = list.size() > 0;
                    break;
                case FULL_LIST:
                    listFinderCommand = findResponse.getFullListFinderCommand();
                    if (findResponse.getFullList() == null) {
                        list = listFinderCommand.getList(selectedMaster);
                        if (list == null) {
                            list = new ArrayList<>();
                        }
                    } else {
                        list = findResponse.getFullList();
                    }
                    matchFound = list.size() > 0;
                    break;
                case PAGE_BY_IDS:
                    idListFinderCommand = findResponse.getIdListFinderCommand();
                    if (findResponse.getIdList() == null) {
                        ids = idListFinderCommand.getIds(selectedMaster);
                        if (ids == null) {
                            ids = new ArrayList<>();
                        }
                    } else {
                        ids = findResponse.getIdList();
                    }
                    matchFound = ids != null && ids.size() > 0;
                    break;
                case PAGE_BY_INDEX_RANGE:
                    listSizeFinderCommand = findResponse.getListSizeFinderCommand();
                    pageByIndexRangeFinderCommand = findResponse.getPageByIndexRangeFinder();
                    if (findResponse.getListSize() == null) {
                        listSize = listSizeFinderCommand.getSize(selectedMaster);
                    } else {
                        listSize = findResponse.getListSize(); 
                    }
                    matchFound = listSize > 0;
                    break;
                }
                if (!config.getRejectNoMatch() || matchFound) {
                    // Find succeeded.  Store list and finder command in cache and redirect to viewer.
                    switch (paginationMode) {
                    case BASE_RECORD_LIST:
                        listCache.setBaseRecordListAndFinder(list, listFinderCommand);
                        break;
                    case FULL_LIST:
                        listCache.setListAndFinder(list, listFinderCommand);
                        break;
                    case PAGE_BY_IDS:
                        listCache.setIdListAndFinder(ids, idListFinderCommand);
                        break;
                    case PAGE_BY_INDEX_RANGE:
                        listCache.setListSizeAndFinder(listSize, listSizeFinderCommand, pageByIndexRangeFinderCommand);
                        break;
                    }
                    itemSorter = getItemSorter();
                    if (itemSorter != null) {
                        listCache.setItemSorter(itemSorter);
                    }
                } else {
                    // No results and results required - write error message 
                    addActionError(config.getRejectMessage());
                }
            }
        }
        catch (Exception e) {
            // Determine error message and write it
            errorMessage = translateFinderException(listFinderCommand, idListFinderCommand, listSizeFinderCommand, e);
            if (errorMessage == null || errorMessage.length() == 0) {
                errorMessage = "System failure: " + e.getMessage();
            }
            getLogger().error(errorMessage, e);
            addActionError(errorMessage);
        }
        
        if (findResponse != null) {
            for (String message: findResponse.getMessages()) {
                addActionMessage(message);
            }
            for (String errorMessage2: findResponse.getErrors()) {
                addActionError(errorMessage2);
            }
            if (!findResponse.getRejected()) {
                return findResponse.getSuccessResultName();
            } else {
                return findResponse.getFailureResultName();
            }
        } else {    // Shouldn't happen
            return "error";
        }
    }

}
