package name.matthewgreet.strutscommons.action;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;

import name.matthewgreet.strutscommons.form.AbstractForm;
import name.matthewgreet.strutscommons.util.ListCache;
import name.matthewgreet.strutscommons.util.ListSizeFinder;
import name.matthewgreet.strutscommons.util.PageByIndexRangeFinder;

/**
 * <P>Template class for creating Command objects that load an index-based list in the {@link ListCache} used by the 
 * target, viewer action.</P>  
 * 
 * <P>Concrete subclasses must implement {@link #getFindIndexListConfig}, which defines the list being displayed from the 
 * {@link ListCache} and a list index.  Subclasses will override {@link #getFindCommand} to generate a Command object 
 * that implements {@link ListSizeFinder} and {@link PageByIndexRangeFinder}.</P>
 * 
 * @param <M> Record type of master list cache if loading a slave list or NA otherwise.
 * @param <T> Record type of list cache.
 * @param <F> Type of form used by this action or NullForm if forms aren't used.
 */
public abstract class AbstractFindIndexListActionSupport<M extends Serializable,T extends Serializable,F extends AbstractForm> extends AbstractFormDrivenActionSupport<F> {
    private static final long serialVersionUID = -5096119843738470346L;

    /**
     * Configures list to be loaded by Struts, finder action and whether empty 
     * lists means search parameters must be rejected.  Configuration is fixed 
     * and does not change during the action's lifetime.
     */
    public static class FindIndexListConfig {
        private boolean rejectNoMatch;
        private String rejectMessage;
        private String errorForwardName;
        
        public FindIndexListConfig() {
            rejectNoMatch = false;
            rejectMessage = "No results found";
            errorForwardName = "error";
        }
        
        /**
         * Indicates that an empty list means manually entered search parameters
         * must be rejected. 
         */
        public boolean getRejectNoMatch() {
            return rejectNoMatch;
        }
        public void setRejectNoMatch(boolean value) {
            rejectNoMatch = value;
        }
        
        /**
         * Written by subclasses to return message to be used if {@link #getRejectNoMatch} is true.  Defaults to 'No 
         * results found'
         */
        public String getRejectMessage() {
            return rejectMessage;
        }
        public void setRejectMessage(String value) {
            rejectMessage = value;
        }
        
        /**
         * Written by subclasses to return name of Struts forward (usually global forward) to be used for unhandleable 
         * errors.  Defaults to 'error'.
         */
        public String getErrorForwardName() {
            return errorForwardName;
        }
        public void setErrorForwardName(String value) {
            errorForwardName = value;
        }
    }

    public static class FindIndexListResponse<M extends Serializable,T extends Serializable> {
        /**
         * Returns response for a failure to create a finder Command or list size was rejected. 
         */
        public static <M extends Serializable,T extends Serializable> FindIndexListResponse<M,T> makeFailureResponse(String message) {
            FindIndexListResponse<M,T> result;
            
            result = new FindIndexListResponse<M,T>();
            result.addErrorMessage(message);
            return result;
        }
        
        /**
         * Returns response for a successfully created finder Command and template class will retrieve list size. 
         */
        public static <M extends Serializable,T extends Serializable> FindIndexListResponse<M,T> makeSuccessResponse(ListSizeFinder<M> sizeFinderCommand,
                PageByIndexRangeFinder<M,T> pageFinderCommand) {
            FindIndexListResponse<M,T> result;
            
            result = new FindIndexListResponse<M,T>(sizeFinderCommand, pageFinderCommand);
            return result;
        }
        
        /**
         * Returns response for a successfully created finder Command and list size has already been retrieved. 
         */
        public static <M extends Serializable,T extends Serializable> FindIndexListResponse<M,T> makeSuccessWithListSizeResponse(ListSizeFinder<M> sizeFinderCommand,
                PageByIndexRangeFinder<M,T> pageFinderCommand, int listSize) {
            FindIndexListResponse<M,T> result;

            result = new FindIndexListResponse<M,T>(sizeFinderCommand, pageFinderCommand);
            result.setListSize(listSize);
            return result;
        }
        
        private boolean rejected = true;
        private String failureResultName = "input";
        private String successResultName = "success";
        private ListSizeFinder<M> sizeFinderCommand;
        private PageByIndexRangeFinder<M,T> pageFinderCommand;
        private Integer listSize = null;
        private Collection<String> errors = new ArrayList<String>();
        private Collection<String> messages = new ArrayList<String>();
        
        /**
         * Constructor for failure to create a finder Command.
         */
        public FindIndexListResponse() {
            this.rejected = true;
        }
        
        /**
         * Constructor for successfully creating a finder Command.
         * @param sizeFinderCommand Command object for retrieving list size, configured with any search parameters. 
         * @param pageFinderCommand Command object for retrieving page, configured with any search parameters. 
         */
        public FindIndexListResponse(ListSizeFinder<M> sizeFinderCommand, PageByIndexRangeFinder<M,T> pageFinderCommand) {
            this.rejected = false;
            this.sizeFinderCommand = sizeFinderCommand;
            this.pageFinderCommand = pageFinderCommand;
        }
        
        /**
         * Indicates creating the finder Command failed or, if it was retrieved, the list size was rejected.  If set, 
         * template base class uses action result named by {@link #getFailureResultName}, otherwise uses action result 
         * named by {@link #getSuccessResultName}.  Defaults to true. 
         */
        public boolean getRejected() {
            return rejected;
        }
        
        /**
         * Returns Command object for reloading list size with the search parameters. 
         */
        public ListSizeFinder<M> getSizeFinderCommand() {
            return sizeFinderCommand;
        }
        
        /**
         * Returns Command object for reloading list size with the search parameters. 
         */
        public PageByIndexRangeFinder<M,T> getPageFinderCommand() {
            return pageFinderCommand;
        }
        
        /**
         * Action result name to use if response reports failure.  Defaults to "input".
         */
        public String getFailureResultName() {
            return failureResultName;
        }
        public void setFailureResultName(String value) {
            failureResultName = value;
        }

        /**
         * Action result name to use if response reports success.  Defaults to "success".
         */
        public String getSuccessResultName() {
            return successResultName;
        }
        public void setSuccessResultName(String value) {
            successResultName = value;
        }

        /**
         * List size returned by sizeFinderCommand if it was already retrieved, or null if not. 
         */
        public Integer getListSize() {
            return listSize;
        }
        public void setListSize(Integer listSize) {
            this.listSize = listSize;
        }

        /**
         * Struts errors messages to be used, usually for update failure.  An error message does not indicate an error 
         * (use default constructor) and is displayed even for success (though that's unusual). 
         */
        public Collection<String> getErrors() {
            return errors;
        }
        
        /**
         * Struts information messages to be used, usually for update success.  An info message does not indicate a 
         * success (use constructor accepting size and page finders) and is displayed even for failure (though that's 
         * unusual).  
         */
        public Collection<String> getMessages() {
            return messages;
        }
        
        /**
         * Convenience function to add error message.
         */
        public void addErrorMessage(String text) {
            errors.add(text);
        }
        
        /**
         * Convenience function to add info message.
         */
        public void addInfoMessage(String text) {
            messages.add(text);
        }

    }

    public AbstractFindIndexListActionSupport() {
        super();
    }
    
    /**
     * Overridden by subclasses to return Command object for retrieving list 
     * (configured with any search parameters) and forwarding in the case of 
     * success. 
     * @param selectedMaster Selected record of master list or null if loading 
     *     master list.
     */
    protected abstract FindIndexListResponse<M,T> getFindCommand(M selectedMaster) throws Exception;

    /**
     * Written by subclasses to configure behaviour of find action. 
     */
    protected abstract FindIndexListConfig getFindIndexListConfig();
    
    /**
     * Written by subclasses to return list cache that is being populated. 
     */
    protected abstract ListCache<M,?,T> getListCache();
 
    /**
     * Written by subclasses to return master list cache or null if the list 
     * cache is the master list. 
     */
    protected abstract ListCache<?,?,M> getMasterListCache();
 
    /**
     * May be overridden by subclasses to translate an exception thrown by a list size finder Command into a Struts 
     * message, so a message is displayed to the user.  May return null to default to a generic error message. 
     * @param sizeFinderCommand Size list finder Command.
     * @param e Exception thrown by {@link ListSizeFinder#getSize} function.
     */
    protected String translateFinderException(ListSizeFinder<M> sizeFinderCommand, Exception e) {
        return null;
    }

    /**
    *
    */
   @Override
   public String execute() throws Exception {
       ListCache<?,?,M> masterListCache;
       ListCache<M,?,T> listCache;
       FindIndexListConfig config;
       FindIndexListResponse<M,T> findResponse;
       ListSizeFinder<M> sizeFinderCommand;
       M selectedMaster;
       String errorMessage;
       int size;
       
       config = getFindIndexListConfig();
       findResponse = null;
       sizeFinderCommand = null;
       errorMessage = null;
       
       masterListCache = getMasterListCache();
       listCache = getListCache();

       try {
           // Get finder command for this list
           if (masterListCache != null && masterListCache != listCache) {
               selectedMaster = masterListCache.getSelected();
           } else {
               selectedMaster = null;
           }
           findResponse = getFindCommand(selectedMaster);
           if (!findResponse.getRejected()) {
               sizeFinderCommand = findResponse.getSizeFinderCommand();
               // Run finder command if not already done
               if (findResponse.getListSize() != null) {
                   size = findResponse.getListSize();
               } else {
                   size = sizeFinderCommand.getSize(selectedMaster);
               }
               if (!config.getRejectNoMatch() || size > 0) {
                   listCache.setListSizeAndFinder(size, sizeFinderCommand, findResponse.getPageFinderCommand());
               } else {
                   addActionError(config.getRejectMessage());
               }
           }
       }
       catch (Exception e) {
           // Determine error message and write it
           if (sizeFinderCommand != null) {
               errorMessage = translateFinderException(sizeFinderCommand, e);
           }
           if (errorMessage == null || errorMessage.length() == 0) {
               errorMessage = "No results found";
           }
           addActionError(errorMessage);
       }
       
       if (findResponse != null) {
           for (String message: findResponse.getMessages()) {
               addActionMessage(message);
           }
           for (String errorMessage2: findResponse.getErrors()) {
               addActionError(errorMessage2);
           }
           if (!findResponse.getRejected()) {
               return findResponse.getSuccessResultName();
           } else {
               return findResponse.getFailureResultName();
           }
       } else {    // Shouldn't happen
           return "error";
       }
   }

}
