package name.matthewgreet.strutscommons.action;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import name.matthewgreet.strutscommons.form.AbstractForm;
import name.matthewgreet.strutscommons.util.IdListFinder;
import name.matthewgreet.strutscommons.util.ListCache;


/**
 * <P>Template class for creating Command objects that load an id list in the {@link ListCache} used by the target, 
 * viewer action.</P>  
 * 
 * <P>Concrete subclasses must implement {@link #getFindIdListConfig getFindIdListConfig}, which defines the list being 
 *    displayed from the {@link ListCache} and a list index.  Subclasses will override 
 *    {@link #getFindCommand getFindCommand} to generate a Command object that implements {@link IdListFinder}.</P>
 * 
 * @param <M> Record type of master list cache if loading a slave list or NA otherwise.
 * @param <K> Primary key type of list cache.
 * @param <F> Type of form used by this action or NullForm if forms aren't used.
 */
public abstract class AbstractFindIdListActionSupport<M extends Serializable,K extends Serializable,F extends AbstractForm> extends AbstractFormDrivenActionSupport<F> {
    private static final long serialVersionUID = -5096119843738470346L;

    /**
     * Configures list to be loaded by Struts, finder action and whether empty 
     * lists means search parameters must be rejected.  Configuration is fixed 
     * and does not change during the action's lifetime.
     */
    public static class FindIdListConfig {
        private boolean rejectNoMatch;
        private String rejectMessage;
        private String errorForwardName;
        
        public FindIdListConfig() {
            rejectNoMatch = false;
            rejectMessage = "No results found";
            errorForwardName = "error";
        }
        
        /**
         * Indicates that an empty list means manually entered search parameters
         * must be rejected. 
         */
        public boolean getRejectNoMatch() {
            return rejectNoMatch;
        }
        public void setRejectNoMatch(boolean value) {
            rejectNoMatch = value;
        }
        
        /**
         * Written by subclasses to return message to be used if {@link #getRejectNoMatch} is true.  Defaults to 'No 
         * results found'
         */
        public String getRejectMessage() {
            return rejectMessage;
        }
        public void setRejectMessage(String value) {
            rejectMessage = value;
        }
        
        /**
         * Written by subclasses to return name of Struts forward (usually global forward) to be used for unhandleable 
         * errors.  Defaults to 'error'.
         */
        public String getErrorForwardName() {
            return errorForwardName;
        }
        public void setErrorForwardName(String value) {
            errorForwardName = value;
        }
    }

    public static class FindIdListResponse<M extends Serializable,K extends Serializable> {
        /**
         * Returns response for a failure to create a finder Command or list was rejected. 
         */
        public static <M extends Serializable,K extends Serializable> FindIdListResponse<M,K> makeFailureResponse(String message) {
            FindIdListResponse<M,K> result;
            
            result = new FindIdListResponse<M,K>();
            result.addErrorMessage(message);
            return result;
        }
        
        /**
         * Returns response for a successfully created finder Command and template class will retrieve list. 
         */
        public static <M extends Serializable,K extends Serializable> FindIdListResponse<M,K> makeSuccessResponse(IdListFinder<M,K> finderCommand) {
            FindIdListResponse<M,K> result;
            
            result = new FindIdListResponse<M,K>(finderCommand);
            return result;
        }
        
        /**
         * Returns response for a successfully created finder Command and list has already been retrieved. 
         */
        public static <M extends Serializable,K extends Serializable> FindIdListResponse<M,K> makeSuccessWithIdsResponse(IdListFinder<M,K> finderCommand, List<K> ids) {
            FindIdListResponse<M,K> result;

            result = new FindIdListResponse<M,K>(finderCommand);
            result.setIds(ids);
            return result;
        }
        
        private boolean rejected = true;
        private String failureResultName = "input";
        private String successResultName = "success";
        private IdListFinder<M,K> finderCommand;
        private List<K> ids = null;
        private Collection<String> errors = new ArrayList<String>();
        private Collection<String> messages = new ArrayList<String>();
        
        /**
         * Constructor for failure to create a finder Command.
         */
        public FindIdListResponse() {
            this.rejected = true;
        }
        
        /**
         * Constructor for successfully creating a finder Command.
         * @param finderCommand Command object for retrieving list, configured with
         *                   any search parameters. 
         */
        public FindIdListResponse(IdListFinder<M,K> finderCommand) {
            this.rejected = false;
            this.finderCommand = finderCommand;
        }
        
        /**
         * Indicates creating the finder Command failed or, if it was retrieved, the id list was rejected.  If set, 
         * template base class uses action result named by {@link #getFailureResultName}, otherwise uses action result 
         * named by {@link #getSuccessResultName}.  Defaults to true. 
         */
        public boolean getRejected() {
            return rejected;
        }
        
        /**
         * Returns Command object for reloading list with the search parameters. 
         */
        public IdListFinder<M,K> getFinderCommand() {
            return finderCommand;
        }
        
        /**
         * Action result name to use if response reports failure.  Defaults to "input".
         */
        public String getFailureResultName() {
            return failureResultName;
        }
        public void setFailureResultName(String value) {
            failureResultName = value;
        }

        /**
         * Action result name to use if response reports success.  Defaults to "success".
         */
        public String getSuccessResultName() {
            return successResultName;
        }
        public void setSuccessResultName(String value) {
            successResultName = value;
        }

        /**
         * List size returned by sizeFinderCommand if it was already retrieved, or null if not. 
         */
        public List<K> getIds() {
            return ids;
        }
        public void setIds(List<K> ids) {
            this.ids = ids;
        }

        /**
         * Struts errors messages to be used, usually for update failure.  An error message does not indicate an error 
         * (use default constructor) and is displayed even for success (though that's unusual). 
         */
        public Collection<String> getErrors() {
            return errors;
        }
        
        /**
         * Struts information messages to be used, usually for update success.  An info message does not indicate a 
         * success (use constructor accepting id finder) and is displayed even for failure (though that's unusual).  
         */
        public Collection<String> getMessages() {
            return messages;
        }
        
        /**
         * Convenience function to add error message.
         */
        public void addErrorMessage(String text) {
            errors.add(text);
        }
        
        /**
         * Convenience function to add info message.
         */
        public void addInfoMessage(String text) {
            messages.add(text);
        }

    }

    public AbstractFindIdListActionSupport() {
        super();
    }
    
    /**
     * Overridden by subclasses to return Command object for retrieving list 
     * (configured with any search parameters) and forwarding in the case of 
     * success. 
     * @param selectedMaster Selected record of master list or null if loading 
     *     master list.
     */
    protected abstract FindIdListResponse<M,K> getFindCommand(M selectedMaster) throws Exception;

    /**
     * Written by subclasses to configure behaviour of find action. 
     */
    protected abstract FindIdListConfig getFindIdListConfig();
    
    /**
     * Written by subclasses to return list cache that is being populated. 
     */
    protected abstract ListCache<M,K,?> getListCache(HttpSession session);
 
    /**
     * Written by subclasses to return master list cache or null if the list 
     * cache is the master list. 
     */
    protected abstract ListCache<?,?,M> getMasterListCache(HttpSession session);
 
    /**
     * May be overridden by subclasses to translate an exception thrown by a list
     * finder Command into a Struts message key, so a message is displayed to 
     * the user.  May return null to default to a generic error message key. 
     * @param finderCommand List finder Command.
     * @param e Exception thrown by {@link IdListFinder#getIds} function.
     */
    protected String translateFinderException(IdListFinder<M,K> finderCommand, Exception e) {
        return null;
    }

    /**
    *
    */
   @Override
   public String execute() throws Exception {
       ListCache<?,?,M> masterListCache;
       ListCache<M,K,?> listCache;
       FindIdListConfig config;
       FindIdListResponse<M,K> findResponse;
       IdListFinder<M,K> finderCommand;
       HttpServletRequest request;
       HttpSession session;
       List<K> ids;
       M selectedMaster;
       String errorMessage;
       
       config = getFindIdListConfig();
       findResponse = null;
       finderCommand = null;
       errorMessage = null;
       
       request = getServletRequest();
       session = request.getSession();
       masterListCache = getMasterListCache(session);
       listCache = getListCache(session);

       try {
           // Get finder commmand for this list
           session = request.getSession();
           if (masterListCache != null && masterListCache != listCache) {
               selectedMaster = masterListCache.getSelected();
           } else {
               selectedMaster = null;
           }
           findResponse = getFindCommand(selectedMaster);
           if (!findResponse.getRejected()) {
               finderCommand = findResponse.getFinderCommand();
               // Run finder command if not already done
               if (findResponse.getIds() != null) {
                   ids = findResponse.getIds();
               } else {
                   ids = finderCommand.getIds(selectedMaster);
               }
               if (ids == null) {
                   ids = new ArrayList<>();
               }
               if (!config.getRejectNoMatch() || ids.size() > 0) {
                   listCache.setIdListAndFinder(ids, finderCommand);
               } else {
                   addActionError(config.getRejectMessage());
               }
           }
       }
       catch (Exception e) {
           // Determine error message and write it
           if (finderCommand != null) {
               errorMessage = translateFinderException(finderCommand, e);
           }
           if (errorMessage == null || errorMessage.length() == 0) {
               errorMessage = "No results found";
           }
           addActionError(errorMessage);
       }
       
       if (findResponse != null) {
           for (String message: findResponse.getMessages()) {
               addActionMessage(message);
           }
           for (String errorMessage2: findResponse.getErrors()) {
               addActionError(errorMessage2);
           }
           if (!findResponse.getRejected()) {
               return findResponse.getSuccessResultName();
           } else {
               return findResponse.getFailureResultName();
           }
       } else {    // Shouldn't happen
           return "error";
       }
   }

}
