package name.matthewgreet.strutscommons.interceptor;


import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.opensymphony.xwork2.Action;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.ValidationAware;
import com.opensymphony.xwork2.interceptor.AbstractInterceptor;
import com.opensymphony.xwork2.interceptor.PreResultListener;
import com.opensymphony.xwork2.util.CompoundRoot;

import name.matthewgreet.strutscommons.action.ValidationAware2;


/**
 * <P>If the action implements ValidationAware, amalgametes all action errors and messages from other validation aware 
 * actions in the Value Stack.</P>
 *   
 * <P><U>Interceptor parameters</U></P>
 * <P>None</P>
 * 
 * <P><U>Extending the interceptor</U></P>
 * <P>This can't be usefully extended.</P>
 * 
 * <P><U>Example code</U></P>
 * <PRE>
 * &#064;InterceptorRefs({
 *   &#064;InterceptorRef(value="messageAmalgamation"),
 *   &#064;InterceptorRef(value="defaultStack")
 *})
 * </PRE>
 */
@SuppressWarnings("deprecation")
public class MessageAmalgamationInterceptor extends AbstractInterceptor implements PreResultListener {
    private static final long serialVersionUID = -6747732006176118658L;

    private boolean disabled;
    

    private void amalgamateMessages(ValidationAware viewAction, ActionInvocation invocation) {
        ValidationAware2 viewAware, otherAware;
        ValidationAware other;
        CompoundRoot root;
        Collection<String> actionErrors, actionMessages, actionWarnings;
        List<String> fieldErrors;
        Map<String,List<String>> allFieldErrors;
        boolean doWarnings;
        
        doWarnings = viewAction instanceof ValidationAware2; 
        actionErrors = viewAction.getActionErrors();
        actionMessages = viewAction.getActionMessages();
        allFieldErrors = viewAction.getFieldErrors();
        if (doWarnings) {
        	viewAware = (ValidationAware2)viewAction;
            actionWarnings = viewAware.getActionWarnings(); 
        } else {
        	viewAware = null;
            actionWarnings = null; 
        }
        
        root = invocation.getStack().getRoot();
        for (Object o: root) {
            // Copy messages from anything that can hold them except the target itself
            if (o instanceof ValidationAware && (o != viewAction)) {
                other = (ValidationAware)o;
                actionErrors.addAll(other.getActionErrors());
                actionMessages.addAll(other.getActionMessages());
                for (Entry<String,List<String>> entrySet: other.getFieldErrors().entrySet()) {
                    fieldErrors = allFieldErrors.get(entrySet.getKey());
                    if (fieldErrors == null) {
                        fieldErrors = new ArrayList<String>();
                    }
                    fieldErrors.addAll(entrySet.getValue());
                    allFieldErrors.put(entrySet.getKey(), fieldErrors);
                }
                if (doWarnings && other instanceof ValidationAware2) {
                	otherAware = (ValidationAware2)other;
                    actionWarnings.addAll(otherAware.getActionWarnings());
                }
            }
        }
        
        viewAction.setActionErrors(actionErrors);
        viewAction.setActionMessages(actionMessages);
        viewAction.setFieldErrors(allFieldErrors);
    }
    
    @Override
    public void beforeResult(ActionInvocation invocation, java.lang.String resultCode) {
        Action action;

        action = (Action)invocation.getAction();
        if (action instanceof ValidationAware) {
            amalgamateMessages((ValidationAware)action, invocation);
        }
    }

    @Override
    public String intercept(ActionInvocation invocation) throws Exception {
        if (!disabled) {
        	invocation.addPreResultListener(this);
        }
        
        return invocation.invoke();
    }

    public boolean getDisabled() {
        return disabled;
    }
    public void setDisabled(boolean value) {
        disabled = value;
    }
    
}
