package name.matthewgreet.strutscommons.util;

import java.io.Serializable;

import name.matthewgreet.strutscommons.util.ListCache.ListCacheConfig;



/**
 * <P>Template for web level, session-based caching a master, record list for editing in various web pages.  Subclasses 
 *    of this, as well as defining the record types, typically set list finders and other Stategies for record list 
 *    relationships and pagination that clients don't set themselves, depending on web page workflow.  Unlike similar 
 *    versions, there are no slave lists.  Instances are typically created as Singletons.</P>
 *    
 * <DL>
 * <DT>KM</DT><DD>Type primary key of master records</DD>
 * <DT>TM</DT><DD>Type of master records</DD>
 * </DL>
 * 
 * <P>As there are a bewildering set of configuration options and client Struts actions are expected to know their 
 *    responsibilities in workflow, subclasses are a convenient Javadoc reference point for documenting this, such as 
 *    below.</P>
 * 
 * <TABLE>
 *   <CAPTION>Example Javadoc</CAPTION>
 *   <TR>
 *     <TH>Slave list no</TH>
 *     <TH>Name</TH>
 *     <TH>Description</TH>
 *     <TH>Viewer action</TH>
 *     <TH>Base record</TH>
 *     <TH>Other base record data</TH>
 *     <TH>Page extension data</TH>
 *     <TH>Detail data</TH>
 *     <TH>Pagination mode</TH>
 *     <TH>List/id/size finder</TH>
 *     <TH>Page finder/ page extension assembler</TH>
 *     <TH>Single item finder</TH>
 *   </TR>
 *   <TR>
 *     <TD>Master</TD>
 *     <TD>Board games</TD>
 *     <TD>Board games in Board Game Geek database up to 2022</TD>
 *     <TD><CODE>ViewGameListAction</CODE></TD>
 *     <TD><CODE>@link GameDTO</CODE></TD>
 *     <TD>&nbsp;</TD>
 *     <TD>Related designers</TD>
 *     <TD>Related artists, mechanics, and publishers</TD>
 *     <TD STYLE="text-align: center;">Depends on search criteria</TD>
 *     <TD STYLE="text-align: center;">Supplied by FindGameListByDesignerNameAction, FindGameListByFamileAction, FindGameListByNameContainsAction and FindGameListByRankAction</TD>
 *     <TD STYLE="text-align: center;">GamePageByIdsFinder, GamePageExtensionAssembler</TD>
 *     <TD STYLE="text-align: center;">GameItemFinder</TD>
 *   </TR>
 *   <TR>
 *     <TD>1</TD>
 *     <TD>Ratings distribution</TD>
 *     <TD>Counts of each user ratings of 0 to 10</TD>
 *     <TD><CODE>ViewRatingDistributionListAction</CODE></TD>
 *     <TD><CODE>GameRatingDTO</CODE></TD>
 *     <TD>&nbsp;</TD>
 *     <TD>&nbsp;</TD>
 *     <TD>&nbsp;</TD>
 *     <TD STYLE="text-align: center;">Full list</TD>
 *     <TD STYLE="text-align: center;">GameRatingByGameIdListFinder</TD>
 *     <TD STYLE="text-align: center;">N/A</TD>
 *     <TD STYLE="text-align: center;">&nbsp;</TD>
 *   </TR>
 *   <TR>
 *     <TD>2</TD>
 *     <TD>User ratings</TD>
 *     <TD>Individual user ratings</TD>
 *     <TD><CODE>ViewUserRatingListAction</CODE></TD>
 *     <TD>UserRatingDTO</TD>
 *     <TD>&nbsp;</TD>
 *     <TD>Owning username</TD>
 *     <TD>&nbsp;</TD>
 *     <TD STYLE="text-align: center;">Base record list</TD>
 *     <TD STYLE="text-align: center;">Supplied by FindUserRatingListByUsernameContainsAction and FindUserRatingListByRatingAction</TD>
 *     <TD STYLE="text-align: center;">UserRatingPageExtensionAssembler</TD>
 *     <TD STYLE="text-align: center;">&nbsp;</TD>
 *   </TR>
 * </TABLE>
 */
@SuppressWarnings("javadoc")
public abstract class AbstractCompositeCache<KM extends Serializable,TM extends Serializable> implements Serializable {
    private static final long serialVersionUID = -780289718998437697L;
    
    private ListCache<NA,KM,TM> masterList;
    
    /**
     * Creates session-based, composite cache with a master list and no slave lists. 
     */
    protected AbstractCompositeCache() {
        super();
        
        ListCacheConfig<NA,KM,TM> masterListCacheConfig = getMasterListCacheConfig();
        masterList = new ListCache<NA,KM,TM>(masterListCacheConfig, null);
        masterList.addListCacheListener(listCache -> {notifyMasterListCacheChanged(listCache);});
    }
    
    public AbstractCompositeCache(AbstractCompositeCache<KM,TM> abstractCompositeCache) throws Exception {
    	masterList = new ListCache<NA,KM,TM>(abstractCompositeCache.getMasterListCache(), null);
        masterList.addListCacheListener(listCache -> {notifyMasterListCacheChanged(listCache);});
    }
    
    /**
     * Overridden by concrete implementations to configure master list cache. 
     */
    protected abstract ListCacheConfig<NA,KM,TM> getMasterListCacheConfig();
    
    
    /**
     * Notification that the master list cache has changed.
     */
    protected void notifyMasterListCacheChanged(ListCache<?,?,TM> listCache) {
        // Nothing but may be overridden.
    }

    /**
     * Returns cache of master list. 
     */
    public ListCache<NA,KM,TM> getMasterListCache() {
        return masterList;
    }

}
