package name.matthewgreet.strutscommons.action;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import name.matthewgreet.strutscommons.util.ListCache;
import name.matthewgreet.strutscommons.util.ListCacheRecordComparator;
import name.matthewgreet.strutscommons.util.ListFinder;
import name.matthewgreet.strutscommons.view.FormattedListDisplay;

/**
 * <P>Struts 2 Template class for displaying and controlling list data on a JSP from an instance of {@link ListCache}, 
 * which, if it's a slave list, is linked to a master list.  The implementing subclass typically displays the whole list 
 * or a page of it, or displays the currently selected record of the list.  URL parameters can control the list, such as 
 * selecting a different item or making the list re-find its contents according to its current search criteria.  Also 
 * collects popup requests and messages made by previous actions and security data.  </P>
 * 
 * <P>Concrete subclasses must implement various template functions:-</P>
 * 
 * <DL>
 *   <DT>{@link #formatRecord}</DT><DD>Returns formatted record for display, derived from record from list.</DD>
 *   <DT>{@link #getListCache}</DT><DD>Returns list cache that is being displayed.</DD>
 *   <DT>{@link #getMasterListCache}</DT><DD>If the list cache is a slave list, returns master list cache that controls
 *        it, otherwise returns null.</DD>
 *   <DT>{@link #getViewListConfig}</DT><DD>Returns other configuration, such as use of page mode </DD>
 * </DL>
 * 
 * <P>Concrete subclasses may implement other template functions:-</P>
 * 
 * <DL>
 *   <DT>{@link #endRecordFormatting}</DT><DD>Cleanup after {@link #formatRecord}.</DD>
 *   <DT>{@link #getAlternateIndex}</DT><DD>Returns index of list cache item to select (including -1 to select none of 
 *       them) usually according to a request parameter, or null to ignore this.</DD>
 *   <DT>{@link #getItemSorter}</DT><DD>Returns a comparator for sorting the list cache, if requested.  Sorting is only 
 *       allowed if list cache uses full list pagination.</DD>
 *   <DT>{@link #getViewResponse}</DT><DD>Loads any additional data and view helpers, such as list of countries, needed 
 *        by JSP and returns Struts result, usually "success".</DD>
 *   <DT>{@link #startRecordFormatting}</DT><DD>Preparation for {@link #formatRecord}.</DD>
 * </DL>
 * 
 * <P>The URL parameters, whose names can be changed by config, that can be used are:-</P>
 * 
 * <DL>
 *   <DT>ascending</DT><DD>If true (and sorting is allowed), re-sorts list in ascending order, otherwise descending.</DD>
 *   <DT>page</DT><DD>Page in list cache to select (which starts at 1).</DD>
 *   <DT>reload</DT><DD>If present and set to 'true', invalidates list cache, forcing it to reload using its current 
 *                      list finder.</DD>
 *   <DT>select</DT><DD>Index of item in list cache to select (which starts at 0).</DD>
 *   <DT>sort</DT><DD>Name of field to re-sort list, which must be recognised by the implementation of 
 *                    {@link #getItemSorter}.</DD>
 * </DL>
 * 
 * <P>Generic types:-</P>
 * 
 * <DL>
 *   <DT>M</DT><DD>Record type of master list or Object if this displays the master list.</DD>
 *   <DT>K</DT><DD>Primary key type of list cache.</DD>
 *   <DT>T</DT><DD>Record type of list.</DD>
 *   <DT>FT</DT><DD>Type of record that is type T formatted for display.</DD>
 * </DL>
 */
public abstract class AbstractViewListActionSupport<M extends Serializable,K extends Serializable,T extends Serializable,FT> extends AbstractViewActionSupport {
    /**
     * Configures list to be loaded by viewer action.  Configuration is fixed 
     * and does not change during the action's lifetime.
     */
    public static class ViewListConfig {
        /**
         * Returns new config for displaying nothing from the list.  Parameters, such as 'select' are still read and
         * processed.
         */
        public static ViewListConfig makeNoListConfig() {
            ViewListConfig result;
            
            result = new ViewListConfig();
            result.setIgnoreList(true);
            result.setPageMode(false);
            result.setRequireSelectedMasterDetail(false);
            result.setRequireSelectedItemDetail(false);
            return result;
        }
        
        /**
         * Returns new config for actions displaying a full list of base records without any detail.  If detail applies,
         * a different action displays them.
         */
        public static ViewListConfig makeNonPagedListConfig() {
            ViewListConfig result;
            
            result = new ViewListConfig();
            result.setPageMode(false);
            result.setRequireSelectedMasterDetail(true);
            result.setRequireSelectedItemDetail(false);
            return result;
        }
        
        /**
         * Returns new config for displaying lists of base records in pages without any detail.  If detail applies,
         * a different action displays them.
         */
        public static ViewListConfig makePagedListConfig(int pageSize) {
            ViewListConfig result;
            
            result = new ViewListConfig();
            result.setPageMode(true);
            result.setPageSize(pageSize);
            result.setRequireSelectedMasterDetail(true);
            result.setRequireSelectedItemDetail(false);
            return result;
        }
        
        /**
         * Returns new config for displaying a single record with any detail.
         */
        public static ViewListConfig makeSingleRecordConfig() {
            ViewListConfig result;
            
            result = new ViewListConfig();
            result.setPageMode(true);
            result.setPageSize(1);
            result.setRequireSelectedMasterDetail(true);
            result.setRequireSelectedItemDetail(true);
            return result;
        }
        
        private boolean ignoreList;
        private boolean pageMode;
        private int pageSize;
        private boolean reloadDefault;
        private boolean requireSelectedMasterDetail;
        private boolean requireSelectedItemDetail;
        private String errorForwardName;
        private String parameterNameIndexSelect;
        private String parameterNameMasterIndexSelect;
        private String parameterNamePage;
        private String parameterNameReload;
        private String parameterNameSortAscending;
        private String parameterNameSortOption;
        
        /**
         * Constructor for displaying entire list (regardless of cache mode). 
         */
        public ViewListConfig() {
            ignoreList = false;
            pageMode = false;
            pageSize = 5;
            requireSelectedMasterDetail = true;
            requireSelectedItemDetail = true;
            errorForwardName = "error";
            parameterNameIndexSelect = PARAMETER_NAME_INDEXSELECT;
            parameterNameMasterIndexSelect = PARAMETER_NAME_MASTER_INDEXSELECT;
            parameterNamePage = PARAMETER_NAME_PAGE;
            parameterNameReload = PARAMETER_NAME_RELOAD;
            parameterNameSortOption = PARAMETER_NAME_SORT_OPTION;
            parameterNameSortAscending = PARAMETER_NAME_SORT_ASCENDING;
            reloadDefault = false;
        }

        /**
         * Whether no list or page is displayed.
         */
        public boolean getIgnoreList() {
            return ignoreList;
        }
        public void setIgnoreList(boolean ignoreList) {
            this.ignoreList = ignoreList;
        }
        
        /**
         * Whether the list should be displayed in pages, rather than the entire list.  This is not the same as the 
         * cache page.  The view controller can decide to display in pages independently of how the cache can retrieve 
         * them.
         */
        public boolean getPageMode() {
            return pageMode;
        }
        public void setPageMode(boolean value) {
            pageMode = value;
        }
        
        /**
         * Returns size of each page, if in page mode.
         */
        public int getPageSize() {
            return pageSize;
        }
        public void setPageSize(int value) {
            if (value > 0) {
                pageSize = value;
            } else {
                pageSize = 1;
            }
        }
        
        /**
         * Returns whether to reload from DB in the absence of the reload parameter.  
         */
        public boolean getReloadDefault() {
            return reloadDefault;
        }
        public void setReloadDefault(boolean value) {
            reloadDefault = value;
        }

        /**
         * Returns whether currently selected item of master list cache must include details.
         */
        public boolean getRequireSelectedMasterDetail() {
            return requireSelectedMasterDetail;
        }
        public void setRequireSelectedMasterDetail(boolean requireSelectedMasterDetail) {
            this.requireSelectedMasterDetail = requireSelectedMasterDetail;
        }

        /**
         * Returns whether currently selected item of list cache must include details.
         */
        public boolean getRequireSelectedItemDetail() {
            return requireSelectedItemDetail;
        }
        public void setRequireSelectedItemDetail(boolean requireSelectedItemDetail) {
            this.requireSelectedItemDetail = requireSelectedItemDetail;
        }

        /**
         * Name of Struts forward (usually global forward) to be used for unhandlable errors.  Defaults to 'error'.
         */
        public String getErrorForwardName() {
            return errorForwardName;
        }
        public void setErrorForwardName(String value) {
            errorForwardName = value;
        }
        
        /**
         * Returns name of request parameter that specifies selected item by its index number (starting at 0).  Defaults
         * to 'select'.   
         */
        public String getParameterNameIndexSelect() {
            return parameterNameIndexSelect;
        }
        public void setParameterNameIndexSelect(String value) {
            parameterNameIndexSelect = value;
        }

        /**
         * Returns name of request parameter that specifies selected item of master list by its index number (starting 
         * at 0).  Defaults to 'masterSelect'.   
         */
        public String getParameterNameMasterIndexSelect() {
            return parameterNameMasterIndexSelect;
        }
        public void setParameterNameMasterIndexSelect(String value) {
            parameterNameMasterIndexSelect = value;
        }

        /**
         * Returns name of request parameter that specifies page to display (if in page mode), starting at 1.  Defaults
         * to 'page'.
         */
        public String getParameterNamePage() {
            return parameterNamePage;
        }
        public void setParameterNamePage(String value) {
            parameterNamePage = value;
        }
        

        /**
         * Returns name of request parameter that specifies reloading from DB.  Defaults to 'reload'.
         */
        public String getParameterNameReload() {
            return parameterNameReload;
        }
        public void setParameterNameReload(String value) {
            parameterNameReload = value;
        }

        /**
         * Returns name of request parameter that specifies ascending sort order.  Defaults to 'ascending'.
         */
        public String getParameterNameSortAscending() {
            return parameterNameSortAscending;
        }
        public void setParameterNameSortAscending(String parameterNameSortAscending) {
            this.parameterNameSortAscending = parameterNameSortAscending;
        }

        /**
         * Returns name of request parameter that specifies sort option, usually a record field name.  Defaults to 
         * 'sort'.
         */
        public String getParameterNameSortOption() {
            return parameterNameSortOption;
        }
        public void setParameterNameSortOption(String parameterNameSortOption) {
            this.parameterNameSortOption = parameterNameSortOption;
        }

    }
    
    private static final long serialVersionUID = -4166079641325035739L;
    // Name of parameter for choosing record of list to display
    public static final String PARAMETER_NAME_INDEXSELECT = "select";
    
    // Name of parameter for choosing page to display.
    public static final String PARAMETER_NAME_PAGE = "page";   

    
    // Name of parameter for forcing reload of displayed list
    public static final String PARAMETER_NAME_RELOAD = "reload";

    // Value of reload parameter to force reload
    public static final String PARAMETER_VALUE_RELOAD_TRUE  = "true";
    
    // Name of parameter for choosing sort in ascending order, which is expected to be 'true' or 'false'.
    public static final String PARAMETER_NAME_SORT_ASCENDING = "ascending";   

    // Name of parameter for choosing sort option, usually the name of a field.
    public static final String PARAMETER_NAME_SORT_OPTION = "sort";

    // Name of parameter for choosing record of master list to display
    public static final String PARAMETER_NAME_MASTER_INDEXSELECT = "masterSelect";
    
    
    private final ViewListConfig config = getViewListConfig();
    
    private List<T> unformattedList;
    private FormattedListDisplay<FT> formattedList;
    
    private int formatRecordIndex;  // Index of record being formatted, only used in formatRecord()

    public AbstractViewListActionSupport() {
        super();
    }
    
    /**
     * Calculates number of page that contains indexed item. 
     */
    private int calcPageNo(int index) {
        if (config.getPageMode()) {
            return (index / config.getPageSize()) + 1;
        } else {
            return 1;
        }
    }
    
    /**
     * Ensure index of selected item, if there is one, would be shown on 
     * displayed list, changing it if needed.  This only applies when displaying
     * in paged mode.  In non-paged mode, the entire list is the displayed list,
     * so the selected item is always in it. 
     */
    private int checkIndexInPage(int currentIndex, int pageNo) throws Exception {
        int pageSize, workingIndex;
        
        pageSize = config.getPageSize();
        workingIndex = currentIndex;
        if (currentIndex > -1 && config.getPageMode()) {
            if (currentIndex < getStartIndex(pageNo, pageSize) || currentIndex > getEndIndex(pageNo, pageSize)) {
                workingIndex = getStartIndex(pageNo, pageSize);
            }
        }
        return workingIndex;
    }
    
    /**
     * Ensure cache has loaded list or id list.  
     */
    private void checkListOrPage(ListCache<M,?,T> listCache) throws Exception {
        if (config.getPageMode()) {
            listCache.getPage();
        } else {
            listCache.getList();
        }
    }
    
    /**
     * Ensure page number falls over current list, as it may have shrunk.  
     */
    private int checkPageNo(int page, int listSize) throws Exception {
        int result, totalPages;
        
        if (config.getPageMode()) {
            result = page;
            result = (result < 1)?1:result;
            totalPages = getTotalPages(listSize);
            result = (result > totalPages)?totalPages:result;
            result = (result < 1)?1:result;
        } else {
            // Always page 1 in full list mode.
            result = 1;
        }
        return result;
    }

    /**
     * Select record by index according to request (if any).
     */
    private Integer checkPageParameter(HttpServletRequest request) {
        String select;
        
        select = request.getParameter(config.getParameterNamePage());
        if (select != null) {
            try {
                return Integer.parseInt(select);
            }
            catch (NumberFormatException e) {
                getLogger().warn("Value of page parameter is not a number  select=" + select);
                return null;
            }
        }
        return null;
    }

    /**
     * Reload list if requested, or parameter not set and reloads by default.  
     */
    private void checkReload(HttpServletRequest request, ListCache<M,?,T> listCache) {
        String reload;
        
        reload = request.getParameter(config.getParameterNameReload());
        if ((reload != null && reload.equals(PARAMETER_VALUE_RELOAD_TRUE)) || 
              (reload == null && config.getReloadDefault()) ) {
                listCache.markReload();
        }
    }
    
    /**
     * Returns index of last record of displayed page. 
     */
    private int getEndIndex(int page, int pageSize) {
        return page * pageSize - 1;
    }

    /**
     * Parse index of record to select according to request parameters, if any.
     */
    private Integer getIndexSelect() {
        HttpServletRequest request;
        String select;
        
        request = getServletRequest();
        select = request.getParameter(config.getParameterNameIndexSelect());
        if (select != null) {
            try {
                return Integer.parseInt(select);
            }
            catch (NumberFormatException e) {
                // Ignore 
                getLogger().warn("Value of select parameter is not a number  select=" + select);
            }
        }
        return null;
    }
    
    /**
     * Parse index of master record to select according to request parameters, if any.
     */
    private Integer getMasterIndexSelect() {
        HttpServletRequest request;
        String masterSelect;
        
        request = getServletRequest();
        masterSelect = request.getParameter(config.getParameterNameMasterIndexSelect());
        if (masterSelect != null) {
            try {
                return Integer.parseInt(masterSelect);
            }
            catch (NumberFormatException e) {
                // Ignore 
                getLogger().warn("Value of masterSelect parameter is not a number  masterSelect=" + masterSelect);
            }
        }
        return null;
    }
    
    /**
     * Returns whether sort should be ascending if requested, or null if not requested.  
     */
    private Boolean getSortAscendingParameter(HttpServletRequest request) {
        String value;
        
        value = getServletRequest().getParameter(config.getParameterNameSortAscending());
        if (value != null) {
            return Boolean.parseBoolean(value);
        }
        return null;
    }

    /**
     * Returns sort option if requested, or null if not requested.  
     */
    private String getSortOptionParameter() {
        String result;
        
        result = getServletRequest().getParameter(config.getParameterNameSortOption());
        return result;
    }
    
    /**
     * Returns index of first record of displayed page. 
     */
    private int getStartIndex(int page, int pageSize) {
        return (page - 1) * pageSize;
    }
    
    private int getTotalPages(int listSize) {
        if (config.getPageMode()) {
            return (int)Math.ceil(((float)listSize) / config.getPageSize());
        } else {
            return 1;
        }
    }

    /**
     * Optionally written by subclasses to check if the main body of the execute function should not be run and returns 
     * the result it should return, usually a redirect result, or null for execute to continue.
     */
    protected String checkRedirect() {
        return null;
    }

    /**
     * Optionally written by subclasses for cleanup after calls to {@link #startRecordFormatting} and 
     * {@link #formatRecord}.
     */
    protected void endRecordFormatting() throws Exception {
       // Empty 
    }

    /**
     * Written by subclasses to return formatted record for display from each list record to be displayed.  This is 
     * called after {@link #startRecordFormatting} and before {@link #endRecordFormatting}.      
     * 
     * @param selectedMaster Selected record of master list or null if this viewer action is for the master list.  This
     *                       may not have details if config.requireSelectedMasterDetail is false.
     * @param record Record from list that will be displayed.  It may not have the additional details loaded.
     */
    protected abstract FT formatRecord(M selectedMaster, T record); 
    
    /**
     * Optionally written by subclasses to return index of record to select, typically by a request parameter, or null 
     * for no change of selection.  
     */
    protected Integer getAlternateIndex(ListCache<M,K,T> listCache) {
        return null;
    }
    
    /**
     * May be overridden by subclasses to return Command object to replace current list finder and use it replace list,
     * or null to skip this.
     *  
     * @param selectedMaster Selected record of master list or null if loading master list.
     */
    protected ListFinder<M,T> getFindCommand(M selectedMaster) throws Exception {
        return null;
    }

    /**
     * Returns index of record in list to be formatted.  This is only used in {@link #formatRecord}.
     */
    protected int getFormatRecordIndex() {
        return formatRecordIndex;
    }

    /**
     * May be overridden by subclasses to return a comparator that can sort entries in the list cache as requested by 
     * the user, or null to leave list order as is.  This only applies if a sort order is requested and the list cache 
     * uses full list pagination mode.  
     * 
     * @param sortOption Value of sort parameter set by user.
     * @param sortAscending True if user set sort order to be ascending, false for descending, or null if not specified.
     */
    protected ListCacheRecordComparator<T> getItemSorter(String sortOption, Boolean sortAscending) {
        return null;
    }

    /**
     * Written by subclasses to return list cache that is being populated. 
     */
    protected abstract ListCache<M,K,T> getListCache();
    
    /**
     * Written by subclasses to return master list cache or null if the display list cache is the master list. 
     */
    protected abstract ListCache<?,?,M> getMasterListCache();
 
    /**
     * Written by subclasses to configure behaviour of view action. 
     */
    protected abstract ViewListConfig getViewListConfig();
 
    /**
     * May be overridden by subclasses to return the Struts 2 result name and, more likely, load additional data needed 
     * by the JSP.  The list that will be displayed can be extracted {@link #getFormattedList}.  This is called after 
     * {@link #endRecordFormatting}.
     *  
     * @param selectedMaster Selected record of master list or null if this viewer action is for the master list.  This
     *                       may not have details if config.requireSelectedMasterDetail is false.
     * @param selectedItem Selected record of list.  This may not have details if config.requireSelectedItemDetail is 
     *                     false.
     * @throws Exception Additional data for model cannot be retrieved.
     */
    protected String getViewResponse(M selectedMaster, T selectedItem) throws Exception {
        return "success";
    }
    
    
    /**
     * Optionally written by subclasses to prepare for calls to {@link #formatRecord}.  The records to be formatted are
     * obtained from {@link #getUnformattedList}. 
     */
    protected void startRecordFormatting() throws Exception {
        // Empty
    }

    @Override
    public String execute() throws Exception {
        ListCache<?,?,M> masterListCache;
        ListCache<M,K,T> listCache;
        ListFinder<M,T> newListFinder;
        HttpServletRequest request;
        ListCacheRecordComparator<T> itemSorter;
        M selectedMaster;
        T selectedItem;
        List<FT> formattedSublist;
        Boolean sortAscending;
        String forward, sortOption;
        Integer indexSelect, masterIndexSelect, pageSelect;
        int newIndex, newPage, startIndex, listSize;
        
        forward = checkRedirect();
        if (forward != null) {
            return forward;
        }

        request = getServletRequest();

        masterListCache = getMasterListCache();
        listCache = getListCache();
        if (masterListCache != null && masterListCache != listCache) {
            masterIndexSelect = getMasterIndexSelect();
            if (masterIndexSelect != null) {
                newIndex = masterIndexSelect;
                // Make sure index is within list boundary or -1.
                if (newIndex >= masterListCache.getListSize()) {
                    newIndex = masterListCache.getListSize() - 1;
                } else if (newIndex < 0) {
                    newIndex = -1;
                }
                masterListCache.setSelectedIndex(newIndex);
            }
            if (config.getRequireSelectedMasterDetail()) {
                selectedMaster = masterListCache.getSelected();
            } else {
                selectedMaster = masterListCache.getSelectedNoDetail();
            }
        } else {
            selectedMaster = null;
        }
        
        // IE sometimes submits twice.  This prevents a second find request stomping over the list.
        synchronized (listCache) {
            newListFinder = getFindCommand(selectedMaster);
            if (newListFinder != null) {
                listCache.setListFinder(newListFinder);
                listCache.markReload();
            }
            checkReload(request, listCache);
            sortOption = getSortOptionParameter();
            if (sortOption != null) {
                sortAscending = getSortAscendingParameter(request);
                itemSorter = getItemSorter(sortOption, sortAscending);
                listCache.setItemSorter(itemSorter);
                listCache.markReSort();
            }
            listSize = listCache.getListSize();
            pageSelect = checkPageParameter(request);
            indexSelect = getIndexSelect();
            if (indexSelect == null) {
                indexSelect = getAlternateIndex(listCache);
            }
            
            if (config.getPageMode()) {
            	listCache.setPageSize(config.getPageSize());
            }
            if (indexSelect != null) {
                newIndex = indexSelect;
                // Make sure index is within list boundary or -1.
                if (newIndex >= listSize) {
                    newIndex = listSize - 1;
                } else if (newIndex < 0) {
                    newIndex = -1;
                }
                newPage = calcPageNo(newIndex);
            } else if (pageSelect != null) {
                newPage = pageSelect;
                newPage = checkPageNo(newPage, listSize);
                listCache.setPageNo(newPage);
                checkListOrPage(listCache);
                newIndex = listCache.getSelectedIndex();
                newIndex = checkIndexInPage(newIndex, newPage);
            } else {
                newIndex = listCache.getSelectedIndex();
                newPage = calcPageNo(newIndex);
            }
            
            checkListOrPage(listCache);
            listCache.setSelectedIndex(newIndex);
            if (config.getRequireSelectedItemDetail()) {
                selectedItem = listCache.getSelected();
            } else {
                selectedItem = listCache.getSelectedNoDetail();
            }
        }
        
        startIndex = getStartIndex(newPage, config.getPageSize());
        // Create formatted model
        formattedList = new FormattedListDisplay<>();
        formattedList.setPageMode(config.getPageMode());
        formattedList.setTotalItems(listCache.getListSize());
        if (config.getIgnoreList()) {
            unformattedList = new ArrayList<>();
            formattedList.setSelectedIndex(-1);
            formattedList.setPage(0);
            formattedList.setPageSize(0);
            formattedList.setTotalPages(getTotalPages(listSize));
        } else if (config.getPageMode()) {
            unformattedList = listCache.getPage();
            formattedList.setSelectedIndex((newPage > 0)?(listCache.getSelectedIndex() - startIndex):-1);
            formattedList.setPage(newPage);
            formattedList.setPageSize(config.getPageSize());
            formattedList.setTotalPages(getTotalPages(listSize));
        } else {
            unformattedList = listCache.getList();
            formattedList.setSelectedIndex(listCache.getSelectedIndex());
            formattedList.setPage(1);
            formattedList.setPageSize(listSize);
            formattedList.setTotalPages(1);
        }
        startRecordFormatting();
        formattedSublist = new ArrayList<>();
        formatRecordIndex = startIndex;
        for (T record: unformattedList) {
            formattedSublist.add(formatRecord(selectedMaster, record));
            formatRecordIndex++;
        }
        endRecordFormatting();
        formattedList.setList(formattedSublist);
        forward = getViewResponse(selectedMaster, selectedItem);
        return forward;
    }
    
    public ViewListConfig getConfig() {
        return config;
    }

    /**
     * Returns current page (which is entire list for non-page mode) of formatted records to be displayed.
     */
    public FormattedListDisplay<FT> getFormattedList() {
        return formattedList;
    }
    
    public FT getFormattedSelectedItem() {
        return formattedList.getSelectedItem();
    }
    
    /**
     * Convenience function returning whether backing list cache has selected an existing item, typically to be updated,
     * rather than a new one to create.  
     */
    public boolean getExistingSelection() {
        return formattedList.getSelectedIndex() > -1;
    }

    /**
     * Returns current page (which is entire list for non-page mode) of unformatted records selected for formatting and 
     * display.  
     */
    public List<T> getUnformattedList() {
        return unformattedList;
    }
    
    public T getUnformattedSelectedItem() {
        if (formattedList.getSelectedIndex() > -1) {
            return unformattedList.get(formattedList.getSelectedIndex());
        } else {
            return null;
        }
    }

}
