package name.matthewgreet.strutscommons.action;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import name.matthewgreet.strutscommons.form.AbstractForm;
import name.matthewgreet.strutscommons.util.AbstractCompositeCache;
import name.matthewgreet.strutscommons.util.ListCache;
import name.matthewgreet.strutscommons.util.ListCacheRecordComparator;
import name.matthewgreet.strutscommons.util.ListFinder;


/**
 * <P>Template class for creating Command objects that load a list cache in the {@link AbstractCompositeCache}.</P>  
 * 
 * <P>Concrete subclasses must implement various template functions:-</P>
 * 
 * <DL>
 *   <DT>{@link #getFindCommand}</DT><DD>Returns a Command object used to
 *       load list cache, which can be initialized with an input form received
 *       by this Struts action.</DD>
 *   <DT>{@link #getFindListConfig}</DT><DD>Returns miscellaneous config, such
 *       as whether no records found means an input form rejection.</DD>
 *   <DT>{@link #getListCache}</DT><DD>Returns list cache that stores the
 *       retrieved Value Objects.</DD>
 *   <DT>{@link #getMasterListCache}</DT><DD>If the list cache is a slave list,
 *       returns master list cache that controls it, otherwise returns null.</DD>
 * </DL>
 * 
 * <DL>
 *   <DT>M</DT><DD>Record type of master list cache if loading a slave list or
 *                 Object otherwise.</DD>
 *   <DT>T</DT><DD>Record type of list cache.</DD>
 *   <DT>F</DT><DD>Struts ActionForm that provides any search form fields or 
 *                 ActionForm if no form is used.</DD>
 * </DL>
 */
public abstract class AbstractFindListActionSupport<M extends Serializable,T extends Serializable,F extends AbstractForm> extends AbstractFormDrivenActionSupport<F> {
    private static final long serialVersionUID = 7571697364740160930L;

    /**
     * Configures list to be loaded by Struts, finder action and whether empty lists means search parameters must be 
     * rejected.  Configuration is fixed and does not change during the action's lifetime.
     */
    public static class FindListConfig {
        private boolean rejectNoMatch;
        private String rejectMessage;
        
        public FindListConfig() {
            rejectNoMatch = false;
            rejectMessage = "No results found";
        }
        
        /**
         * Indicates that an empty list means manually entered search parameters
         * must be rejected. 
         */
        public boolean getRejectNoMatch() {
            return rejectNoMatch;
        }
        public void setRejectNoMatch(boolean value) {
            rejectNoMatch = value;
        }
        
        /**
         * Written by subclasses to return name of message to be used if 
         * {@link #getRejectNoMatch} is true.  Defaults to 'No results found'.
         */
        public String getRejectMessage() {
            return rejectMessage;
        }
        public void setRejectMessage(String value) {
            rejectMessage = value;
        }
        
    }

    public static class FindListResponse<M extends Serializable,T extends Serializable> {
        /**
         * Returns response for a failure to create a finder Command or list was rejected. 
         */
        public static <M extends Serializable,T extends Serializable> FindListResponse<M,T> makeFailureResponse(String message) {
            FindListResponse<M,T> result;
            
            result = new FindListResponse<M,T>();
            result.addErrorMessage(message);
            return result;
        }
        
        /**
         * Returns response for a successfully created finder Command and template class will retrieve list. 
         */
        public static <M extends Serializable,T extends Serializable> FindListResponse<M,T> makeSuccessResponse(ListFinder<M,T> finderCommand) {
            FindListResponse<M,T> result;
            
            result = new FindListResponse<M,T>(finderCommand);
            return result;
        }
        
        /**
         * Returns response for a successfully created finder Command and list has already been retrieved. 
         */
        public static <M extends Serializable,T extends Serializable> FindListResponse<M,T> makeSuccessWithListResponse(ListFinder<M,T> finderCommand, List<T> list) {
            FindListResponse<M,T> result;
            
            result = new FindListResponse<M,T>(finderCommand);
            result.setList(list);
            return result;
        }
        
        private boolean rejected = true;
        private String failureResultName = "input";
        private String successResultName = "success";
        private ListFinder<M,T> finderCommand;
        private List<T> list = null;
        private Collection<String> errors = new ArrayList<String>();
        private Collection<String> messages = new ArrayList<String>();
        
        /**
         * Constructor for failure to create a finder Command or list rejected.
         */
        public FindListResponse() {
            this.rejected = true;
        }
        
        /**
         * Constructor for successfully creating a finder Command.
         * @param finderCommand Command object for retrieving list, configured with any search parameters. 
         */
        public FindListResponse(ListFinder<M,T> finderCommand) {
            this.rejected = false;
            this.finderCommand = finderCommand;
        }
        
        /**
         * Indicates creating the finder Command failed or, if it was retrieved, the list was rejected.  If set, 
         * template base class uses action result named by {@link #getFailureResultName}, otherwise uses action result 
         * named by {@link #getSuccessResultName}.  Defaults to true. 
         */
        public boolean getRejected() {
            return rejected;
        }
        
        /**
         * Returns Command object for reloading list with the search parameters. 
         */
        public ListFinder<M,T> getFinderCommand() {
            return finderCommand;
        }
        
        /**
         * Action result name to use if response reports failure.  Defaults to "input".
         */
        public String getFailureResultName() {
            return failureResultName;
        }
        public void setFailureResultName(String value) {
            failureResultName = value;
        }

        /**
         * Action result name to use if response reports success.  Defaults to "success".
         */
        public String getSuccessResultName() {
            return successResultName;
        }
        public void setSuccessResultName(String value) {
            successResultName = value;
        }

        /**
         * List returned by finderCommand if it was already retrieved, or null if not. 
         */
        public List<T> getList() {
            return list;
        }
        public void setList(List<T> list) {
            this.list = list;
        }
        
        /**
         * Struts errors messages to be used, usually for update failure.  An error message does not indicate an error 
         * (use default constructor) and is displayed even for success (though that's unusual). 
         */
        public Collection<String> getErrors() {
            return errors;
        }
        
        /**
         * Struts information messages to be used, usually for update success.  An info message does not indicate a 
         * success (use constructor accepting list finder) and is displayed even for failure (though that's unusual).  
         */
        public Collection<String> getMessages() {
            return messages;
        }
        
        /**
         * Convenience function to add error message.
         */
        public void addErrorMessage(String text) {
            errors.add(text);
        }
        
        /**
         * Convenience function to add info message.
         */
        public void addInfoMessage(String text) {
            messages.add(text);
        }

    }

    public AbstractFindListActionSupport() {
        super();
    }
    
    /**
     * Overridden by subclasses to return Command object for retrieving list (configured with any search parameters) and 
     * forwarding in the case of success. 
     * @param selectedMaster Selected record of master list or null if loading master list.
     */
    protected abstract FindListResponse<M,T> getFindCommand(M selectedMaster) throws Exception;

    /**
     * Written by subclasses to configure behaviour of find action. 
     */
    protected abstract FindListConfig getFindListConfig();
    
    /**
     * Written by subclasses to return item comparator for sorting list after finding it, or null for no post-find 
     * sorting. 
     */
    protected ListCacheRecordComparator<T> getItemSorter() {
        return null;
    }
 
    /**
     * Written by subclasses to return list cache that is being populated. 
     */
    protected abstract ListCache<M,?,T> getListCache();
 
    /**
     * Written by subclasses to return master list cache or null if the list 
     * cache is the master list. 
     */
    protected abstract ListCache<?,?,M> getMasterListCache();
 
    /**
     * May be overridden by subclasses to translate an exception thrown by a list
     * finder Command into a Struts message key, so a message is displayed to 
     * the user.  May return null to default to a generic error message key. 
     * @param finderCommand List finder Command.
     * @param e Exception thrown by {@link ListFinder#getList} function.
     */
    protected String translateFinderException(ListFinder<M,T> finderCommand, Exception e) {
        return null;
    }

    /**
     * 
     */
    @Override
    public String execute() throws Exception {
        ListCacheRecordComparator<T> itemSorter;
        ListCache<?,?,M> masterListCache;
        ListCache<M,?,T> listCache;
        FindListConfig config;
        FindListResponse<M,T> findResponse;
        ListFinder<M,T> finderCommand;
        List<T> list;
        M selectedMaster;
        String errorMessage;
        
        config = getFindListConfig();
        findResponse = null;
        finderCommand = null;
        errorMessage = null;
        
        masterListCache = getMasterListCache();
        listCache = getListCache();

        try {
            // Get finder commmand for this list
            if (masterListCache != null && masterListCache != listCache) {
                selectedMaster = masterListCache.getSelected();
            } else {
                selectedMaster = null;
            }
            findResponse = getFindCommand(selectedMaster);
            if (!findResponse.getRejected()) {
                finderCommand = findResponse.getFinderCommand();
                // Run finder command if not already done
                if (findResponse.getList() != null) {
                    list = findResponse.getList();
                } else {
                    list = finderCommand.getList(selectedMaster);
                }
                if (list == null) {
                    list = new ArrayList<T>();
                }
                if (!config.getRejectNoMatch() || (list.size() > 0)) {
                    // Find succeeded.  Store list and finder command in cache and redirect to viewer.
                    listCache.setListAndFinder(list, finderCommand);
                    itemSorter = getItemSorter();
                    if (itemSorter != null) {
                        listCache.setItemSorter(itemSorter);
                    }
                } else {
                    // No results and results required - write error message 
                    addActionError(config.getRejectMessage());
                }
            }
        }
        catch (Exception e) {
            // Determine error message and write it
            if (finderCommand != null) {
                errorMessage = translateFinderException(finderCommand, e);
            }
            if (errorMessage == null || errorMessage.length() == 0) {
                errorMessage = "System failure: " + e.getMessage();
            }
            getLogger().error(errorMessage, e);
            addActionError(errorMessage);
        }
        
        if (findResponse != null) {
            for (String message: findResponse.getMessages()) {
                addActionMessage(message);
            }
            for (String errorMessage2: findResponse.getErrors()) {
                addActionError(errorMessage2);
            }
            if (!findResponse.getRejected()) {
                return findResponse.getSuccessResultName();
            } else {
                return findResponse.getFailureResultName();
            }
        } else {    // Shouldn't happen
            return "error";
        }
    }

}
