package name.matthewgreet.strutscommons.util;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.opensymphony.xwork2.ActionContext;
import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.conversion.impl.ConversionData;
import com.opensymphony.xwork2.util.ValueStack;

import name.matthewgreet.strutscommons.annotation.Form;
import name.matthewgreet.strutscommons.form.FormatResult;
import name.matthewgreet.strutscommons.form.FormattableForm;
import name.matthewgreet.strutscommons.interceptor.FormFormatterInterceptor;
import name.matthewgreet.strutscommons.util.DefaultFormFormatter.FieldContext.FieldStep;
import name.matthewgreet.strutscommons.util.DefaultFormFormatter.FieldContext.FormatMode;
import name.matthewgreet.strutscommons.util.DefaultFormFormatter.InterceptorContext.InterceptorStep;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.AnnotationEntries;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.CategoriseFieldResult;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.FieldUsage;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.PairFieldUsage;

/**
 * <P>Implementation of {@link FormFormatter} used by {@link FormFormatterInterceptor}.</P>
 */
@SuppressWarnings("deprecation")
public class DefaultFormFormatter implements FormFormatter {
    /**
     * Thread local execution state of algorithm about a field.  Use {@link #getInstance} to obtain the current state.  
     * Properties are set as the algorithm progresses and these are divided into sets, shown below.  Documentation of 
     * each library function defines the sets in use as well as its purpose. 
     * 
     * <H2>Property set 1</H2>
     * <UL>
     *   <LI>{@link #getStep step}</LI>
     * </UL>
     * 
     * <H2>Property set 2</H2>
     * <UL>
     *   <LI>{@link #getFormatMode formatMode}</LI>
     *   <LI>{@link #getFieldUsage fieldUsage}</LI>
     *   <LI>{@link #getPairFieldUsage pairFieldUsage}</LI>
     *   <LI>{@link #getUnformattedField unformattedField}</LI>
     *   <LI>{@link #getAnnountationEntries annountationEntries}</LI>
     *   <LI>{@link #getFieldName fieldName}</LI>
     *   <LI>{@link #getUnformattedValue unformattedValue}</LI>
     *   <LI>{@link #getUnformattedCollectionValue unformattedCollectionValue}</LI>
     * </UL>
     * 
     * <H2>Property set 3</H2>
     * <UL>
     *   <LI>{@link #getFormattedValue formattedValue}</LI>
     *   <LI>{@link #getFormattedMultipleValues formattedMultipleValues}</LI>
     * </UL>
     */
    public static class FieldContext<T> {
    	public enum FormatMode {AUTO, DEFAULT, NONE, PAIR, SET_ONLY}
    	public enum FieldStep {START, FORMATTING, END}
    	
    	private static ThreadLocal<FieldContext<?>> instance = new ThreadLocal<>();
    	
    	@SuppressWarnings("unchecked")
		public static <T> FieldContext<T> getInstance() {
    		return (FieldContext<T>)instance.get();
    	}
    	public static <T> void setInstance(FieldContext<T> value) {
    		instance.set(value);
    	}

    	// Set 1
    	private FieldStep step;
    	// Set 2
    	private FormatMode formatMode;
    	private FieldUsage<T> fieldUsage;
    	private PairFieldUsage<T> pairFieldUsage;
    	private Field unformattedField;
    	private AnnotationEntries<T> annountationEntries;
    	private String fieldName;
        private T unformattedValue;
    	private Collection<T> unformattedCollectionValue;
    	// Set 3
    	private String formattedValue;
    	private List<String> formattedMultipleValues;


		protected FieldContext() {
			super();
			reset();
		}

		
		public void reset() {
			step = null;
			formatMode = null;
	    	fieldUsage = null;
	    	pairFieldUsage = null;
	    	unformattedField = null;
	    	annountationEntries = null;
	    	fieldName = null;
	    	unformattedValue = null;
	    	unformattedCollectionValue = null;
	    	
	    	formattedValue = null;
	    	formattedMultipleValues = null;
		}
		
		
		/**
		 * Returns current, field-level processing step.
		 */
		public FieldStep getStep() {
			return step;
		}
		public void setStep(FieldStep step) {
			this.step = step;
		}

		/**
		 * Returns format mode, defining the processing of a formatted form field value. 
		 */
		public FormatMode getFormatMode() {
			return formatMode;
		}
		public void setFormatType(FormatMode formatMode) {
			this.formatMode = formatMode;
		}
		
		/**
		 * Returns field and configured policies to apply except for pair conversion mode.
		 */
		public FieldUsage<T> getFieldUsage() {
			return fieldUsage;
		}
		public void setFieldUsage(FieldUsage<T> fieldUsage) {
			this.fieldUsage = fieldUsage;
		}

		/**
		 * Returns field and configured policies to apply for pair conversion mode.
		 */
		public PairFieldUsage<T> getPairFieldUsage() {
			return pairFieldUsage;
		}
		public void setPairFieldUsage(PairFieldUsage<T> pairFieldUsage) {
			this.pairFieldUsage = pairFieldUsage;
		}

		/**
		 * Returns field containing value to be formatted. 
		 */
		public Field getUnformattedField() {
			return unformattedField;
		}
		public void setUnformattedField(Field unformattedField) {
			this.unformattedField = unformattedField;
		}

		/**
		 * Returns configured policies of form field.
		 */
		public AnnotationEntries<T> getAnnountationEntries() {
			return annountationEntries;
		}
		public void setAnnountationEntries(AnnotationEntries<T> annountationEntries) {
			this.annountationEntries = annountationEntries;
		}
		
		/**
		 * Returns name of field containing unformatted value.
		 */
		public String getFieldName() {
			return fieldName;
		}
		public void setFieldName(String fieldName) {
			this.fieldName = fieldName;
		}

		/**
		 * Returns unformatted value for single value fields. 
		 */
		public T getUnformattedValue() {
			return unformattedValue;
		}
		public void setUnformattedValue(T unformattedValue) {
			this.unformattedValue = unformattedValue;
		}

		/**
		 * Returns unformatted value for collection fields. 
		 */
		public Collection<T> getUnformattedCollectionValue() {
			return unformattedCollectionValue;
		}
		public void setUnformattedCollectionValue(Collection<T> unformattedCollectionValue) {
			this.unformattedCollectionValue = unformattedCollectionValue;
		}

		/**
		 * Returns formatted value for single value fields. 
		 */
		public String getFormattedValue() {
			return formattedValue;
		}
		public void setFormattedValue(String formattedValue) {
			this.formattedValue = formattedValue;
		}
		
		/**
		 * Returns formatted value for collection fields. 
		 */
		public List<String> getFormattedMultipleValues() {
			return formattedMultipleValues;
		}
		public void setFormattedMultipleValues(List<String> formattedMultipleValues) {
			this.formattedMultipleValues = formattedMultipleValues;
		}

    }
    	
    /**
     * Thread local execution state of algorithm.  Use {@link #getInstance} to obtain the current state.  Properties 
     * are set as the algorithm progresses and these are divided into sets, shown below.  Documentation of each library 
     * function defines the sets in use as well as its purpose. 
     * 
     * <H2>Property set 1</H2>
     * <UL>
     *   <LI>{@link #getStep step}</LI>
     * </UL>
     * 
     * <H2>Property set 2</H2>
     * <UL>
     *   <LI>{@link #getInvocation invocation}</LI>
     *   <LI>{@link #getAction action}</LI>
     *   <LI>{@link #getFakeForms fakeForms} - formatted form field values to display</LI>
     * </UL>
     * 
     * <H2>Property set 3</H2>
     * <UL>
     *   <LI>{@link #getActionField actionField}</LI>
     *   <LI>{@link #getForm form}</LI>
     *   <LI>{@link #getFormattableForm formattableForm}</LI>
     *   <LI>{@link #getFormattableForm skipConversionErrors} - if true, form was processed and conversion errors may exist</LI>
     * </UL>
     * 
     * <H2>Property set 4</H2>
     * <UL>
     *   <LI>{@link #getAutoParameterConversionFields autoParameterConversionFields}</LI>
     *   <LI>{@link #getDefaultParameterConversionFields defaultParameterConversionFields}</LI>
     *   <LI>{@link #getManualParameterConversionFields manualParameterConversionFields}</LI>
     *   <LI>{@link #getNoConversionFields noConversionFields}</LI>
     *   <LI>{@link #getPairConversionFields pairConversionFields}</LI>
     *   <LI>{@link #getSetOnlyFields setOnlyFields}</LI>
     * </UL>
     * 
     */
    public static class InterceptorContext {
    	public enum InterceptorStep {START, FORM_PROCESSING_START, ANNOTATION_PARSING, FIELD_PROCESSING, 
    		MANUAL_FORMATTING, FORM_PROCESSING_END, FORMAT_WRITING, END}

    	private static ThreadLocal<InterceptorContext> instance = new ThreadLocal<>();
    	
    	public static InterceptorContext getInstance() {
    		return instance.get();
    	}
    	public static void setInstance(InterceptorContext value) {
    		instance.set(value);
    	}

    	// Set 1
    	private InterceptorStep step;
    	// Set 2
    	private ActionInvocation invocation;
    	private Object action;
    	private Map<String,Map<String,Object>> fakeForms;
    	// Set 3
    	private Field actionField;
    	private Object form;
    	private FormattableForm formattableForm;
    	private boolean skipConversionErrors;								// Form was processed and may have conversions errors
    	// Set 4
    	private Collection<FieldUsage<?>> autoParameterConversionFields;	// Formats using annotated converter
    	private Collection<FieldUsage<?>> defaultParameterConversionFields;	// Formats using default converter
    	private Collection<FieldUsage<?>> manualParameterConversionFields;	// Calls form to format field
    	private Collection<FieldUsage<?>> noConversionFields;				// No formatting required 
    	private Collection<PairFieldUsage<?>> pairConversionFields;			// Formats from unformatted field to formatted field
    	private Collection<FieldUsage<?>> setOnlyFields;				    // No formatting required 
    	
    	
		protected InterceptorContext() {
			super();
			reset();
		}
		
		public void reset() {
	    	step = null;
	    	invocation = null;
	    	action = null;
	    	
	    	fakeForms = new HashMap<>();
	    	
	    	actionField = null;
	    	form = null;
	    	formattableForm = null;
	    	skipConversionErrors = false;
	    	autoParameterConversionFields = new ArrayList<>();
	    	defaultParameterConversionFields = new ArrayList<>();
	    	manualParameterConversionFields = new ArrayList<>();
	    	noConversionFields = new ArrayList<>(); 
	    	pairConversionFields = new ArrayList<>();
		}
		
		public InterceptorStep getStep() {
			return step;
		}
		public void setStep(InterceptorStep step) {
			this.step = step;
		}

		public ActionInvocation getInvocation() {
			return invocation;
		}
		public void setInvocation(ActionInvocation invocation) {
			this.invocation = invocation;
		}

		public Object getAction() {
			return action;
		}
		public void setAction(Object action) {
			this.action = action;
		}
		
		public Map<String, Map<String,Object>> getFakeForms() {
			return fakeForms;
		}
		public void setFakeForms(Map<String, Map<String,Object>> fakeForms) {
			this.fakeForms = fakeForms;
		}

		public Field getActionField() {
			return actionField;
		}
		public void setActionField(Field actionField) {
			this.actionField = actionField;
		}
		
		public Object getForm() {
			return form;
		}
		public void setForm(Object form) {
			this.form = form;
		}
		
		public FormattableForm getFormattableForm() {
			return formattableForm;
		}
		public void setFormattableForm(FormattableForm formattableForm) {
			this.formattableForm = formattableForm;
		}
		
		public boolean getSkipConversionErrors() {
			return skipConversionErrors;
		}
		public void setSkipConversionErrors(boolean skipConversionErrors) {
			this.skipConversionErrors = skipConversionErrors;
		}

		
		public Collection<FieldUsage<?>> getAutoParameterConversionFields() {
			return autoParameterConversionFields;
		}
		public void setAutoParameterConversionFields(Collection<FieldUsage<?>> autoParameterConversionFields) {
			this.autoParameterConversionFields = autoParameterConversionFields;
		}

		public Collection<FieldUsage<?>> getDefaultParameterConversionFields() {
			return defaultParameterConversionFields;
		}
		public void setDefaultParameterConversionFields(Collection<FieldUsage<?>> defaultParameterConversionFields) {
			this.defaultParameterConversionFields = defaultParameterConversionFields;
		}
		
		public Collection<FieldUsage<?>> getManualParameterConversionFields() {
			return manualParameterConversionFields;
		}
		public void setManualParameterConversionFields(Collection<FieldUsage<?>> manualParameterConversionFields) {
			this.manualParameterConversionFields = manualParameterConversionFields;
		}
		
		public Collection<FieldUsage<?>> getNoConversionFields() {
			return noConversionFields;
		}
		public void setNoConversionFields(Collection<FieldUsage<?>> noConversionFields) {
			this.noConversionFields = noConversionFields;
		}
		
		public Collection<PairFieldUsage<?>> getPairConversionFields() {
			return pairConversionFields;
		}
		public void setPairConversionFields(Collection<PairFieldUsage<?>> pairConversionFields) {
			this.pairConversionFields = pairConversionFields;
		}
		
		public Collection<FieldUsage<?>> getSetOnlyFields() {
			return setOnlyFields;
		}
		public void setSetOnlyFields(Collection<FieldUsage<?>> setOnlyFields) {
			this.setOnlyFields = setOnlyFields;
		}
		
    }
    
    private Logger LOG = LogManager.getLogger(DefaultFormFormatter.class);
    

    /**
     * <P>Sets how fields of a form should be formatted (InterceptorContext setting property set 4).</P>
     * 
     * <P>Can access InterceptorContext property set 3 and lower.</P>
     */
    protected CategoriseFieldResult categoriseFields(Collection<Field> formFields, Object action, Object form) {
    	CategoriseFieldResult result;
    	InterceptorContext interceptorContext;
    	
    	interceptorContext = InterceptorContext.getInstance();
    	result = InterceptorCommonLibrary.categoriseFormFields(formFields);
    	interceptorContext.setAutoParameterConversionFields(result.getAutoParameterConversionFields());
    	interceptorContext.setDefaultParameterConversionFields(result.getDefaultParameterConversionFields());
    	interceptorContext.setManualParameterConversionFields(result.getManualParameterConversionFields());
    	interceptorContext.setNoConversionFields(result.getNoConversionFields());
    	interceptorContext.setPairConversionFields(result.getPairConversionFields());
    	interceptorContext.setSetOnlyFields(result.getSetOnlyFields());
    	return result;
	}

	/**
     * Called when processing of a form field is complete.  Just sets end state and really exists for extending 
     * subclasses. 
     */
    protected <T> void endFieldContext() {
    	FieldContext<T> fieldContext;
    	
    	fieldContext = FieldContext.getInstance();
    	fieldContext.setStep(FieldStep.END);
	}

	/**
     * Called when formatting a form completes.  Does nothing and really exists for extending subclasses. 
     */
	protected void endFormProcessing() {
		// Empty
	}

	/**
     * Called when this interceptor completes (and is not disabled).  Does nothing and really exists for extending subclasses. 
     */
	protected void endInterceptorContext() {
		// Empty
	}
	
	/**
	 * Returns whether Action field is a formatted form. 
	 */
    protected boolean fieldIsForm(Field actionField) {
    	Class<?> fieldClass;
    	
    	fieldClass = actionField.getType();
		return FormattableForm.class.isAssignableFrom(fieldClass);
	}

    /**
     * Returns whether field of an Action received a stored form. 
     */
    protected boolean fieldReceivedStoredForm(Field actionField) {
		return InterceptorCommonLibrary.fieldReceivedStoredForm(actionField);
	}
    
    /**
     * Returns form fields from parameter excluding those that are definitely not form fields and, if form was 
     * processed, excluding those linked to conversionErrors. 
     * @param conversionErrors 
     */
    protected Collection<Field> filterFormFields(Collection<Field> allFormFields, Map<String, ConversionData> conversionErrors) {
    	Collection<Field> working;
    	
    	working = allFormFields;
    	working = InterceptorCommonLibrary.filterNonConversionErrorFormFields(working, conversionErrors);
    	working = InterceptorCommonLibrary.filterPlausibleFormFields(working);
    	return working;
    }
    
	/**
	 * <P>Formats all fields of a form.  Ignores forms that aren't {@link FormattableForm}.</P> 
     * 
     * <P>Can access InterceptorContext property set 3 and lower.</P>
	 */
    protected void formatForm() {
    	CategoriseFieldResult categoriseFieldResult;
    	FieldContext<?> fieldContext;
    	InterceptorContext interceptorContext;
        Collection<Field> allFormFields;
        Object action, form;
		
    	interceptorContext = InterceptorContext.getInstance();
    	action = interceptorContext.getAction();
    	form = interceptorContext.getForm();
    	if (form == null) {
    		return;
    	}
    	if (!(form instanceof FormattableForm)) {
    		return;
    	}
    	interceptorContext.setStep(InterceptorStep.ANNOTATION_PARSING);
        allFormFields = getProperties(form.getClass());
        if (interceptorContext.getSkipConversionErrors()) {
        	allFormFields = filterFormFields(allFormFields, ActionContext.getContext().getConversionErrors());
        } else {
        	allFormFields = filterFormFields(allFormFields, null);
        }
        categoriseFieldResult = categoriseFields(allFormFields, action, form);
        
    	interceptorContext.setStep(InterceptorStep.FIELD_PROCESSING);
        for (FieldUsage<?> fieldUsage: categoriseFieldResult.getAutoParameterConversionFields()) {
        	try {
        		fieldContext = makeFieldContext();
        		FieldContext.setInstance(fieldContext);
        		startFieldContext(FormatMode.AUTO, fieldUsage, null);
        		
        		processAutoField();
        		
        		endFieldContext();
        	}
        	catch (Exception e) {
                LOG.error("Formatting auto conversion field failed  Struts action=" + action.getClass() + 
                    "  form=" + form.getClass() + "  unformatted field=" + fieldUsage.getName(), e);
        	}
        }
        for (FieldUsage<?> fieldUsage: categoriseFieldResult.getDefaultParameterConversionFields()) {
        	try {
        		fieldContext = makeFieldContext();
        		FieldContext.setInstance(fieldContext);
        		startFieldContext(FormatMode.DEFAULT, fieldUsage, null);
        		
        		processAutoField();
        		
        		endFieldContext();
        	}
        	catch (Exception e) {
                LOG.error("Formatting default conversion field failed  Struts action=" + action.getClass() + 
                    "  form=" + form.getClass() + "  unformatted field=" + fieldUsage.getName(), e);
        	}
        }
    	for (FieldUsage<?> fieldUsage: categoriseFieldResult.getNoConversionFields()) {
        	try {
        		fieldContext = makeFieldContext();
        		FieldContext.setInstance(fieldContext);
        		startFieldContext(FormatMode.NONE, fieldUsage, null);
        		
        		processNoConversionField();
        		
        		endFieldContext();
        	}
        	catch (Exception e) {
                LOG.error("Formatting no conversion field failed  Struts action=" + action.getClass() + 
                    "  form=" + form.getClass() + "  unformatted field=" + fieldUsage.getName(), e);
        	}
    	}
        for (PairFieldUsage<?> pairFieldUsage: categoriseFieldResult.getPairConversionFields()) {
        	try {
        		fieldContext = makeFieldContext();
        		FieldContext.setInstance(fieldContext);
        		startFieldContext(FormatMode.PAIR, null, pairFieldUsage);
        		
        		processPairField();
        		
        		endFieldContext();
        	}
        	catch (Exception e) {
                LOG.error("Formatting pair conversion field failed  Struts action=" + action.getClass() + 
                    "  form=" + form.getClass() + "  formatted field=" + pairFieldUsage.getFormattedName() + 
                    "  unformatted field=" + pairFieldUsage.getUnformattedName(), e);
        	}
        }
        for (FieldUsage<?> fieldUsage: categoriseFieldResult.getSetOnlyFields()) {
        	try {
        		fieldContext = makeFieldContext();
        		FieldContext.setInstance(fieldContext);
        		startFieldContext(FormatMode.SET_ONLY, fieldUsage, null);
        		
        		processSetOnlyField();
        		
        		endFieldContext();
        	}
        	catch (Exception e) {
                LOG.error("Formatting default conversion field failed  Struts action=" + action.getClass() + 
                    "  form=" + form.getClass() + "  unformatted field=" + fieldUsage.getName(), e);
        	}
        }
    	interceptorContext.setStep(InterceptorStep.MANUAL_FORMATTING);
    	try {
    		processManualParameterField();
    	}
    	catch (Exception e) {
            LOG.error("Formatting manual parameters failed  Struts action=" + action.getClass() + 
                "  form=" + form.getClass(), e);
    	}
	}

    /**
     * <P>Returns all form fields of defined by form class.</P> 
     * 
     * <P>Can access InterceptorContext property set 3 and lower.</P>
     */
	protected Collection<Field> getProperties(Class<?> type) {
    	return InterceptorCommonLibrary.getProperties(type);
    }

	protected <T> FieldContext<T> makeFieldContext() {
    	return new FieldContext<T>();
    }

    protected InterceptorContext makeInterceptorContext() {
    	return new InterceptorContext();
    }
    
	/**
	 * <P>Formats non-string form field annotated by converter.</P> 
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 2 and lower.</P>
	 */
	protected <T> void processAutoField() {
        FieldContext<T> fieldContext;
        FieldUsage<T> fieldUsage;
        String formattedValue;
        
        fieldContext = FieldContext.getInstance();
        fieldUsage = fieldContext.getFieldUsage();
        if (fieldUsage.getCollection()) {
        	formattedValue = processCollectionConverter();
        	fieldContext.setFormattedValue(formattedValue);
        	fieldContext.setFormattedMultipleValues(null);
        	putFakeFormField();
        } else {
        	formattedValue = processConverter();
        	fieldContext.setFormattedValue(formattedValue);
        	fieldContext.setFormattedMultipleValues(null);
        	putFakeFormField();
        }
	}

    protected <T> String processCollectionConverter() {
        FieldContext<T> fieldContext;
        InterceptorContext interceptorContext;
		Collection<T> unformattedValue;
		String result;
		
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        unformattedValue = null;
		try {
			unformattedValue = fieldContext.getUnformattedCollectionValue();
			if (unformattedValue != null && unformattedValue.size() > 0) {
				result = fieldContext.getAnnountationEntries().getCollectionConverter().getCollectionConverter().format(unformattedValue);
			} else {
				result = "";
			}
			return result;
		}
		catch (IllegalArgumentException e) {
            LOG.error("Security violation when accessing form" + 
                    "  Struts action=" + interceptorContext.getAction().getClass() + "  field=" + fieldContext.getFieldName(), e);
		}
		catch (ClassCastException e) {
            LOG.error("Form field type doesn't match converter" + 
                    "  Struts action=" + interceptorContext.getAction().getClass() + "  field=" + fieldContext.getFieldName(), e);
		}
		catch (Exception e) {
            LOG.error("Converter failed" + 
                "  Struts action=" + interceptorContext.getAction().getClass() + "  field=" + fieldContext.getFieldName() + 
                "  value=" + unformattedValue, e);
		}
		return null;
	}


    protected <T> String processConverter() {
        FieldContext<T> fieldContext;
        InterceptorContext interceptorContext;
		T unformattedValue;
		String result;
		
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        unformattedValue = null;
		try {
			unformattedValue = fieldContext.getUnformattedValue();
			if (unformattedValue != null) {
				result = fieldContext.getAnnountationEntries().getConverter().getConverter().format(unformattedValue);
			} else {
				result = "";
			}
			return result;
		}
		catch (IllegalArgumentException e) {
            LOG.error("Security violation when accessing form" + 
                    "  Struts action=" + interceptorContext.getAction().getClass() + "  field=" + fieldContext.getFieldName(), e);
		}
		catch (ClassCastException e) {
            LOG.error("Form field type doesn't match converter" + 
                    "  Struts action=" + interceptorContext.getAction().getClass() + "  field=" + fieldContext.getFieldName(), e);
		}
		catch (Exception e) {
            LOG.error("Converter failed" + 
                "  Struts action=" + interceptorContext.getAction().getClass() + "  field=" + fieldContext.getFieldName() + 
                "  value=" + unformattedValue, e);
		}
		return null;
	}

	/**
	 * <P>Formats non-string form field not annotated by converter, using default converter.</P> 
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 2 and lower.</P>
	 */
	protected <T> void processDefaultField() {
        FieldContext<T> fieldContext;
        FieldUsage<T> fieldUsage;
        String formattedValue;
        
        fieldContext = FieldContext.getInstance();
        fieldUsage = fieldContext.getFieldUsage();
        if (fieldUsage.getCollection()) {
        	formattedValue = processCollectionConverter();
        	fieldContext.setFormattedValue(formattedValue);
        	fieldContext.setFormattedMultipleValues(null);
        	putFakeFormField();
        } else {
        	formattedValue = processConverter();
        	fieldContext.setFormattedValue(formattedValue);
        	fieldContext.setFormattedMultipleValues(null);
        	putFakeFormField();
        }
	}

	protected void processFormattedValues() {
        InterceptorContext interceptorContext;
        ValueStack valueStack;
    	Map<String,Map<String,Object>> fakeForms;

        interceptorContext = InterceptorContext.getInstance();
		fakeForms = interceptorContext.getFakeForms();
        InterceptorCommonLibrary.getActionContextFormattedForms().putAll(fakeForms);
		valueStack = interceptorContext.getInvocation().getStack();
		valueStack.push(fakeForms);
	}
	
	protected <T> void processManualParameterField() {
		FieldContext<T> fieldContext;
        FormattableForm formattableForm;
        FormatResult formatResult;
        InterceptorContext interceptorContext;
        List<String> fieldValues;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        formattableForm = interceptorContext.getFormattableForm();
        if (formattableForm != null) {
        	formatResult = formattableForm.format();
        	for (Entry<String, String> entry: formatResult.getFormattedSingleFieldValues().entrySet()) {
        		fieldContext.setUnformattedField(null);
        		fieldContext.setFieldName(entry.getKey());
	        	fieldContext.setFormattedValue(entry.getValue());
	        	fieldContext.setFormattedMultipleValues(null);
	        	putFakeFormField();
        	}
        	for (Entry<String, List<String>> fieldNameEntry: formatResult.getFormattedMultipleFieldValues().entrySet()) {
        		fieldValues = new ArrayList<>();
        		for (String fieldValue: fieldNameEntry.getValue()) {
        			fieldValues.add(fieldValue);
        		}
        		fieldContext.setUnformattedField(null);
        		fieldContext.setFieldName(fieldNameEntry.getKey());
	        	fieldContext.setFormattedValue(null);
	        	fieldContext.setFormattedMultipleValues(fieldValues);
	        	putFakeMultipleFormField();
        	}
        }
	}
	
	/**
	 * <P>Called for string form fields, which just copies form field value to fake form.</P> 
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 2 and lower.</P>
	 */
	protected void processNoConversionField() {
        FieldContext<?> fieldContext;
        String formattedValue, unformattedValue;
        
        fieldContext = FieldContext.getInstance();
        
        unformattedValue = (String)fieldContext.getUnformattedValue();
        if (unformattedValue != null) {
        	formattedValue = unformattedValue;
        } else {
        	formattedValue = "";
        }
    	fieldContext.setFormattedValue(formattedValue);
    	fieldContext.setFormattedMultipleValues(null);
    	putFakeFormField();
	}

	/**
	 * <P>Called for string arrays and string collection form fields, which just copies form field values to fake form.</P> 
     * 
     * <P>Can access all InterceptorContext properties and FieldContext property set 2 and lower.</P>
	 */
	@SuppressWarnings("unchecked")
	protected void processSetOnlyField() {
        FieldContext<?> fieldContext;
        FieldUsage<?> fieldUsage;
        List<String> formattedCollectionValue;
        String formattedValue;
        
        fieldContext = FieldContext.getInstance();
        
        fieldUsage = fieldContext.getFieldUsage();
        if (fieldUsage.getArray()) {
        	formattedCollectionValue = (List<String>)fieldContext.getUnformattedCollectionValue();
        	fieldContext.setFormattedValue(null);
        	fieldContext.setFormattedMultipleValues(formattedCollectionValue);
        	putFakeMultipleFormField();
        } else if (fieldUsage.getCollection()) {
        	formattedCollectionValue = (List<String>)fieldContext.getUnformattedCollectionValue();
        	fieldContext.setFormattedValue(null);
        	fieldContext.setFormattedMultipleValues(formattedCollectionValue);
        	putFakeMultipleFormField();
        } else {
        	formattedValue = (String)fieldContext.getUnformattedValue();
        	fieldContext.setFormattedValue(formattedValue);
        	fieldContext.setFormattedMultipleValues(null);
        	putFakeFormField();
        }
	}

	protected <T> void putFakeMultipleFormField() {
        FieldContext<T> fieldContext;
        InterceptorContext interceptorContext;
        Map<String,Object> fakeForm;
        String formName, fieldName;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        formName = interceptorContext.getActionField().getName();
        fieldName = fieldContext.getFieldName();
        
        fakeForm = interceptorContext.getFakeForms().get(formName);
        if (fakeForm == null) {
        	fakeForm = new HashMap<>();
        	interceptorContext.getFakeForms().put(formName, fakeForm);
        }
        fakeForm.put(fieldName, fieldContext.getFormattedMultipleValues());
	}

	protected <T> void putFakeFormField() {
        FieldContext<T> fieldContext;
        InterceptorContext interceptorContext;
        Map<String,Object> fakeForm;
        String formName, fieldName;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        formName = interceptorContext.getActionField().getName();
        fieldName = fieldContext.getFieldName();
        
        fakeForm = interceptorContext.getFakeForms().get(formName);
        if (fakeForm == null) {
        	fakeForm = new HashMap<>();
        	interceptorContext.getFakeForms().put(formName, fakeForm);
        }
        fakeForm.put(fieldName, fieldContext.getFormattedValue());
	}

	protected <T> void processPairField() throws Exception {
        FieldContext<T> fieldContext;
        FormattableForm form;
        InterceptorContext interceptorContext;
        PairFieldUsage<T> pairFieldUsage;
        String formattedValue;
        
        interceptorContext = InterceptorContext.getInstance();
        fieldContext = FieldContext.getInstance();
        form = interceptorContext.getFormattableForm();
        pairFieldUsage = fieldContext.getPairFieldUsage();
        
    	formattedValue = processConverter();
    	fieldContext.setFormattedValue(formattedValue);
    	fieldContext.setFormattedMultipleValues(null);
    	pairFieldUsage.getFormattedField().setAccessible(true);
		pairFieldUsage.getFormattedField().set(form, formattedValue);
    	putFakeFormField();
	}
	
	/**
	 * Called when algorithm is about to start for a field in earnest and sets FieldContext property set 2. 
	 */
	@SuppressWarnings("unchecked")
	protected <T> void startFieldContext(FormatMode formatType, FieldUsage<T> fieldUsage, PairFieldUsage<T> pairFieldUsage) throws Exception {
		FieldContext<T> fieldContext;
		InterceptorContext interceptorContext;
		T unformattedValue;
		T[] unformattedArrayValues;
		Collection<T> unformattedCollectionValues;
		
		interceptorContext = InterceptorContext.getInstance();
		fieldContext = FieldContext.getInstance();
		fieldContext.setStep(FieldStep.START);
		fieldContext.setFormatType(formatType);
		fieldContext.setFieldUsage(fieldUsage);
		fieldContext.setPairFieldUsage(pairFieldUsage);
		switch (formatType) {
		case AUTO:
		case DEFAULT:
		case NONE:
		case SET_ONLY:
			fieldContext.setUnformattedField(fieldUsage.getField());
			fieldContext.setAnnountationEntries(fieldUsage.getAnnountationEntries());
			fieldContext.setFieldName(fieldUsage.getName());
			fieldUsage.getField().setAccessible(true);
			if (fieldUsage.getArray()) {
				unformattedArrayValues = (T[])fieldContext.getUnformattedField().get(interceptorContext.getForm());
				fieldContext.setUnformattedValue(null);
				fieldContext.setUnformattedCollectionValue(Arrays.asList(unformattedArrayValues));
			} else if (fieldUsage.getCollection()) {
				unformattedCollectionValues = (Collection<T>)fieldContext.getUnformattedField().get(interceptorContext.getForm());
				fieldContext.setUnformattedValue(null);
				fieldContext.setUnformattedCollectionValue(unformattedCollectionValues);
			} else {
				unformattedValue = (T)fieldContext.getUnformattedField().get(interceptorContext.getForm());
				fieldContext.setUnformattedValue(unformattedValue);
				fieldContext.setUnformattedCollectionValue(null);
			}
			break;
		case PAIR:
			fieldContext.setUnformattedField(pairFieldUsage.getUnformattedField());
			fieldContext.setAnnountationEntries(pairFieldUsage.getAnnountationEntries());
			fieldContext.setFieldName(pairFieldUsage.getFormattedName());
			pairFieldUsage.getUnformattedField().setAccessible(true);
			pairFieldUsage.getFormattedField().setAccessible(true);
			// Pair conversions don't apply to arrays or conversions 
			unformattedValue = (T)fieldContext.getUnformattedField().get(interceptorContext.getForm());
			fieldContext.setUnformattedValue(unformattedValue);
			fieldContext.setUnformattedCollectionValue(null);
		    break;
		}
	}

	/**
	 * <P>Called before formatting a form and sets property set 3 of InterceptorContext.</P> 
     * 
     * <P>Can access InterceptorContext property set 2 and lower.</P>
	 */
	protected void startFormProcessing(Field actionField, boolean skipConversionErrors) throws Exception {
		FormattableForm formattableForm;
    	InterceptorContext interceptorContext;
    	Object action, form;
    	
    	
    	interceptorContext = InterceptorContext.getInstance();
    	interceptorContext.setActionField(actionField);
        action = interceptorContext.getAction();
        form = actionField.get(action);
        interceptorContext.setForm(form);
        if (form instanceof FormattableForm) {
        	formattableForm = (FormattableForm)form;
        	interceptorContext.setFormattableForm(formattableForm);
        }
        interceptorContext.setSkipConversionErrors(skipConversionErrors);
	}

    protected void startInterceptorContext(ActionInvocation invocation) {
    	InterceptorContext interceptorContext;
    	Object action;
    	
    	interceptorContext = InterceptorContext.getInstance();
    	interceptorContext.setInvocation(invocation);
        action = invocation.getAction();
        interceptorContext.setAction(action);
        interceptorContext.setFakeForms(new HashMap<>());
    }

	/**
	 * Entry point called by {@link FormFormatterInterceptor}.
	 */
	@Override
	public void formatForms() {
    	Form formAnnotation;
    	InterceptorContext interceptorContext;
        Collection<Field> actionFields;
        Object action;
        boolean doForm, skipConversionErrors;
        
    	interceptorContext = makeInterceptorContext();
    	try {
        	InterceptorContext.setInstance(interceptorContext);
        	interceptorContext.setStep(InterceptorStep.START);
        	startInterceptorContext(ActionContext.getContext().getActionInvocation());
        	
            action = ActionContext.getContext().getActionInvocation().getAction();
            actionFields = getProperties(action.getClass());
            for (Field actionField: actionFields) {
            	skipConversionErrors = fieldReceivedStoredForm(actionField);
                doForm = fieldIsForm(actionField);
                if (doForm) {
                    try {
                        actionField.setAccessible(true);
                        formAnnotation = actionField.getAnnotation(Form.class);
                        if (formAnnotation == null || !formAnnotation.disableFormatting()) {
        		        	interceptorContext.setStep(InterceptorStep.FORM_PROCESSING_START);
        		        	startFormProcessing(actionField, skipConversionErrors);
        		        	
                        	formatForm();
                        	
        		        	interceptorContext.setStep(InterceptorStep.FORM_PROCESSING_END);
        		        	endFormProcessing();
                        }
                    }
                    catch (Exception e) {
                        LOG.error("Security violation when accessing form" + 
                                "  Struts action=" + action.getClass() + "  field=" + actionField.getName(), e);
                        continue;
                    }
                }
            }
        	interceptorContext.setStep(InterceptorStep.FORMAT_WRITING);
        	processFormattedValues();
    	}
    	finally {
        	interceptorContext.setStep(InterceptorStep.END);
    		endInterceptorContext();
    	}
	}

}
