package name.matthewgreet.strutscommons.util;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import name.matthewgreet.strutscommons.annotation.DisableFormatting;
import name.matthewgreet.strutscommons.policy.AbstractCustomCollectionFormatterSupport;
import name.matthewgreet.strutscommons.policy.AbstractCustomFormatterSupport;
import name.matthewgreet.strutscommons.policy.CollectionConverter;
import name.matthewgreet.strutscommons.policy.Converter;
import name.matthewgreet.strutscommons.util.InterceptorCommonLibrary.AnnotationEntries;

/**
 * <P>Default library for formatting a record to a view helper, according to converter annotations or default 
 * converter on the view helper fields, in Struts 2.  Converter annotations are found in 
 * <CODE>name.matthewgreet,strutscommons.annotation</CODE> (which allows custom converters). This must be executed in a 
 * Struts 2 Action context, whether in Action code or in an Interceptor.</P>
 * 
 * <P>Such converter annotations are for converting a single string value to a single value form field or collection 
 * form field, so this library only supports formatting a single record field to a single display field, formatting a 
 * collection record field to a single display field.  However, this library is designed with extensions in mind.  See
 * {@link #getDisplayFormatMode}.</P>
 */
public class DefaultDisplayFormatter implements DisplayFormatter {
	public static class DisplayFieldSource {
    	private Field displayField;
    	private Field recordField;
    	
    	
		public Field getDisplayField() {
			return displayField;
		}
		public Field getRecordField() {
			return recordField;
		}
		public void setDisplayField(Field displayField) {
			this.displayField = displayField;
		}
		public void setRecordField(Field recordField) {
			this.recordField = recordField;
		}
    }
    
    public static class DisplayFieldUsage<T> {
    	private AnnotationEntries<T> annountationEntries;
    	private Class<T> fieldType;
    	private boolean displayCollection;
    	private boolean displayArray;
    	private Field displayField;
    	private String displayName;
    	private boolean recordCollection;
    	private boolean recordArray;
    	private Field recordField;
    	private String recordName;
    	
    	
		public AnnotationEntries<T> getAnnountationEntries() {
			return annountationEntries;
		}
		public boolean getDisplayArray() {
			return displayArray;
		}
		
		public boolean getDisplayCollection() {
			return displayCollection;
		}
		public Field getDisplayField() {
			return displayField;
		}

		public String getDisplayName() {
			return displayName;
		}
		public Class<T> getFieldType() {
			return fieldType;
		}

		public boolean getRecordArray() {
			return recordArray;
		}
		public boolean getRecordCollection() {
			return recordCollection;
		}

		public Field getRecordField() {
			return recordField;
		}
		public String getRecordName() {
			return recordName;
		}

		public void setAnnountationEntries(AnnotationEntries<T> annountationEntries) {
			this.annountationEntries = annountationEntries;
		}
		public void setDisplayArray(boolean displayArray) {
			this.displayArray = displayArray;
		}

		public void setDisplayCollection(boolean displayCollection) {
			this.displayCollection = displayCollection;
		}
		public void setDisplayField(Field displayField) {
			this.displayField = displayField;
		}

		public void setDisplayName(String displayName) {
			this.displayName = displayName;
		}
		public void setFieldType(Class<T> fieldType) {
			this.fieldType = fieldType;
		}

		public void setRecordArray(boolean recordArray) {
			this.recordArray = recordArray;
		}
		public void setRecordCollection(boolean recordCollection) {
			this.recordCollection = recordCollection;
		}

		public void setRecordField(Field recordField) {
			this.recordField = recordField;
		}
		public void setRecordName(String recordName) {
			this.recordName = recordName;
		}

    }
    
    /**
	 *  Modes of how an unformatted, record field is converted to a formatted, display field, each with their own 
	 *  processing function.  See {@link #getDisplayFormatMode}.
	 */
    public enum DisplayFormatMode {SINGLE_TO_SINGLE, SINGLE_TO_ARRAY, SINGLE_TO_COLLECTION, ARRAY_TO_SINGLE,	
    	ARRAY_TO_ARRAY, ARRAY_TO_COLLECTION, COLLECTION_TO_SINGLE, COLLECTION_TO_ARRAY, COLLECTION_TO_COLLECTION, 
    	OTHER1, OTHER2, OTHER3, OTHER4, OTHER5, OTHER6, OTHER7, OTHER8, OTHER9, OTHER10, UNSUPPORTED}
    

    private Logger LOG = LogManager.getLogger(DefaultDisplayFormatter.class);
    
    
    /**
     * Returns whether a converter's target data type can convert to the form field's data type. 
     */
	protected <T> boolean checkFieldClass(Class<?> conversionClass, Class<?> fieldClass) {
		return InterceptorCommonLibrary.checkFieldClass(conversionClass, fieldClass);
	}

	/**
	 * Returns whether display field is allowed to receive a formatted value, which is not annotated with 
	 * {@link DisableFormatting}.  
	 */
	protected boolean filterDisplayField(Field displayField) {
		boolean disableFound;
		
		disableFound = false;
        for (Annotation annotation: displayField.getAnnotations()) {
            if (annotation instanceof DisableFormatting) {
            	disableFound = true;
            	break;
            }
        }
		return !disableFound;
	}

	/**
	 * Returns all recognised annotations and their linked policies.
	 */
    protected <T> AnnotationEntries<T> getAnnotationEntries(Field displayField) {
		return InterceptorCommonLibrary.getAnnotationEntries(displayField);
	}
    
	/**
	 * Returns default collection converter for a collection form field or display field, or null if none available. 
	 */
    @SuppressWarnings("unchecked")
	protected <T> CollectionConverter<?,T> getDefaultCollectionConverter(DisplayFieldUsage<T> displayFieldUsage) {
		return (CollectionConverter<?,T>)InterceptorCommonLibrary.getDefaultCollectionConverter(displayFieldUsage.getRecordField());
	}
    
	/**
	 * Returns default converter for a single value form field or display field, or null if none available. 
	 */
    @SuppressWarnings("unchecked")
	protected <T> Converter<?,T> getDefaultConverter(DisplayFieldUsage<T> displayFieldUsage) {
		return (Converter<?,T>)InterceptorCommonLibrary.getDefaultConverter(displayFieldUsage.getRecordField());
	}
	
	/**
     * Returns every display field that will receive a formatted value and the record field containing the value to be 
     * formatted. 
     */
    protected Collection<DisplayFieldSource> getDisplayFieldSources(Collection<Field> displayFields, Collection<Field> recordFields) {
    	DisplayFieldSource displayFieldSource;
    	Field recordField;
		Map<String,Field> recordFieldLookup;
		Collection<DisplayFieldSource> result;
		
        recordFieldLookup = recordFields.stream().collect(Collectors.toMap(f -> f.getName(), f -> f));
        result = new ArrayList<>();
        for (Field displayField: displayFields) {
        	recordField = recordFieldLookup.get(displayField.getName());
        	if (recordField != null) {
        		displayFieldSource = new DisplayFieldSource();
        		displayFieldSource.setDisplayField(displayField);
        		displayFieldSource.setRecordField(recordField);
        		result.add(displayFieldSource);
        	}
        }
        return result;
    }
	
	/**
     * Analyses converter annotations on display field and returns configured formatter to formatted from source field 
     * display field, or null if not applicable.
     */
	@SuppressWarnings("unchecked")
	protected <T> DisplayFieldUsage<T> getDisplayFieldUsage(Field recordField, Field displayField) {
   	    AnnotationEntries<T> annotationEntries;
	    CollectionConverter<?,T> collectionConverter;
    	Converter<?,T> converter;
    	DisplayFieldUsage<T> result;
    	Class<?> displayType;
    	Class<T> recordType;
    	boolean explicitConverter, isDisplayArray, isDisplayCollection, isRecordArray, isRecordCollection, compatible;
        
    	explicitConverter = false;
    	annotationEntries = getAnnotationEntries(displayField);
    	if (annotationEntries.getCollectionConverter() != null) {
    		collectionConverter = annotationEntries.getCollectionConverter().getCollectionConverter();
    		explicitConverter = true;
    	} else {
    		collectionConverter = null;
    	}
    	if (annotationEntries.getConverter() != null) {
    		converter = annotationEntries.getConverter().getConverter();
    		explicitConverter = true;
    	} else {
    		converter = null;
    	}
    	
    	isDisplayArray = displayField.getType().isArray();
    	isDisplayCollection = Collection.class.isAssignableFrom(displayField.getType());
    	if (isDisplayArray) {
    		displayType = displayField.getType().getComponentType();
    	} else if (isDisplayCollection) {
    		displayType = getTypeFromCollectionField(displayField);
    	} else {
    		displayType = displayField.getType();
    	}
    	if (!String.class.isAssignableFrom(displayType)) {
    		if (explicitConverter) {
                LOG.error("Annotated display fields must be a string  display=" + displayField.getDeclaringClass().getName() + 
                    "  field=" + displayField.getName());
    		}
    		return null;
    	}
    	isRecordArray = recordField.getType().isArray();
    	isRecordCollection = Collection.class.isAssignableFrom(recordField.getType());
    	if (isRecordArray) {
    		recordType = (Class<T>)recordField.getType().getComponentType();
    	} else if (isRecordCollection) {
    		recordType = getTypeFromCollectionField(recordField);
    	} else {
    		recordType = (Class<T>)recordField.getType();
    	}
    	
    	if (converter != null) {
    	    compatible = checkFieldClass(converter.getRecipientClass(), recordType);
    	    if (!compatible) {
                LOG.error("Annotation on display field does not match type of record field  display=" + displayField.getDeclaringClass().getName() + 
                    "  field=" + displayField.getName() + "  annotation=" + converter.getAnnotation().annotationType() + 
                 	"  record=" + recordField.getDeclaringClass() + "  field=" + recordField.getName());
                return null;
    	    }
	    } else if (collectionConverter != null) {
	    	compatible = checkFieldClass(collectionConverter.getRecipientClass(), recordType);
    	    if (!compatible) {
                LOG.error("Annotation on display field does not match type of record field  display=" + displayField.getDeclaringClass().getName() + 
                    "  field=" + displayField.getName() + "  annotation=" + collectionConverter.getAnnotation().annotationType() + 
                  	"  record=" + recordField.getDeclaringClass() + "  field=" + recordField.getName());
                return null;
    	    }
	    }
    	
    	result = new DisplayFieldUsage<T>();
    	result.setAnnountationEntries(annotationEntries);
    	result.setDisplayArray(isDisplayArray);
    	result.setDisplayCollection(isDisplayCollection);
    	result.setDisplayField(displayField);
    	result.setDisplayName(displayField.getName());
    	result.setFieldType(recordType);
    	result.setRecordArray(isRecordArray);
    	result.setRecordCollection(isRecordCollection);
    	result.setRecordField(recordField);
    	result.setRecordName(recordField.getName());
    	return result;
	}

	/**
     * Returns mode for formatting a record field to the matching display field.  Only SINGLE_TO_SINGLE and 
     * COLLECTION_TO_SINGLE are actually used and the rest regarded as inconsistent with converter annotation usage.  
     * Subclasses can override this support other modes.
     */
    protected DisplayFormatMode getDisplayFormatMode(DisplayFieldUsage<?> displayFieldUsage) {
    	if (displayFieldUsage.getDisplayArray() || displayFieldUsage.getDisplayCollection()) {
    		return DisplayFormatMode.UNSUPPORTED;
    	} else if (displayFieldUsage.getRecordArray()) {
    		return DisplayFormatMode.UNSUPPORTED;
    	} else if (displayFieldUsage.getRecordCollection()) {
    		return DisplayFormatMode.COLLECTION_TO_SINGLE;
    	} else {
    		return DisplayFormatMode.SINGLE_TO_SINGLE;
    	}
    }

	/**
     * Returns all properties of a class, even private ones, whether directly declared or inherited. 
     */
    protected Collection<Field> getProperties(Class<?> type) {
    	return InterceptorCommonLibrary.getProperties(type);
    }
	
    /**
     * Returns element type of a collection-based form field. 
     */
    protected <T> Class<T> getTypeFromCollectionField(Field displayField) {
		return InterceptorCommonLibrary.getTypeFromCollectionField(displayField);
	}

	/**
     * Returns an empty, modifiable collection from a collection type. 
     */
    protected <T> Collection<T> makeCollectionForRecipient(Class<?> recipientClass) {
    	return InterceptorCommonLibrary.makeCollectionForRecipient(recipientClass);
	}
	
	/**
	 * <P>Sets array display field from each member of an array record field formatted using converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string 
	 * collection, copies source values, otherwise sets empty array.</P>
	 * 
	 * <P>This is not actually used and exists for future use.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageArrayToArray(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		Collection<String> formattedValues;
		T[] unformattedValues;
		String[] finalValue;
		String formattedValue;
		int i;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValues = (T[])displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValues = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		formattedValues = new ArrayList<>();
		if (converter != null && unformattedValues != null) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null || converter.getProcessNoValue()) {
					formattedValue = converter.format(unformattedValue);
					formattedValues.add(formattedValue);
				}
			}
		}
		finalValue = new String[formattedValues.size()];
		i = 0;
		for (String formattedValue2: formattedValues) {
			finalValue[i] = formattedValue2;
			i++;
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * <P>Sets collection display field from each member of an array record field formatted using converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string 
	 * collection, copies source values, otherwise sets empty collection.</P>
	 * 
	 * <P>This is not actually used and exists for future use.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageArrayToCollection(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		Collection<String> finalValue;
		T[] unformattedValues;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValues = (T[])displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValues = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		finalValue = new ArrayList<>();
		if (converter != null && unformattedValues != null) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null || converter.getProcessNoValue()) {
					formattedValue = converter.format(unformattedValue);
					finalValue.add(formattedValue);
				}
			}
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * <P>Sets single display field from first member of an array record field formatted using converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string 
	 * collection, copies source values, otherwise sets empty string.</P>
	 * 
	 * <P>This is not actually used, not considered useful, and exists for the sake of completeness.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageArrayToSingle(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		T[] unformattedValues;
		T unformattedValue;
		String finalValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValues = (T[])displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValues = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		finalValue = "";
		if (converter != null && unformattedValues != null && unformattedValues.length > 0) {
			unformattedValue = unformattedValues[0];
			if (unformattedValue != null || converter.getProcessNoValue()) {
				finalValue = converter.format(unformattedValue);
			}
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * <P>Sets array display field from each member of a collection record field formatted using converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string 
	 * collection, copies source values, otherwise sets empty array.</P>
	 * 
	 * <P>This is not actually used and exists for future use.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageCollectionToArray(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		Collection<T> unformattedValues;
		Collection<String> formattedValues;
		String[] finalValue;
		String formattedValue;
		int i;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValues = (Collection<T>)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValues = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		formattedValues = new ArrayList<>();
		if (converter != null && unformattedValues != null) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null || converter.getProcessNoValue()) {
					formattedValue = converter.format(unformattedValue);
					formattedValues.add(formattedValue);
				}
			}
		}
		finalValue = new String[formattedValues.size()];
		i = 0;
		for (String formattedValue2: formattedValues) {
			finalValue[i] = formattedValue2;
			i++;
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * <P>Sets collection display field from each member of a collection record field formatted using converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string 
	 * collection, copies source values, otherwise sets empty collection.</P>
	 * 
	 * <P>This is not actually used and exists for future use.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageCollectionToCollection(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		Collection<T> unformattedValues;
		Collection<String> finalValue;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValues = (Collection<T>)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValues = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		finalValue = new ArrayList<>();
		if (converter != null && unformattedValues != null) {
			for (T unformattedValue: unformattedValues) {
				if (unformattedValue != null || converter.getProcessNoValue()) {
					formattedValue = converter.format(unformattedValue);
					finalValue.add(formattedValue);
				}
			}
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * Sets single display field from collection record field formatted using collection converter.  If no collection 
	 * converter is set, uses default collection converter for source field.  If no default converter, sets empty 
	 * string.
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageCollectionToSingle(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		CollectionConverter<?,T> collectionconverter;
		Collection<T> unformattedValue;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValue = (Collection<T>)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValue = null;
		}
		collectionconverter = null;
		if (displayFieldUsage.getAnnountationEntries().getCollectionConverter() != null) {
			collectionconverter = displayFieldUsage.getAnnountationEntries().getCollectionConverter().getCollectionConverter();
		}
		if (collectionconverter == null) {
			collectionconverter = (CollectionConverter<?,T>)getDefaultCollectionConverter(displayFieldUsage);
		}
		if (collectionconverter != null) {
			if ((unformattedValue != null && unformattedValue.size() > 0) || collectionconverter.getProcessNoValue()) {
				formattedValue = collectionconverter.format(unformattedValue);
			} else {
				formattedValue = "";
			}
		} else {
			formattedValue = "";
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, formattedValue);
	}
	
	/**
	 * <P>Called for OTHER1 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode}, to define OTHER1 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther1(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER10 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode}, to define OTHER10 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther10(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER2 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode}, to define OTHER2 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther2(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER3 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode}, to define OTHER3 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther3(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER4 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode}, to define OTHER4 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther4(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER5 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode}, to define OTHER5 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther5(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER6 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode}, to define OTHER6 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther6(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER7 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode}, to define OTHER7 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther7(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER8 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode}, to define OTHER8 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther8(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Called for OTHER9 formatting mode and does nothing.  This exists so subclasses can override it, as well as 
	 * {@link #getDisplayFormatMode}, to define OTHER9 formatting mode.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	protected <T> void processDisplayFieldUsageOther9(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		// Does nothing
	}
		
	/**
	 * <P>Sets array display field from from single record field formatted using configured converter.  If no converter 
	 * is set, uses default converter for source field.  If no default converter and source is a string, copies source 
	 * value, otherwise uses empty string.  If the formatted value is not an empty string, sets an array of just that 
	 * value, otherwise sets an empty array.</P>
	 * 
	 * <P>This is not actually used, not considered useful and exists for the sake of completeness.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageSingleToArray(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		String[] finalValue;
		T unformattedValue;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValue = (T)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValue = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		if (converter != null) {
			if (unformattedValue != null || converter.getProcessNoValue()) {
				formattedValue = converter.format(unformattedValue);
			} else {
				formattedValue = "";
			}
		} else if (unformattedValue != null && String.class.isAssignableFrom(unformattedValue.getClass())) {
			formattedValue = (String)unformattedValue;
		} else {
			formattedValue = "";
		}
		if (formattedValue.length() > 0) {
			finalValue = new String[1];
			finalValue[0] = formattedValue;
		} else {
			finalValue = new String[0];
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * <P>Sets collection display field from from single record field formatted using configured converter.  If no 
	 * converter is set, uses default converter for source field.  If no default converter and source is a string, 
	 * copies source value, otherwise uses empty string.  If the formatted value is not an empty string, sets a 
	 * collection of just that value, otherwise sets an empty collection.</P>
	 * 
	 * <P>This is not actually used, not considered useful and exists for the sake of completeness.</P>
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageSingleToCollection(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		Collection<String> finalValue;
		T unformattedValue;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValue = (T)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValue = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		if (converter != null) {
			if (unformattedValue != null || converter.getProcessNoValue()) {
				formattedValue = converter.format(unformattedValue);
			} else {
				formattedValue = "";
			}
		} else if (unformattedValue != null && String.class.isAssignableFrom(unformattedValue.getClass())) {
			formattedValue = (String)unformattedValue;
		} else {
			formattedValue = "";
		}
		finalValue = makeCollectionForRecipient(displayFieldUsage.getDisplayField().getType());
		if (formattedValue.length() > 0) {
			finalValue.add(formattedValue);
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, finalValue);
	}

	/**
	 * Sets single display field from from single record field formatted using configured converter.  If no converter is 
	 * set, uses default converter for source field.  If no default converter and source is a string, copies source 
	 * value, otherwise sets empty string.
	 * 
	 * @param record Record containing unformatted field values.  Can be null.
	 * @param display View helper containing fields to receive formatted values.
	 * @param displayFieldUsage Source and target fields and converter configured from annotation.
	 */
	@SuppressWarnings("unchecked")
	protected <T> void processDisplayFieldUsageSingleToSingle(Object record, Object display, DisplayFieldUsage<T> displayFieldUsage) throws Exception {
		Converter<?,T> converter;
		T unformattedValue;
		String formattedValue;
		
		if (record != null) {
			displayFieldUsage.getRecordField().setAccessible(true);
			unformattedValue = (T)displayFieldUsage.getRecordField().get(record);
		} else {
			unformattedValue = null;
		}
		converter = null;
		if (displayFieldUsage.getAnnountationEntries().getConverter() != null) {
			converter = displayFieldUsage.getAnnountationEntries().getConverter().getConverter();
		}
		if (converter == null) {
			converter = (Converter<?,T>)getDefaultConverter(displayFieldUsage);
		}
		if (converter != null) {
			if (unformattedValue != null || converter.getProcessNoValue()) {
				formattedValue = converter.format(unformattedValue);
			} else {
				formattedValue = "";
			}
		} else if (unformattedValue != null && String.class.isAssignableFrom(displayFieldUsage.getFieldType())) {
			formattedValue = (String)unformattedValue;
		} else {
			formattedValue = "";
		}
		displayFieldUsage.getDisplayField().setAccessible(true);
		displayFieldUsage.getDisplayField().set(display, formattedValue);
	}

	/**
     * <P>Sets string fields of <CODE>display</CODE> from fields of <CODE>record</CODE> with the same name but formatted 
     * according to the conversion annotations on <CODE>display</CODE> fields, or default converter if no annotation.  
     * If <CODE>record</CODE> is null, recipient <CODE>display</CODE> fields are set to empty string.  This function is 
     * useful for displaying database records.</P>
     * 
     * <P>Miscellaneous notes.</P>
     * <UL>
     *   <LI>Source field type must apply to converter's type.</LI>
     *   <LI>Recipient field must be a string, not even string array or string collection.</LI>
     *   <LI>Custom converters only for formatting are best derived from {@link AbstractCustomFormatterSupport} or 
     *       {@link AbstractCustomCollectionFormatterSupport}.</LI>
     *   <LI>The default converters for booleans and enumerations don't format to user friendly text, so custom 
     *       converters are better.</LI>
     *   <LI>Does not follow associations.</LI>
     * </UL> 
     */
	@Override
	public void updateDisplay(Class<?> recordClass, Object record, Object display) {
		DisplayFormatMode conversionMode;
		DisplayFieldUsage<?> displayFieldUsage;
        Field displayField, recordField;
		Collection<DisplayFieldSource> displayFieldSources;
        Collection<Field> recordFields, displayFields;
        Object unformattedValue;
        
        recordFields = getProperties(recordClass);
        displayFields = getProperties(display.getClass());
        displayFields = displayFields.stream().filter(this::filterDisplayField).collect(Collectors.toList());
        displayFieldSources = getDisplayFieldSources(displayFields, recordFields);
        
        for (DisplayFieldSource displayFieldSource: displayFieldSources) {
        	displayField = displayFieldSource.getDisplayField();
        	recordField = displayFieldSource.getRecordField();
    		unformattedValue = null;
    		try {
				displayFieldUsage = getDisplayFieldUsage(recordField, displayField);
				if (displayFieldUsage != null) {
					if (record != null) {
						displayFieldUsage.getRecordField().setAccessible(true);
	        			unformattedValue = displayFieldUsage.getRecordField().get(record) ;
					} else {
						unformattedValue = "";
					}
					conversionMode = getDisplayFormatMode(displayFieldUsage);
		        	
		        	switch (conversionMode) {
					case ARRAY_TO_ARRAY:
	        			processDisplayFieldUsageArrayToArray(record, display, displayFieldUsage);
						break;
					case ARRAY_TO_COLLECTION:
	        			processDisplayFieldUsageArrayToCollection(record, display, displayFieldUsage);
						break;
					case ARRAY_TO_SINGLE:
	        			processDisplayFieldUsageArrayToSingle(record, display, displayFieldUsage);
						break;
					case COLLECTION_TO_ARRAY:
	        			processDisplayFieldUsageCollectionToArray(record, display, displayFieldUsage);
						break;
					case COLLECTION_TO_COLLECTION:
	        			processDisplayFieldUsageCollectionToCollection(record, display, displayFieldUsage);
						break;
					case COLLECTION_TO_SINGLE:
	        			processDisplayFieldUsageCollectionToSingle(record, display, displayFieldUsage);
						break;
					case OTHER1:
	        			processDisplayFieldUsageOther1(record, display, displayFieldUsage);
						break;
					case OTHER2:
	        			processDisplayFieldUsageOther2(record, display, displayFieldUsage);
						break;
					case OTHER3:
	        			processDisplayFieldUsageOther3(record, display, displayFieldUsage);
						break;
					case OTHER4:
	        			processDisplayFieldUsageOther4(record, display, displayFieldUsage);
						break;
					case OTHER5:
	        			processDisplayFieldUsageOther5(record, display, displayFieldUsage);
						break;
					case OTHER6:
	        			processDisplayFieldUsageOther6(record, display, displayFieldUsage);
						break;
					case OTHER7:
	        			processDisplayFieldUsageOther7(record, display, displayFieldUsage);
						break;
					case OTHER8:
	        			processDisplayFieldUsageOther8(record, display, displayFieldUsage);
						break;
					case OTHER9:
	        			processDisplayFieldUsageOther9(record, display, displayFieldUsage);
						break;
					case OTHER10:
	        			processDisplayFieldUsageOther10(record, display, displayFieldUsage);
						break;
					case SINGLE_TO_ARRAY:
	        			processDisplayFieldUsageSingleToArray(record, display, displayFieldUsage);
						break;
					case SINGLE_TO_COLLECTION:
	        			processDisplayFieldUsageSingleToCollection(record, display, displayFieldUsage);
						break;
					case SINGLE_TO_SINGLE:
	        			processDisplayFieldUsageSingleToSingle(record, display, displayFieldUsage);
						break;
					case UNSUPPORTED:
						break;
		        	}
				}
    		}
    		catch (IllegalArgumentException | IllegalAccessException e) {
                LOG.error("Security violation when accessing record or setting display   display=" + display.getClass() + 
                        "  field=" + displayField.getName() + "  record=" + record.getClass() + "  field=" + recordField.getName(), e);
			}
    		catch (Exception e) {
                LOG.error("Formatter failed   display=" + display.getClass() + "  field=" + displayField.getName() +
                	"  value=" + unformattedValue, e);
			}
        }
	}

}
