package name.matthewgreet.strutscommons.interceptor;



import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.struts2.ServletActionContext;

import com.opensymphony.xwork2.ActionInvocation;
import com.opensymphony.xwork2.interceptor.AbstractInterceptor;

import name.matthewgreet.strutscommons.action.BrowserTabAware2;
import name.matthewgreet.strutscommons.action.BrowserTabSession;
import name.matthewgreet.strutscommons.action.BrowserTabSessionImpl;
import name.matthewgreet.strutscommons.util.ServletLibrary;


/**
 * If the action implements BrowserTabAware, sets tab id from cookies.  The requesting web page must be set up for this 
 * to be correct.  See {@link ServletLibrary#getBrowserTabId}.  This differs from BrowserTabInterceptor by detecting a 
 * user has opened a link in a new tab and copying data from the old tab so the user can treat them as independent.
 * See <A HREF="http://www.matthewgreet.name/programming/struts2_per-tab/injected_per-tab_session.html">Struts 2 - Server State per Browser Tab, Injected Per-tab Session</A>
 * for more details of how this works.   
 *   
 * <H2>Interceptor parameters</H2>
 * <DL>
 * <DT>disabled</DT><DD>If true, all processing is disabled.  Defaults to false.</DD>
 * </DL>
 * 
 * <H2>Extending the interceptor</H2>
 * <P>The following methods can be overridden :-</P>
 * <DL>
 * <DT>attributeCloneAction</DT><DD>Whether to ignore, copy or clone a tab-specific attribute of the old tab.</DD>
 * <DT>attributeFromOldTabToNewClone</DT><DD>Clones a private, tab-specific attribute from the old tab to the new.</DD>
 * <DT>attributeFromOldTabToNewCopyConstructor</DT><DD>Copies a private, tab-specific attribute from the old tab to the 
 *                                                     new using the copy constructor.</DD>
 * <DT>attributeFromOldTabToNewCopyReference</DT><DD>Copies reference to a private, tab-specific attribute from the old 
 *                                                   tab to the new.</DD>
 * <DT>attributeFromOldTabToNewIgnore</DT><DD>Notification that a private, tab-specific attribute found in the old tab 
                                              will not be copied.</DD>
 * <DT>attributeFromOldTabToNewNull</DT><DD>Copies null value of a private, tab-specific attribute from the old tab to 
 *                                          the new.</DD>
 * <DT>attributesFromOldTabToNewTab</DT><DD>Copies private, tab-specific attributes from the old tab to the new by 
 *                                          various means.</DD>
 * </DL>
 * 
 * <H2>Example code</H2>
 * <PRE>
 * &#064;InterceptorRefs({
 *   &#064;InterceptorRef(value="browserTab2"),
 *   &#064;InterceptorRef(value="defaultStack")
 *})
 * </PRE>
 */
@SuppressWarnings("deprecation")
public class BrowserTabInterceptor2 extends AbstractInterceptor {
    public enum AttributeCloneAction {CLONE, COPY_CONSTRUCTOR, COPY_REFERENCE, IGNORE, NULL}
    
    private static final long serialVersionUID = -1555224920319813415L;
    
    private boolean disabled;
    
    /**
     * Returns what to do with existing attributes of an existing page if the user is opening a hyperlink in a new 
     * browser tab.
     */
    protected AttributeCloneAction attributeCloneAction(BrowserTabAware2 browserTabAware, ActionInvocation invocation,
			String attributeName, Object attributeValue) {
		Class<?> valueClass;
		
    	if (attributeValue == null) {
    		return AttributeCloneAction.NULL;
    	}
		valueClass = attributeValue.getClass();
		try {
			valueClass.getConstructor(valueClass);
	        return AttributeCloneAction.COPY_CONSTRUCTOR;
		}
		catch (NoSuchMethodException | SecurityException e) {
			// Ignore;
		}
    	if (attributeValue instanceof Cloneable) {
            return AttributeCloneAction.CLONE;
    	}
        return AttributeCloneAction.COPY_REFERENCE;
	}

	/**
	 * Handles an attribute from the old tab that should be cloned and returns whether it succeeded.
	 */
	protected boolean attributeFromOldTabToNewTabClone(BrowserTabAware2 browserTabAware, ActionInvocation invocation,
			BrowserTabSession oldBrowserTabSession, BrowserTabSession newBrowserTabSession, Entry<String,Object> oldAttributeEntry) {
		Method cloneMethod;
		
		if (oldAttributeEntry.getValue() instanceof Cloneable) {
			try {
				cloneMethod = oldAttributeEntry.getValue().getClass().getMethod("clone");
				newBrowserTabSession.setAttribute(oldAttributeEntry.getKey(), cloneMethod.invoke(oldAttributeEntry.getValue()));
				return true;
			}
			catch (NoSuchMethodException | SecurityException | IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
				return false;
			}
		} else {
			return false;
		}
	}
	
	/**
	 * Handles an attribute from the old tab that that should be copied with a copy constructor and returns whether it 
	 * succeeded.
	 */
	protected boolean attributeFromOldTabToNewTabCopyConstructor(BrowserTabAware2 browserTabAware, ActionInvocation invocation,
			BrowserTabSession oldBrowserTabSession, BrowserTabSession newBrowserTabSession, Entry<String,Object> oldAttributeEntry) {
		Class<?> valueClass;
		Constructor<?> copyConstructor;
		Object newValue;
		
		try {
			valueClass = oldAttributeEntry.getValue().getClass();
			copyConstructor = valueClass.getConstructor(valueClass);
			newValue = copyConstructor.newInstance(oldAttributeEntry.getValue());
			newBrowserTabSession.setAttribute(oldAttributeEntry.getKey(), newValue);
			return true;
		}
		catch (NoSuchMethodException | SecurityException | InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException e) {
			return false;
		}
	}

	/**
	 * Handles an attribute from the old tab that where its reference should be copied and returns whether it succeeded.
	 */
	protected boolean attributeFromOldTabToNewTabCopyReference(BrowserTabAware2 browserTabAware, ActionInvocation invocation,
			BrowserTabSession oldBrowserTabSession, BrowserTabSession newBrowserTabSession, Entry<String,Object> oldAttributeEntry) {
		newBrowserTabSession.setAttribute(oldAttributeEntry.getKey(), oldAttributeEntry.getValue());
		return true;
	}

	/**
	 * Handles an attribute from the old tab that should be ignored and returns whether it succeeded, which does nothing.
	 */
	protected boolean attributeFromOldTabToNewTabIgnore(BrowserTabAware2 browserTabAware, ActionInvocation invocation,
			BrowserTabSession oldBrowserTabSession, BrowserTabSession newBrowserTabSession, Entry<String,Object> oldAttributeEntry) {
		// Does nothing
		return true;
	}

	/**
	 * Handles an attribute from the old tab that that has a null value and returns whether it succeeded.
	 */
	protected boolean attributeFromOldTabToNewTabNull(BrowserTabAware2 browserTabAware, ActionInvocation invocation,
			BrowserTabSession oldBrowserTabSession, BrowserTabSession newBrowserTabSession, Entry<String,Object> oldAttributeEntry) {
		newBrowserTabSession.setAttribute(oldAttributeEntry.getKey(), oldAttributeEntry.getValue());
		return true;
	}

	/**
     * Called when the user's web browser reloads because the page has been opened in a new tab, to copy private, 
     * tab-specific attributes from the old tab to the new.
     */
	protected void attributesFromOldTabToNewTab(BrowserTabAware2 browserTabAware, ActionInvocation invocation,
			BrowserTabSession oldBrowserTabSession, BrowserTabSession newBrowserTabSession) {
		AttributeCloneAction attributeCloneAction;
		Map<String,Object> oldPrivateMap, newPrivateMap;
		Set<String> attributeFailures, attributeIgnores;
		String oldAttributeName;
		boolean success;
		
		attributeFailures = new HashSet<>();
		attributeIgnores = new HashSet<>();
		browserTabAware.copyAttributesFromOldTabToNewTab(oldBrowserTabSession, newBrowserTabSession, attributeIgnores, attributeFailures);
		oldPrivateMap = oldBrowserTabSession.getPrivateAttributeMap();
		newPrivateMap = newBrowserTabSession.getPrivateAttributeMap();
		for (Entry<String,Object> oldAttributeEntry: oldPrivateMap.entrySet()) {
			oldAttributeName = oldAttributeEntry.getKey();
			if (!attributeIgnores.contains(oldAttributeName) && !newPrivateMap.containsKey(oldAttributeName)) {
				attributeCloneAction = attributeCloneAction(browserTabAware, invocation, oldAttributeName, oldAttributeEntry.getValue());
				success = true;
				switch (attributeCloneAction) {
				case CLONE:
					success = attributeFromOldTabToNewTabClone(browserTabAware, invocation, oldBrowserTabSession, newBrowserTabSession, oldAttributeEntry);
					break;
				case COPY_CONSTRUCTOR:
					success = attributeFromOldTabToNewTabCopyConstructor(browserTabAware, invocation, oldBrowserTabSession, newBrowserTabSession, oldAttributeEntry);
					break;
				case COPY_REFERENCE:
					success = attributeFromOldTabToNewTabCopyReference(browserTabAware, invocation, oldBrowserTabSession, newBrowserTabSession, oldAttributeEntry);
					break;
				case IGNORE:
					success = attributeFromOldTabToNewTabIgnore(browserTabAware, invocation, oldBrowserTabSession, newBrowserTabSession, oldAttributeEntry);
					break;
				case NULL:
					success = attributeFromOldTabToNewTabNull(browserTabAware, invocation, oldBrowserTabSession, newBrowserTabSession, oldAttributeEntry);
					break;
				}
				if (!success) {
					attributeFailures.add(oldAttributeName);
				}
			}
		}
		browserTabAware.handleAttributeFailures(oldBrowserTabSession, newBrowserTabSession, attributeFailures);
	}
	
	/**
	 * Returns previous tab id when page detects it's in a new browser tab, or empty string if not applicable.
	 */
	protected String getOldTabId(ActionInvocation invocation) {
        Cookie[] cookies;
        HttpServletRequest request;
		String result;
		
        request = invocation.getInvocationContext().getServletRequest();
        cookies = request.getCookies();
		result = ServletLibrary.getBrowserOldTabId(cookies);
        result = (result != null)?result:"";
        return result;
	}

	/**
	 * Returns tab id of current browser tab, creating it if one doesn't exist. 
	 */
    protected String getTabId(ActionInvocation invocation) {
        HttpServletRequest request;
        HttpServletResponse response;
        String result;
        
        request = (HttpServletRequest)invocation.getInvocationContext().get(ServletActionContext.HTTP_REQUEST);
        response = (HttpServletResponse)invocation.getInvocationContext().get(ServletActionContext.HTTP_RESPONSE);
        result = ServletLibrary.getGuaranteedBrowserTabId(request, response);
        return result;
    }
    
    public boolean getDisabled() {
		return disabled;
	}
	public void setDisabled(boolean disabled) {
		this.disabled = disabled;
	}

	@Override
    public String intercept(ActionInvocation invocation) throws Exception {
    	BrowserTabAware2 browserTabAware;
    	BrowserTabSession browserTabSession, oldBrowserTabSession;
        String result, tabId, oldTabId;
    	
        if (!disabled && invocation.getAction() instanceof BrowserTabAware2) {
            tabId = getTabId(invocation);
            oldTabId = getOldTabId(invocation);
            browserTabSession = new BrowserTabSessionImpl(invocation.getInvocationContext().getServletRequest().getSession(), tabId);
            browserTabAware = (BrowserTabAware2)invocation.getAction();
            if (oldTabId.length() > 0 && !oldTabId.equals(tabId)) {
            	oldBrowserTabSession = new BrowserTabSessionImpl(invocation.getInvocationContext().getServletRequest().getSession(), oldTabId);
            	attributesFromOldTabToNewTab(browserTabAware, invocation, oldBrowserTabSession, browserTabSession);
            }
            browserTabAware.setBrowserTabSession(browserTabSession);
        }
        result = invocation.invoke();
        return result;
    }

}
