package name.matthewgreet.strutscommons.annotation;

import java.lang.annotation.Documented;
import java.lang.annotation.ElementType;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import name.matthewgreet.strutscommons.annotation.Required.MessageType;
import name.matthewgreet.strutscommons.policy.AbstractCustomConverterSupport;
import name.matthewgreet.strutscommons.policy.Converter;

/**
 * <P>Defines a single value form field (whether on a Struts Action or model of ModelDriven Struts Action) should be 
 * converted using a client supplied converter to another and error message to display if this fails.  The custom 
 * converter must implement {@link Converter}, preferably extending {@link AbstractCustomConverterSupport}, and have a 
 * default public constructor.  Also, defines a view helper field receives a value formatted by the converter.</P>
 *
 * <P>There are two modes of behaviour for conversion, depending on the type of form field.</P>
 * <TABLE>
 *   <CAPTION>Annotation Usage</CAPTION>
 *   <TR>
 *     <TH>Form field type</TH>
 *     <TH>Mode</TH>
 *     <TH>Description</TH>
 *   </TR>
 *   <TR>
 *     <TD>String</TD>
 *     <TD>Pair conversion</TD>
 *     <TD>Converts string field value field named by parsedFieldName</TD>
 *   </TR>
 *   <TR>
 *     <TD>non-String</TD>
 *     <TD>Auto conversion</TD>
 *     <TD>Converts directly from request parameter</TD>
 *   </TR>
 * </TABLE>
 * 
 * <P>By default, messages are added to the actionErrors list but this can be overridden by <CODE>messageType</CODE>:</P>
 * <DL>
 *   <DT>ERROR</DT><DD>Message added to Action level error messages.  Default.</DD>
 *   <DT>FIELD</DT><DD>Message added to field error messages.</DD>
 *   <DT>MESSAGE</DT><DD>Message added to Action level info messages.</DD>
 * </DL>
 * 
 * <P>Other parameters</P>
 * <DL>
 *   <DT>message</DT><DD>If not empty string, text message to display.  Used if <CODE>messageKey</CODE> is empty string 
 *                       or finds no message.</DD>
 *   <DT>messageKey</DT><DD>If not empty string, message key to retrieve from Action's TextProvider.  Uses 
 *                          <CODE>message</CODE> if no key provided or no message found.</DD>
 *   <DT>param1</DT><DD>Parameter recognised by custom validator.  Defaults to empty string.</DD>
 *   <DT>param2</DT><DD>Parameter recognised by custom validator.  Defaults to empty string.</DD>
 *   <DT>param3</DT><DD>Parameter recognised by custom validator.  Defaults to empty string.</DD>
 *   <DT>param4</DT><DD>Parameter recognised by custom validator.  Defaults to empty string.</DD>
 *   <DT>param5</DT><DD>Parameter recognised by custom validator.  Defaults to empty string.</DD>
 *   <DT>parsedFieldName</DT><DD>If not empty string, name of field to contain parsed value.  Defaults to 'parsed' plus 
 *                               capitalised field name, such as 'parsedLimit'.  Only applies for pair conversion.</DD>
 *   <DT>processNoValue</DT><DD>If true, can convert empty string and format null value.  Defaults to false.</DD>
 *   <DT>validatorClass</DT><DD>Class of client supplied validator.  See notes above.</DD>
 * </DL>
 */
@Documented
@Inherited
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface CustomConversion {
    public String message() default "";
    public String messageKey() default "";
    public MessageType messageType() default MessageType.ERROR;
    public String param1() default "";
    public String param2() default "";
    public String param3() default "";
    public String param4() default "";
    public String param5() default "";
    public String parsedFieldName() default "";
    public boolean processNoValue() default false;
    public Class<? extends Converter<CustomConversion,?>> validatorClass();
}
