package name.matthewgreet.example11.policy;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.opensymphony.xwork2.ActionContext;

import name.matthewgreet.example11.annotation.DateRangeListConversion;
import name.matthewgreet.example11.dto.DateRange;
import name.matthewgreet.strutscommons.annotation.DateConversion.Style;
import name.matthewgreet.strutscommons.annotation.Required.MessageType;
import name.matthewgreet.strutscommons.policy.AbstractDefaultCollectionConverterSupport;
import name.matthewgreet.strutscommons.policy.ConversionResult;
import name.matthewgreet.strutscommons.policy.DateConverter;

@SuppressWarnings("deprecation")
public class DateRangeListConverter extends AbstractDefaultCollectionConverterSupport<DateRangeListConversion,DateRange> {
    private static final Logger LOG = LogManager.getLogger(DateConverter.class);
    
    public static int getDateFormatStyle(Style style) {
        switch (style) {
        case FULL:      return DateFormat.FULL;
        case LONG:      return DateFormat.LONG;
        case MEDIUM:    return DateFormat.MEDIUM;
        case SHORT:     return DateFormat.SHORT;
        }
        return DateFormat.SHORT;
    }
    
    
    @DateRangeListConversion
    private static boolean annotationPlaceholder;

	
	private DateRange parseDateRange(String value) throws ParseException {
		DateRange result;
    	DateFormat dateFormat;
    	Date start, end;
		String[] parts;
		
		parts = value.split("~",2);
		if (parts.length != 2) {
			throw new ParseException("Date range does not match <date>~<date> format", 0);
		}
        dateFormat = makeDateFormat();
        start = dateFormat.parse(parts[0]);
        if (parts[1].length() > 0) {
        	end = dateFormat.parse(parts[1]);
        } else {
        	end = null;
        }
        if (end == null || end.after(start)) {
        	result = new DateRange(start,end);
        } else {
        	result = new DateRange(end,start);
        }
        return result;
	}

	protected String formatDateRamgeList(Collection<DateRange> unformattedValues, String separator) {
    	DateFormat dateFormat;
        String result;
        
        dateFormat = makeDateFormat();
        result = "";
        for (DateRange value: unformattedValues) {
            if (result.length() > 0) {
                result = result + separator;
            }
            if (value.getStart() != null) {
            	result = result + dateFormat.format(value.getStart());
            }
            result = result + "~";
            if (value.getEnd() != null) {
            	result = result + dateFormat.format(value.getEnd());
            }
        }
        return result;
	}
	
    protected DateFormat makeDateFormat() {
        DateFormat result;
        Locale locale;
        
        if (getAnnotation().format().length() > 0) {
            result = new SimpleDateFormat(getAnnotation().format());
        } else if (getAnnotation().localeCountry().length() > 0 && getAnnotation().localeLanguage().length() > 0) {
            locale = new Locale(getAnnotation().localeLanguage(), getAnnotation().localeCountry());
            result = DateFormat.getDateInstance(getDateFormatStyle(getAnnotation().style()), locale);
        } else {
            locale = ActionContext.getContext().getLocale();
            result = DateFormat.getDateInstance(getDateFormatStyle(getAnnotation().style()), locale);
        }
        result.setLenient(false);
        result.setTimeZone(TimeZone.getTimeZone("UTC"));
        return result;
    }

	@Override
	protected DateRangeListConversion makeDefaultAnnotation() {
		try {
			return DateRangeListConverter.class.getDeclaredField("annotationPlaceholder").getAnnotation(DateRangeListConversion.class);
		}
    	catch (Exception e) {
			LOG.error("Creation of default annotation failed", e);
			return null;
		}
	}

	@Override
	public ConversionResult<DateRange> convert(String formValue, Class<?> recipientFieldClass, Class<? extends DateRange> recipientClass) throws Exception {
        Collection<DateRange> parsedValue;
        String[] values;
        
        parsedValue = makeCollectionForRecipient(recipientFieldClass);
        try {
            values = formValue.split(getAnnotation().separator());
            for (String value: values) {
                parsedValue.add(parseDateRange(value.trim()));
            }
        }
        catch (ParseException e) {
            return ConversionResult.makeFailureResult();
        }
        return ConversionResult.makeSuccessCollectionResult(parsedValue);
	}

    @Override
    public String format(Collection<DateRange> unformattedValues) throws Exception {
    	return formatDateRamgeList(unformattedValues, getAnnotation().separator());
    }


	@Override
	public MessageType getMessageType() {
		return getAnnotation().messageType();
	}


	@Override
	public String getMessage() {
		return getAnnotation().message();
	}


	@Override
	public String getMessageKey() {
		return getAnnotation().messageKey();
	}


	@Override
	public Class<DateRange> getRecipientClass() {
		return DateRange.class;
	}

	@Override
	public String getRecipientFieldName() {
		return getAnnotation().parsedFieldName();
	}


	@Override
	public boolean getProcessNoValue() {
		return false;
	}


}
