package name.matthewgreet.example11.policy;

import java.util.Date;

import name.matthewgreet.example11.annotation.AgeRange;
import name.matthewgreet.strutscommons.annotation.Required.MessageType;
import name.matthewgreet.strutscommons.policy.AbstractPostConversionValidatorSupport;
import name.matthewgreet.strutscommons.policy.ValidationResult;

/**
 * Bespoke post conversion validator that checks if a date is in a sane range.
 * 
 * <DL>
 *   <DT>minAge</DT><DD>Minimum age in years</DD>
 *   <DT>maxAge</DT><DD>Maximum age in years</DD>
 * </DL>
 */
public class AgeRangeValidator extends AbstractPostConversionValidatorSupport<AgeRange,Date> {
	public enum AgeRangeRejection {NONE, MIN_AGE, MAX_AGE}
	
	private AgeRangeRejection ageRangeRejection = AgeRangeRejection.NONE;
	
    @Override
    public Class<Date> getRecipientClass() {
        return Date.class;
    }
    
	@Override
	public ValidationResult validate(Date fieldValue) {
        int minAge, maxAge, age;
        
        minAge = getAnnotation().minAge();
        maxAge = getAnnotation().maxAge();
        // Not strictly accurate, considering leap years, but it's for sanity checking
        age = (int)((System.currentTimeMillis() - fieldValue.getTime()) / 31556952000L);
        if (age < minAge) {
        	ageRangeRejection = AgeRangeRejection.MIN_AGE;
            return ValidationResult.makeFailureResult();
        } else if (age > maxAge) {
        	ageRangeRejection = AgeRangeRejection.MAX_AGE;
            return ValidationResult.makeFailureResult();
        } else {
            return ValidationResult.makeSuccessResult();
        }
	}

	@Override
	public MessageType getMessageType() {
		return getAnnotation().messageType();
	}

	@Override
	public String getMessage() {
		switch (ageRangeRejection) {
		case MAX_AGE:	return getAnnotation().maxAgeMessage();
		case MIN_AGE:	return getAnnotation().minAgeMessage();
		case NONE:		return "";
		}
		return "";
	}

	@Override
	public String getMessageKey() {
		switch (ageRangeRejection) {
		case MAX_AGE:	return getAnnotation().maxAgeMessageKey();
		case MIN_AGE:	return getAnnotation().minAgeMessageKey();
		case NONE:		return "";
		}
		return "";
	}

	@Override
	public boolean getShortCircuit() {
		return getAnnotation().shortCircuit();
	}

	@Override
	public boolean getProcessNoValue() {
		return false;
	}

}